// ImageExtractor 1.0.0
// Useless Utility, The Gathering 2004
// Stian A. Bergum, April 9, 2004

// Licensed under the GNU GPL

// Some Swing stuff
import javax.swing.*;
import javax.swing.event.*;

// Other GUI packages
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

// Input/output
import java.io.*;


public class ImageExtractor extends JFrame {
	
	// The color white is defined as a constant
	final private Color WHITE = new Color(255,255,255);

	// The file chooser dialog box
	private JFileChooser chooser;
	
	// A raw image object, which we will alter pixel by pixel
	private BufferedImage image;
	
	// The ImageIcon class is used for easily display the image as as Swing component
	private ImageIcon imageicon;
	
	// This label contains the graphic, or a text if no graphic is created yet
	private JLabel label;
	
	// The status bar (surprisingly)
	private JLabel statusbar;
	
	// The file that we shall turn into an image
	private String filename = "";
	
	// Just an integer to keep order on how many pixels of generation that remain
	private int remaining = 0;
	
	// The main container
	private Container container;
	
	public ImageExtractor () {
		
		// A call to the superclass (JFrame) constructor will create a new Swing window
		super("Image Extractor");
		
		// Try to set the Look and Feel to match the system the app is run on.
		// An error message is displayed if the "system default" L&F does not exist.
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		}
		catch (Exception e) {
			JOptionPane.showMessageDialog(null, "Unable to load Look and Feel. This is possibly due to an error in your Java Runtime Engine configuration.", "Cannot load Look and Feel", JOptionPane.ERROR_MESSAGE);
		}

		// Initializing the file chooser
		chooser = new JFileChooser();
		
		// Initializing the graphic "label" with a default text
		label = new JLabel("Please open a file using the Open command in the File menu.");

		// Fetching the windows content pane
		container = getContentPane();
		
		// BorderLayout divides into north, east, west, south and center
		container.setLayout(new BorderLayout());
		
		// The menubar
		JMenuBar menubar = new JMenuBar();
		
		// Action handler for the menu items (handles clicks on these)
		ActionHandler actionHandler = new ActionHandler();
		
		// "File" menu (hotkey F)
		JMenu fileMenu = new JMenu("File");
		fileMenu.setMnemonic('F');
		
		// "Open" item (hotkey O or shortcut Ctrl+O)
		JMenuItem fileMenuOpen = new JMenuItem("Open...", KeyEvent.VK_O);
		fileMenuOpen.addActionListener(actionHandler);
		fileMenuOpen.setActionCommand("file:open");
		fileMenuOpen.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.CTRL_MASK));
		
		// "Exit" item (hotkey X or shortcut Alt+F4)
		JMenuItem fileMenuExit = new JMenuItem("Exit", KeyEvent.VK_X);
		fileMenuExit.addActionListener(actionHandler);
		fileMenuExit.setActionCommand("file:exit");
		fileMenuExit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4, InputEvent.ALT_MASK));
		
		// Adding items to the menu
		fileMenu.add(fileMenuOpen);
		fileMenu.addSeparator();
		fileMenu.add(fileMenuExit);
		
		// "Help" menu (hotkey H)
		JMenu helpMenu = new JMenu("Help");
		helpMenu.setMnemonic('H');
		
		// "General Help" item (hotkey G or shortcut F1)
		JMenuItem helpMenuGeneral = new JMenuItem("General Help...", KeyEvent.VK_G);
		helpMenuGeneral.addActionListener(actionHandler);
		helpMenuGeneral.setActionCommand("help:general");
		helpMenuGeneral.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0));
		
		// "About" item (hotkey A)
		JMenuItem helpMenuAbout = new JMenuItem("About...", KeyEvent.VK_A);
		helpMenuAbout.addActionListener(actionHandler);
		helpMenuAbout.setActionCommand("help:about");
		
		// Adding items to the menu
		helpMenu.add(helpMenuGeneral);
		helpMenu.add(helpMenuAbout);
		
		// Adding menus to the menu bar		
		menubar.add(fileMenu);
		menubar.add(helpMenu);

		// Adding the menubar to the window
		container.add(menubar, BorderLayout.NORTH);
		
		// Initializing the status bar and adding it to the window
		statusbar = new JLabel("Ready for action.");
		container.add(statusbar, BorderLayout.SOUTH);
		
		// Setting the text alignment of the graphic "label" to center (default is left)
		label.setHorizontalAlignment(SwingConstants.CENTER);
		// Adding the label to the window
		container.add(label, BorderLayout.CENTER);
		
		// When the window is closed, the app terminates
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		
		// The "restored" size is set to 800x600, but the app is maximized on startup
		setSize(800, 600);
		setExtendedState(JFrame.MAXIMIZED_BOTH);
		
		// Display the window
		show();
	}
	
	// Handles clicks on the menu items
	private class ActionHandler implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			// "File > Open"
			if (e.getActionCommand().equals("file:open")) {
				if (chooser.showOpenDialog(null) == JFileChooser.APPROVE_OPTION) {
					// Fetch a filename from the chooser
					filename = chooser.getSelectedFile().getPath();
					// Initialize a new renderThread
					RenderThread renderThread = new RenderThread();
					// Start the renderThread
					renderThread.start();
				}
			}
			// "File > Exit"
			else if (e.getActionCommand().equals("file:exit")) {
				// Terminates the app
				System.exit(0);
			}
			// "Help > General Help"
			else if (e.getActionCommand().equals("help:general")) {
				// Display a message box with the help text
				JOptionPane.showMessageDialog(null, "This application will create nice-looking graphics out of any file. Instructions:\n\n1. Open a file using the Open command in the File menu.\n2. Wait a couple of seconds...\n3. Enjoy the result.\n4. Try another file if you'd like to.", "General Help", JOptionPane.INFORMATION_MESSAGE);
			}
			// "Help > About"
			else if (e.getActionCommand().equals("help:about")) {
				// Display a message box with "about" information
				JOptionPane.showMessageDialog(null, "Image Extractor 1.0.0\n\nDeveloped during The Gathering 2004 by geek` aka. Stian A. Bergum, and sumitted for the Useless Utility competition.\nSun Java Runtime Environment 1.3 or newer is recommended. For support inquiries, please contact stian@vennesla.net.\n\nCopyright 2004. Licensed under the GNU General Public License.", "General Help", JOptionPane.INFORMATION_MESSAGE);
			}
		}
	}
	
	// The RenderThread loads the file and renders the image.
	// This can be quite timeconsuming, so we run that in a separate thread to make sure
	// the GUI doesn't freeze for a long time.
	private class RenderThread extends Thread {
		public RenderThread () {
			// Call the superclass (Thread) constructor to create a new thread
			super("render");
		}
		public void run() {
			// Fetches the image
			image = createGraphic();
			// Convert the image into a ImageIcon object
			imageicon = new ImageIcon(image);
			
			// Update the GUI with the newly created image
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					label.setText(null);
					label.setIcon(imageicon);
					statusbar.setText("Done, ready for more action.");
				}
			});
		}
	}
	
	private BufferedImage createGraphic () {
		
		// 640x480 equals to 921600 pixels, and we create a byte array of that size
		byte[] bytes = new byte[921600];
		
		// We also create a BufferedImage (directly writable pixel by pixel) of the right size
		BufferedImage image = new BufferedImage(640, 480, BufferedImage.TYPE_INT_ARGB);

		// Try to read the first bytes of the file into the bytes array, and give a proper
		// error message if this fails for some reason.
		try {
			FileInputStream file = new FileInputStream(filename);    
			file.read(bytes);
			file.close();
		}
		catch (FileNotFoundException e) {
			JOptionPane.showMessageDialog(null, "I'm not able to find the file.", "Cannot open file", JOptionPane.ERROR_MESSAGE);
			System.exit(0);
		}
		catch (IOException e) {
			JOptionPane.showMessageDialog(null, "I'm not able to read the file.", "Cannot read file", JOptionPane.ERROR_MESSAGE);
			System.exit(0);
		}
		
		// Controls which row and column of the graphic that should be altered
		int xCounter = 0;
		int yCounter = 0;
		
		// Loops through the byte array
		for (int i = 0; i < (bytes.length - 2); i+= 3) {
			
			// If the xCounter is 639, we have hit the end of the row, and should
			// proceed to the next row
			if (xCounter == 639) {
				xCounter = 0;
				yCounter++;
			}
			// ...but if not, we proceed at the next column
			else {
				xCounter++;
			}

			// We dont want to modify non-existing rows, so we jump out of the loop if the
			// yCounter reaches 480.
			if (yCounter >= 480) break;
			
			// Reads the current and the two next bytes (red, green and blue, respectively)
			Byte rByte = new Byte(bytes[i]);
			Byte gByte = new Byte(bytes[i+1]);
			Byte bByte = new Byte(bytes[i+2]);
			
			// Convert the bytes into integer values.
			// If these are less than zero, we invert the numbers.
			
			int red = rByte.intValue();
			if (red < 0) red *= -1;

			int green = gByte.intValue();
			if (green < 0) green *= -1;

			int blue = bByte.intValue();
			if (blue < 0) blue *= -1;
	
			// Create a new Color with the three bytes from the file
			Color color = new Color(red, green, blue);
			// Alters the right pixel of the image
			image.setRGB(xCounter, yCounter, color.getRGB());
			
			// Just some stuff to update the status bar.
			if (i == 0 || i % 10 == 0) {
				remaining = 307200 - (i / 3);
				SwingUtilities.invokeLater(new Runnable() {
					public void run () {
						statusbar.setText("Creating image, " + remaining + " pixels remaining.");
					}
				});
			}

		}
		
		// Just some stuff to update the status bar.
		SwingUtilities.invokeLater(new Runnable() {
			public void run () {
				statusbar.setText("Drawing image...");
			}
		});
		
		return image;
		
	}
	
	// The main method starts up the application
	public static void main (String args[]) {
		ImageExtractor application = new ImageExtractor();
	}	
}