#include "encyclopedia.h"

#include "commands.h"
#include "encyclopediapage.h"
#include "menu.h"
#include "music.h"
#include "options.h"

#include "SDL.h"

#include <sstream>

//############################################################################
// Construction ##############################################################
//############################################################################

/** Constructor for encyclopedia. Simply laod all pages here.
 */
Encyclopedia::Encyclopedia()
  : current_page(0)
{
  this->add("encyclopedia/contents.txt");
  this->add("encyclopedia/how_to_play_1.txt");
  this->add("encyclopedia/how_to_play_2.txt");
  this->add("encyclopedia/story_1.txt");
  this->add("encyclopedia/story_2.txt");
  this->add("encyclopedia/story_3.txt");
  this->add("encyclopedia/interdimensional_battleship.txt");
  this->add("encyclopedia/r-9a.txt");
  this->add("encyclopedia/r-9b2.txt");
  this->add("encyclopedia/r-9d.txt");
  this->add("encyclopedia/r-9d2.txt");
  this->add("encyclopedia/r-9dh2.txt");
  this->add("encyclopedia/r-9dv2.txt");
  this->add("encyclopedia/r-9k.txt");
  this->add("encyclopedia/r-9o.txt");
  this->add("encyclopedia/r-11a.txt");
  this->add("encyclopedia/r-11s2.txt");
  this->add("encyclopedia/r-13a.txt");
  this->add("encyclopedia/r-13b.txt");
  this->add("encyclopedia/rx-10.txt");
  this->add("encyclopedia/tx-t.txt");
  this->add("encyclopedia/b-1a3.txt");
  this->add("encyclopedia/b-1d3.txt");
  this->add("encyclopedia/bx-2.txt");
  this->add("encyclopedia/genon.txt");
  this->add("encyclopedia/light_bydo.txt");
  this->add("encyclopedia/nulaylaf.txt");
  this->add("encyclopedia/nulayragom.txt");
  this->add("encyclopedia/surge.txt");
  this->add("encyclopedia/rva-818.txt");
  this->add("encyclopedia/authors.txt");

  // Create the model.
  this->model = new libfhi::PostModel();
  this->model->set_rotation(libfhi::Orientation::ROTATION_ABSORB);
  this->model->set_pos(0.0f, 0.0f, 0.0f);
}

/** Default destructor walks the pages and deletes them all.
 */
Encyclopedia::~Encyclopedia()
{
  for(std::vector<EncyclopediaPage*>::iterator i = this->pages.begin(),
      e = this->pages.end(); (i != e); ++i)
  {
    delete *i;
  }
  delete this->model;
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Add one encyclopedia page from a file.
 * @param str File to load from.
 */
void Encyclopedia::add(const char *str)
{
  this->pages.push_back(new EncyclopediaPage(str));
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Draw this encyclopedia page.
 * @param screen Screen to draw to.
 */
void Encyclopedia::draw(libfhi::Surface *screen)
{
  // Bound current page.
  int numpages = static_cast<int>(this->pages.size());

  this->current_page = libfhi::congr(this->current_page, numpages);
  
  int prevpage = libfhi::congr(this->current_page - 1, numpages),
      nextpage = libfhi::congr(this->current_page + 1, numpages);

  // Draw to screen.
  this->pages[this->current_page]->draw(screen);

  // Page indicator in the bottom.
  std::stringstream statustext;
  statustext << (this->current_page + 1) << " of " << numpages;

  std::stringstream nexttext;
  nexttext << "Prev: \"" << this->pages[prevpage]->get_name() << "\" Next: \""
    << this->pages[nextpage]->get_name() << "\"";

  // Get the status font.
  libfhi::Font *font = Menu::get_font_desc();

  libfhi::Surface::draw_text(4, screen->get_h() - 5,
      Menu::get_color_text(), font, statustext.str().c_str());

  libfhi::Surface::draw_text(
      screen->get_w() - 2 - 5 -
        font->get_string_width(nexttext.str().c_str()),
      screen->get_h() - 2 - 5,
      Menu::get_color_text(),
      font,
      nexttext.str().c_str());
}

/** Keydown.
 * @param key Key code pressed.
 */
void Encyclopedia::keydown(int key)
{
  // Down.
  if((key == SDLK_DOWN) || (key == SDLK_RIGHT) ||
      (key == Options::key_down) || (key == Options::key_right))
  {
    ++this->current_page;
    this->model->set_angles(-8192, 0, 0);
    this->model->rotation_set(0, 0, 0);
  }
  // Up.
  else if((key == SDLK_UP) || (key == SDLK_LEFT) ||
      (key == Options::key_up) || (key == Options::key_left))
  {
    --this->current_page;
    this->model->set_angles(-8192, 0, 0);
    this->model->rotation_set(0, 0, 0);
  }
  // Escape.
  else if((key == SDLK_ESCAPE) || (key == Options::key_absorb) ||
      (key == SDLK_BACKSPACE))
  {
    command_end_game(); // FIXME: Better name plz.
  }
  // Otherwise some compiled-in shit.
  else switch(key)
  {
    case SDLK_HOME:
      this->model->rotation_add(-256, 0, 0);
      break;
    case SDLK_END:
      this->model->rotation_add(256, 0, 0);
      break;
    case SDLK_PAGEDOWN:
      this->model->rotation_add(0, -256, 0);
      break;
    case SDLK_DELETE:
      this->model->rotation_add(0, 256, 0);
      break;
    case SDLK_PAGEUP:
      this->model->rotation_add(0, 0, -256);
      break;
    case SDLK_INSERT:
      this->model->rotation_add(0, 0, 256);
      break;

    default:
      break;
  }
}

/** Keyup.
 * @param key Key code released.
 */
void Encyclopedia::keyup(int key)
{
  switch(key)
  {
    case SDLK_HOME:
      this->model->rotation_add(256, 0, 0);
      break;
    case SDLK_END:
      this->model->rotation_add(-256, 0, 0);
      break;
    case SDLK_PAGEDOWN:
      this->model->rotation_add(0, 256, 0);
      break;
    case SDLK_DELETE:
      this->model->rotation_add(0, -256, 0);
      break;
    case SDLK_PAGEUP:
      this->model->rotation_add(0, 0, 256);
      break;
    case SDLK_INSERT:
      this->model->rotation_add(0, 0, -256);
      break;

    default:
      break;
  }
}

/** Returning just restarts the music.
 */
void Encyclopedia::return_to()
{
  Music *musa = Music::instance_get();
  musa->Init("music/Onomatopoetic_Justice_-_Short_Sun.ogg");
}

/** Tick for encyclopedia.
 */
void Encyclopedia::tick()
{
  this->model->tick();
}

//############################################################################
// End #######################################################################
//############################################################################

