#include "gamesurvival.h"

#include "commands.h"
#include "data.h"
#include "entity.h"
#include "entityarchtype.h"
#include "menu.h"
#include "weapon.h"

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param filename Filename to load the parameters from.
 */
GameSurvival::GameSurvival(const char *filename)
  : Game(filename), stringbox(NULL), phase(1)
{
  // Create new player.
  Entity *meh = this->entity_new(ENTITY_LIGHTBYDO, FACTION_PLAYER);
  meh->set_pos(VIEW_AREA / 2.0f, VIEW_AREA / 2.0f);

  // Mark the light bydo as player.
  this->player = meh;
  this->follow = meh;

  // Survival mode has 'bullet time', i.e. enemy bullets are way slower.
  Weapon::set_slowdown(GAME_SURVIVAL_BULLET_SLOWDOWN);

  //meh = this->entity_new(ENTITY_RVA818, FACTION_ENEMY);
  //meh->set_pos(6.0f, 6.0f);

  //meh = this->entity_new(ENTITY_INTERDIMENSIONAL_BATTLESHIP, FACTION_ENEMY);
  //meh->set_pos(12.0f, 12.0f);

  //meh = this->entity_new(Data::load_ent("entity/genon/formation_hex.ent"),
  //	FACTION_ENEMY);
  //meh->set_pos(12.0f, 12.0f);

  this->phase_new();
}

/** Default destructor.
 */
GameSurvival::~GameSurvival()
{
  // Do nothing.
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Only draw the backplane, leave the drawing method itself to the base
 * method.
 * @param screen Screen to output to.
 */
void GameSurvival::draw(libfhi::Surface *screen)
{
  // Clear only the depth buffer, since all will be drawn.
  screen->clear(libfhi::Boundary::ZBUFFER_MAX);

  // Use host drawing method for objects.
  this->Game::draw(screen);

  // Use host score drawing to render scores.
  this->draw_score(screen);

  /** If displaying info, do it. */
  if(this->info_display > 0)
  {
    int x1 = 0,
	y1 = 0,
        x2 = screen->get_h() - 1,
    	y2 = screen->get_h() - 1;

    // Darken most of the screen.
    screen->set_boundary(x1, y1, x2, y2);
    screen->select_2d();
    libfhi::Surface::nc_rect(x1, y1, x2, y2,
	libfhi::Surface::makecol4(0, 0, 0, 92));

    Menu::draw_text_box(x1 + 10, y1 + 10, x2 - 10, y2 - 10,
	this->stringbox->str().c_str());
  }
}

/** Ticker for game does a lot of stuff.
 */
void GameSurvival::tick()
{
  if(this->info_display > 0)
  {
    this->pause = true;
    --this->info_display;

    // Make sure player is not dying.
    if(this->player)
    {
      this->player->set_life(0xFFFF); // Much.
      this->player->set_absorb_state(0); // Nullify any wait.
    }
  }
  else if(this->info_display == 0)
  {
    this->pause = false;

    // Reset player life.
    if(this->player)
    {
      this->player->set_life(
	  this->player->get_base_class()->get_life_maximum());
    }

    this->info_display = -1;
  }

  // Call host method to tick players, enemies and weapons.
  Game::tick();

  // End game if reached correct frame.
  if((Game::frame_end) && (Game::frame_number >= Game::frame_end))
  {
    command_end_game();
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Delete an existing entity. Also remove it from the entity table.
 * @param iter Iterator to the entity list in this.
 * @param allow_scoring Allow score accumulation, default to false.
 */
void GameSurvival::entity_delete(std::list<Entity*>::iterator iter,
    bool allow_scoring)
{
  // Actually delete via fallback.
  Game::entity_delete(iter, allow_scoring);

  // If there is only one left (highlander-style), start a new phase.
  if((this->entities.size() == 1) && (this->player))
  {
    this->phase_new();
  }
}

/** Spawn a new branch of enemies. Increment phase by one, display text.
 */
void GameSurvival::phase_new()
{
  // Get terrain.
  Terrain *terrain = Terrain::instance;

  // Inserted entities.
  int inserts[ENTITY_COUNT];
  memset(inserts, 0, sizeof(int) * ENTITY_COUNT);

  // Delete old primer string and reinitialize with new.
  delete this->stringbox;
  this->stringbox = new std::stringstream();
  (*(this->stringbox)) << "Beginning phase " << this->phase <<
    ".\n \nGenerated enemies:\n \n";

  // Set info display.
  this->info_display = GAME_PHASE_TIMEOUT;

  // Generate new entities until points run out.
  int points = GAME_PHASE_POINT_RATIO * this->phase;

  // Generate entity randomization probability numbers.
  int difficulties_phase[GAME_DIFFICULTY_MAX];
  for(int i = 0, total = 0; (i < GAME_DIFFICULTY_MAX); ++i)
  {
    total += GAME_DIFFICULTY_MAX - i;
    difficulties_phase[i] = total;
  }

  while(points > 0)
  {
    // Get new difficulty. Easier enemies are more probable.
    int difficulty = rand() %
      difficulties_phase[stdmin(GAME_DIFFICULTY_MAX - 1, this->phase - 1)];

    for(int i = 0; (i < GAME_DIFFICULTY_MAX); ++i)
    {
      int inverse = GAME_DIFFICULTY_MAX - i;

      if(difficulty > inverse)
      {
	difficulty -= inverse;
      }
      else
      {
	difficulty = i + 1;
	break;
      }
    }

    // Get entity corresponding to the difficulty setting.
    EntityEnum etype = this->get_random_entity(difficulty);
    Entity *ent = this->entity_new(etype, FACTION_ENEMY);
    bool inserted = false;

    for(int cnt = ENTITY_INSERTION_TRY_COUNT; (--cnt);)
    {
      libfhi::Vector2 ipos = terrain->get_random_position();
      uint16_t idir = rand() % 65536;

      libfhi::Vector2 rel = 
	terrain->get_relative_bounded_position(ipos, this->player->get_pos());

      if(this->entity_insert(ent, ipos, idir, true, GAME_SAFE_DISTANCE))
      {
	inserted = true;
	inserts[etype]++;
	break;
      }
    }

    // If not inserted, remove the entity from wasting memory.
    if(inserted)
    {
      points -= difficulty;
    }
    else
    {
      this->Game::entity_delete(ent);
    }
  }

  // List the inserted entities.
  for(int i = 0; (i < static_cast<int>(ENTITY_COUNT)); ++i)
  {
    if(inserts[i])
    {
      (*(this->stringbox)) << inserts[i] << "x " <<
	this->entity_archtypes[i]->get_name() << std::endl;
    }
  }

  // Move on to next phase.
  ++this->phase;
  this->destroy_tick = Game::frame_number - GAME_TIME_BONUS_AREA *
    GAME_TIME_BONUS_GRANULARITY;
}

//############################################################################
// End #######################################################################
//############################################################################

