#include "menu.h"

#include "data.h"
#include "entity.h"
#include "options.h"
#include "terrain.h"

#include "SDL.h"

#include <sstream>

//############################################################################
// Variables #################################################################
//############################################################################

int Menu::color_background,
    Menu::color_border,
    Menu::color_hilight,
    Menu::color_text,
    Menu::color_title,
    Menu::text_x,
    Menu::text_y,
    Menu::text_y_title,
    Menu::fontsize,
    Menu::fontsize_desc,
    Menu::fontsize_title,
    Menu::text_padding;

libfhi::Font
  *Menu::font,
  *Menu::font_desc,
  *Menu::font_title;

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param str The name of the main menu.
 */
Menu::Menu(const char *str)
  : menu(str)
{
  // Set the current.
  this->current = &this->menu;
}

/** Default destructor.
 */
Menu::~Menu()
{
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Implementation of draw.
 * @param screen Screen to draw to.
 */
void Menu::draw(libfhi::Surface *screen)
{
  int x1 = screen->get_h(),
      x2 = screen->get_w() - 1,
      y1 = 0,
      y2 = screen->get_h() - 1;
  MenuItem *parent;

  // Set boundary to the right part of the screen.
  screen->set_boundary(x1, y1, x2, y2);
  screen->select_2d();

  // Draw background.
  libfhi::Surface::nc_rect(x1, y1, x1 + 1, y2, color_border);
  libfhi::Surface::nc_rect(x1 + 2, y1, x2, y2, color_background);

  // Ensure that the current item is not the root of the tree as it is when
  // menus are initially created and the current is set to point to the root.
  parent = this->current->get_parent();
  if(parent == NULL)
  {
    parent = this->current;
    this->current = this->current->get_item(0);
  }

  // Draw the title.
  libfhi::Surface::draw_text(Menu::text_x, Menu::text_y_title,
      this->color_title, Menu::font_title, parent->get_name().c_str());

  // Iterate through the children and draw on the way.
  for(int i = 0, y = Menu::text_y;
      (i < parent->num_items());
      ++i, y += Menu::text_padding + Menu::fontsize)
  {
    MenuItem *item = parent->get_item(i);

    libfhi::Surface::draw_text(Menu::text_x, y,
	(item == this->current) ? this->color_hilight : this->color_text,
	Menu::font, item->get_name().c_str());
  }

  // Return the boundary to the left part of the screen to draw anything in the
  // supermenu.
  screen->set_boundary(0, 0, x1 - 1, x1 - 1);
  screen->select_2d();
}

/** Implementation of keydown.
 * @param key Key pressed.
 */
void Menu::keydown(int key)
{
  MenuItem *parent = this->current->get_parent();

  // Precaution, if no parent is found, abort. It will be okay the next
  // frame.
  if(!parent)
  {
    return;
  }

  // Down.
  if((key == SDLK_DOWN) || (key == SDLK_RIGHT) ||
      (key == Options::key_down) || (key == Options::key_right))
  {
    int idx = this->current->get_index();

    if(++idx >= parent->num_items())
    {
      this->current = parent->get_item(0);
    }
    else
    {
      this->current = parent->get_item(idx);
    }
  }
  // Up.
  else if((key == SDLK_UP) || (key == SDLK_LEFT) ||
      (key == Options::key_up) || (key == Options::key_left))
  {
    int idx = this->current->get_index();

    if(--idx < 0)
    {
      this->current = parent->get_item(parent->num_items() - 1);
    }
    else
    {
      this->current = parent->get_item(idx);
    }
  }
  // Escape.
  else if((key == SDLK_ESCAPE) || (key == Options::key_absorb) ||
      (key == SDLK_BACKSPACE))
  {
    if(parent->get_parent())
    {
      this->current = parent;
    }
  }
  // Descend.
  else if((key == SDLK_RETURN) || (key == Options::key_fire))
  {
    MenuItem *item = this->current->execute();

    if(item)
    {
      this->current = item;
    }
  }
}

//############################################################################
// Other methods #############################################################
//############################################################################

/** Draw one 'charge' bar.
 * @param y Starting point of the charge bar.
 * @param percent How long is the bar (fraction from maximum).
 * @param top Maximum value.
 * @param col Color.
 * @param str Name to display on the left.
 * @param screen Screen to draw to.
 * @return Y coordinate of the next charge bar.
 */
int Menu::draw_charge_bar(int y, float percent, int col, const char *str,
    libfhi::Surface *screen)
{
  // Print name.
  int name_x = screen->get_h() + 2 + 1,
      name_y = y + fontsize_desc + 1,
      name_w = libfhi::Surface::draw_text(name_x, name_y, Menu::color_border,
	  Menu::font_desc, str);

  // Now can derieve latter ones.
  int y2 = y + fontsize_desc + 2,
      x1 = name_x + name_w + 1 + 3,
      x2 = screen->get_w() - 1,
      full_w = (x2 - x1),
      w = static_cast<int>(percent * static_cast<float>(full_w));

  // Print vertical border.
  libfhi::Surface::nc_rect(x1 - 2, y, x1 - 1, y2, Menu::color_border);

  if(w)
  {
    libfhi::Surface::nc_rect(x1, y, x1 + w, y2, col);
  }
  if(w != full_w)
  {
    libfhi::Surface::nc_rect(x1 + w + 1, y, x2, y2, 0);
  }
  libfhi::Surface::nc_rect(name_x - 1, y2 + 1, x2, y2 + 2, Menu::color_border);

  return y2 + 3;
}

/** Draw the map. This means, only the entities, not the background (yet).
 * @param x1 First coordinate available in x.
 * @param x2 Last coordinate available in x.
 * @param lst List of all entities in game.
 * @return The first available y coordinate after the map has been drawn,
 */
int Menu::draw_map(int x1, int x2, const libfhi::Vector2 &pos,
    std::list<Entity*> *lst)
{
  static int own_color = libfhi::Surface::makecol3(255, 0, 0),
	     enemy_color = libfhi::Surface::makecol3(0, 255, 255);

  Terrain *terrain = Terrain::instance;

  int w = x2 - x1 - 2,
      ret = w + 2 + 2,
      y1 = 2;
  float wfloat = static_cast<float>(w),
    	whalf = wfloat * 0.5f,
    	centerx = static_cast<float>(x1) + whalf,
	centery = static_cast<float>(y1) + whalf,
	xmul = wfloat / terrain->get_size_x(),
	ymul = wfloat / terrain->get_size_y();

  // Draw borders.
  libfhi::Surface::nc_rect(x1, 0, x2, 1, Menu::color_border);
  libfhi::Surface::nc_rect(x1, w + 2, x2, w + 3, Menu::color_border);
  libfhi::Surface::nc_rect(x2 - 1, 2, x2, w + 1, Menu::color_border);

  for(std::list<Entity*>::iterator i = lst->begin(),
      e = lst->end(); (i != e); ++i)
  {
    libfhi::Vector2 opos = terrain->get_relative_bounded_position(
	(*i)->get_pos(), pos);

    int
      ix = static_cast<int>(centerx - opos.xf * xmul),
      iy = static_cast<int>(centery + opos.yf * ymul);

    libfhi::Surface::rect(ix - 1, iy - 1, ix + 1, iy + 1,
	((*i)->get_faction() & FACTION_ENEMY) ? enemy_color : own_color);
  }

  return ret;
}

/** Implementation of draw_text_box for menu. Defines draw borders and outputs
 * text inside them.
 * @param x1 Box to draw into - left border
 * @param y1 Box to draw into - top border
 * @param x2 Box to draw into - right border
 * @param y2 Box to draw into - bottom border
 * @param input Text to draw into the box.
 * @return True if all of the text fit in the box, false otherwise (BROKEN).
 */
bool Menu::draw_text_box(int x1, int y1, int x2, int y2, const char* input,
    int col)
{
  std::vector<std::string> lines;

  // Copy input.
  std::string txtcopy(input);

  // Loop for copying the data
  size_t last = 0;
  do {
    size_t next = txtcopy.find("\n", last);

    // Reached end.
    if(next == std::string::npos)
    {
      lines.push_back(txtcopy.substr(last));
      break;
    }
    else
    {
      lines.push_back(txtcopy.substr(last, (next - last)));
      last = next + 1;
    }
  } while(true);

  // Some other variables.
  int cury = y1 + Menu::fontsize_desc,
      width = x2 - x1,
      space_len = Menu::font_desc->get_string_width(" ");

  for(std::vector<std::string>::iterator i = lines.begin(),
      e = lines.end(); (i != e); ++i)
  {
    // Use this for output.
    std::stringstream *nexttext = new std::stringstream();

    // Make a copy of the current token for further tokenization.
    char *token_copy = new char[(*i).length() + 1];
    strcpy(token_copy, (*i).c_str());

    for(char *tok = strtok(token_copy, " "); (tok != NULL);
	tok = strtok(NULL, " "))
    {
      int token_len = Menu::font_desc->get_string_width(tok),
	  nexttext_len = Menu::font_desc->get_string_width(
	      nexttext->str().c_str());

      if(nexttext->str().length() <= 0)
      {
	(*nexttext) << tok;
      }
      else if((token_len + nexttext_len + space_len) <= width)
      {
	(*nexttext) << " " << tok;
      }
      else
      {
        libfhi::Surface::draw_text(x1, cury, col, Menu::font_desc,
	      nexttext->str().c_str());

	delete nexttext;
        nexttext = new std::stringstream();
	(*nexttext) << tok;

	// Advance.
        cury += Menu::fontsize_desc;
	if(cury > y2)
	{
	  return false;
	}
      }
    }

    if(nexttext->str().length() > 0)
    {
      libfhi::Surface::draw_text(x1, cury, col, Menu::font_desc,
	  nexttext->str().c_str());      
    }

    // Delete the copy of this token.
    delete[] token_copy;
    delete nexttext;

    // Advance.
    cury += Menu::fontsize_desc;
    if(cury > y2)
    {
      return false;
    }
  }

  return true;
}

/** Menus have some variables common to them all menus, containing the
 * info on sizes, etc. These have to be initialized with this method.
 * @param screen Surface to use to determine menu variables.
 */
bool Menu::init(libfhi::Surface *screen)
{
  // Init resolutions from screen.
  int xres = screen->get_w(),
      yres = screen->get_h();

  // Resolution determines font size.
  float diff = static_cast<float>(xres - yres);
  Menu::fontsize_title = libfhi::round(diff * 0.13f);
  Menu::fontsize = libfhi::round(diff * 0.11f);
  Menu::fontsize_desc = libfhi::round(diff * 0.08f);
  Menu::text_padding = libfhi::round(diff * 0.03f); 

  // Load fonts.
  Data::chdir_to_datadir();
  Menu::font = libfhi::font_new("DejaVu-Sans-Bold.ttf");
  Menu::font_desc = libfhi::font_new("DejaVu-Sans-Bold.ttf");
  Menu::font_title = libfhi::font_new("DejaVu-Sans-Bold.ttf");
  Menu::font->set_size(Menu::fontsize);
  Menu::font_desc->set_size(Menu::fontsize_desc);
  Menu::font_title->set_size(Menu::fontsize_title);

  // Set the borders of text.
  Menu::text_x = yres + 1 + Menu::text_padding;
  Menu::text_y_title = Menu::text_padding + Menu::fontsize_title;
  Menu::text_y = Menu::text_y_title + 2 * Menu::text_padding + Menu::fontsize;

  // Set default colors.
  Menu::color_background = libfhi::Surface::makecol3(30, 30, 30);
  Menu::color_border = libfhi::Surface::makecol3(100, 100, 100);
  Menu::color_hilight = libfhi::Surface::makecol3(230, 50, 50);
  Menu::color_text = libfhi::Surface::makecol3(192, 192, 192);
  Menu::color_title = libfhi::Surface::makecol3(218, 165, 32);

  // Successful.
  return true;
}

//############################################################################
// End #######################################################################
//############################################################################

