#ifndef menu_h
#define menu_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "eventhandler.h"
#include "menuitem.h"

#include "libfhi.h"

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class Entity;

//############################################################################
// Class #####################################################################
//############################################################################

/** Menu is a superclass of all menu constructs. It handles displaying menu
 * strings and descending or advancing into submenus. Menu also may take care
 * of displaying information in the main view if neccessary.
 */
class Menu :
  public EventHandler
{
    // Upon initialization, menu generates the correct font sizes & such,
    // they are saved to these staric menu variables.
  protected:
    /** Color of the background of this menu. */
    static int color_background;

    /** Color of the border of this menu. */
    static int color_border;

    /** Color of the hilight of this menu. */
    static int color_hilight;

    /** Color of the text of this menu. */
    static int color_text;

    /** Color of the title. */
    static int color_title;

    /** Menu text left border. */
    static int text_x;

    /** Menu text top border. */
    static int text_y;

    /** Menu title top border. */
    static int text_y_title;

    /** Menu text size. */
    static int fontsize;

    /** Menu title size. */
    static int fontsize_desc;

    /** Menu title size. */
    static int fontsize_title;

    /** Padding between menu rows. */
    static int text_padding;

    /** Menu font index. */
    static libfhi::Font* font;

    /** Menu font index. */
    static libfhi::Font* font_desc;

    /** Title font index. */
    static libfhi::Font* font_title;

  protected:
    /** List containing all menu items here. */
    MenuItem menu;

    /** Current menu item hilighted. */
    MenuItem *current;

  public:
    Menu(const char*);
    ~Menu();

    // Static methods.
  public:
    static int draw_charge_bar(int, float, int, const char*,
	libfhi::Surface*);
    static int draw_map(int, int, const libfhi::Vector2&,
	std::list<Entity*>*);
    static bool draw_text_box(int, int, int, int, const char*,
	int = Menu::get_color_text());
    static bool init(libfhi::Surface*);

    // Virtual methods derieved from EventHandler.
  public:
    virtual void draw(libfhi::Surface*);
    virtual void keydown(int key);

    // Inline methods.
  public:
    static inline int get_color_border();
    static inline int get_color_text();
    static inline int get_color_title();
    static inline libfhi::Font* get_font();
    static inline libfhi::Font* get_font_desc();
    static inline int get_fontsize();
    static inline int get_fontsize_desc();
    static inline int get_text_bottom_y(const libfhi::Surface*);
    static inline int get_text_x();
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get the menu border color.
 * @return color as int.
 */
int Menu::get_color_border()
{
  return Menu::color_border;
}

/** Get the menu text color.
 * @return color as int.
 */
int Menu::get_color_text()
{
  return Menu::color_text;
}

/** Get the menu title color.
 * @return color as int.
 */
int Menu::get_color_title()
{
  return Menu::color_title;
}

/** Return the normal drawing font.
 * @return Font index.
 */
libfhi::Font* Menu::get_font()
{
  return Menu::font;
}

/** Return the description font.
 * @return Font index.
 */
libfhi::Font* Menu::get_font_desc()
{
  return Menu::font_desc;
}

/** Return the font size.
 * @return Font size as integer.
 */
int Menu::get_fontsize()
{
  return Menu::fontsize;
}

/** Return the description font size.
 * @return Font size as integer.
 */
int Menu::get_fontsize_desc()
{
  return Menu::fontsize_desc;
}

/** Return the x position of text.
 * @return X position as integer.
 */
int Menu::get_text_x()
{
  return Menu::text_x;
}

/** Return the lowest place to draw text given the screen.
 * @param screen Screen to give the lowest position to.
 * @return Font size of text.
 */
int Menu::get_text_bottom_y(const libfhi::Surface *screen)
{
  return screen->get_h() - 1 - Menu::fontsize;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

