#ifndef libfhi_configfile_include
#define libfhi_configfile_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_default.h"

#include <vector>
#include <string>

namespace libfhi {

//############################################################################
// Class #####################################################################
//############################################################################

/** ConfigFile represents one file that is read and parsed for values, each
 * value is stored as integer, floating point and string representation.
 */
class ConfigFile
{
  private:
    /** Integers. */
    std::vector<int> vint;

    /** Floats. */
    std::vector<float> vfloat;

    /** Strings. */
    std::vector<std::string> vstring;

    /** File name we opened. */
    std::string filename;

    /** Line we are in. */
    int line;

    /** The identifier, that is, the first token encoutered in a line. */
    std::string identifier;

    /** File handle. */
    FILE *filu;

  public:
    ConfigFile(const char*);
    ~ConfigFile();

    // Methods.
  public:
    bool advance();
    std::string get_combined_string();
    bool has_id_arg(const char*, size_t) const;
    void warn(const char*) const;
    void warn_empty();

    // Inline.
  public:
    inline float get_float(size_t) const;
    inline int get_int(size_t) const;
    inline const std::string& get_identifier() const;
    inline size_t get_num_arg() const;
    inline const std::string& get_string(size_t) const;
    inline bool has_id(const char*) const;
    inline bool is_ok() const;
};

//############################################################################
// ConfigFile inline #########################################################
//############################################################################

/** Get float by index. Segfault if out of bounds.
 * @param Index of float to return.
 * @return The float in given slot.
 */
float ConfigFile::get_float(size_t op) const
{
  return this->vfloat[op];
}

/** Get the integer by index. Segfault if out of bounds.
 * @param op Index of integer to return.
 * @return The integer in given slot.
 */
int ConfigFile::get_int(size_t op) const
{
  return this->vint[op];
}

/** Get the identifier string.
 * @return Reference to the identifier string.
 */
const std::string& ConfigFile::get_identifier() const
{
  return this->identifier;
}

/** Get number of parameters extracted.
 * @return Result as size_t.
 */
size_t ConfigFile::get_num_arg() const
{
  // They're all of same length.
  return this->vstring.size();
}

/** Get the string by index. Segfault if out of bounds.
 * @param op Index of string to return.
 * @return Pointer to string in given slot.
 */
const std::string& ConfigFile::get_string(size_t op) const
{
  return this->vstring[op];
}

/** Tell if the identifier compares to given string.
 * @param op Character string to compare to.
 * @return True if yes, false if not.
 */
bool ConfigFile::has_id(const char *op) const
{
  return (this->identifier.compare(op) == 0);
}

/** Tell if there is a file open here.
 * @return True if yes, false if no.
 */
bool ConfigFile::is_ok() const
{
  return (this->filu != NULL);
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

