#ifndef libfhi_fixed_include
#define libfhi_fixed_include

//############################################################################
// Includes ##################################################################
//############################################################################

#include "libfhi_default.h"

namespace libfhi {

//############################################################################
// Defines ###################################################################
//############################################################################

/** Fixed point shift value is half of the 32-bit integer. */
#define LIBFHI_FIXED_SHIFT	15

//############################################################################
// Fixed #####################################################################
//############################################################################

/** A fixed point math class, basically fully inline and written here as such.
 * Notice that this is an infinitely simplified model, the class is not meant
 * for arbituary fixed point calculation, but rather as a supplementary to the
 * software renderer. For example, transformation to an integer while the
 * value is < 0 gives an erraneous result, moreover, most of the basic math
 * operations are not implemented.
 */
class Fixed
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Fixed)
#endif

  private:
    /** The value that resembles the shift in floating point. */
    static const float FIXED_MUL;

    /** Half variable. */
    static const int32_t FIXED_HALF = 1 << (LIBFHI_FIXED_SHIFT - 1);

    /** One variable. */
    static const int32_t FIXED_ONE = 1 << LIBFHI_FIXED_SHIFT;

    /** The value of this, must be signed int32. */
    /*union
    {
      int32_t val;

      struct
      {
	uint16_t i161, i162;
      };
    };*/
    int32_t val;

  public:
    // Functions.
    inline int32_t get_val() const;
    inline void inc_roundable();
    inline bool is_null() const;
    inline bool not_null() const;
    inline void set_null();
    inline void set_roundable(int);
    inline void set_roundable(float);
    inline void set_val(int32_t);

    // Static functions.
    static inline const Fixed from(int);
    static inline const Fixed from(float);
    static inline const Fixed abs(const Fixed&);

    // Operators.
    inline const Fixed operator+(const Fixed&) const;
    inline const Fixed operator+(int) const;
    inline const Fixed operator-(const Fixed&) const;
    inline const Fixed operator-(int) const;
    inline const Fixed operator*(int) const;
    inline const Fixed operator/(int) const;
    inline void operator+=(const Fixed&);
    inline void operator+=(int);
    inline void operator-=(const Fixed&);
    inline void operator-=(int);
    inline void operator*=(int);
    inline void operator/=(int);
    inline void operator++();
    inline void operator--();
    inline operator int() const;
    inline operator uint16_t() const;
    inline bool operator==(int32_t) const;
    inline bool operator==(const Fixed&) const;
    inline bool operator!=(const Fixed&) const;
    inline bool operator<=(const Fixed&) const;
    inline bool operator>=(const Fixed&) const;
    inline bool operator<(const Fixed&) const;
    inline bool operator>(const Fixed&) const;
};

//############################################################################
// Static functions ##########################################################
//############################################################################

/** Creates a fixed point variable from an integer.
 * @param op Integer to convert.
 * @return The converted fixed point variable.
 */
const Fixed Fixed::from(int op)
{
  Fixed ret;

  ret.val = op << LIBFHI_FIXED_SHIFT;

  return ret;
}

/** Creates a fixed point variable from a floating point number.
 * @param op Floating point value to convert.
 * @return The converted fixed point variable.
 */
const Fixed Fixed::from(float op)
{
  Fixed ret;

  ret.val = static_cast<int32_t>(op * FIXED_MUL);

  return ret;
}

/** Calculate the absolute value of given fixed point value.
 * @param op Fixed point variable to get the absolute value of.
 */
const Fixed Fixed::abs(const Fixed& op)
{
  Fixed ret;

  if(op.val < 0)
  {
    ret.val = -op.val;
  }
  else
  {
    ret.val = op.val;
  }

  return ret;
}

//############################################################################
// Assignments ###############################################################
//############################################################################

/** Return the value as-is with no conversion.
 * @return Signed 32-bit integer containing the value.
 */
int32_t Fixed::get_val() const
{
  return val;
}

/** Set the actual (already fixed point) value.
 * @param op The value this is set to.
 */
void Fixed::set_val(int32_t op)
{
  this->val = op;
}

/** Nullify this variable.
 */
void Fixed::set_null()
{
  this->val = 0;
}

//############################################################################
// Local arimethic ###########################################################
//############################################################################

/** Addition of two fixed-point variables.
 * @param rhs Right-hand side of the operation.
 */
const Fixed Fixed::operator+(const Fixed& rhs) const
{
  Fixed ret;

  ret.val = this->val + rhs.val;

  return ret;
}

/** Addition of a fixed-point variable and an integer.
 * @param rhs Right-hand side of the operation.
 */
const Fixed Fixed::operator+(int rhs) const
{
  Fixed ret;

  ret.val = this->val + (rhs << LIBFHI_FIXED_SHIFT);

  return ret;
}

/** Substraction of two fixed-point variables.
 * @param rhs Right-hand side of the operation.
 */
const Fixed Fixed::operator-(const Fixed& rhs) const
{
  Fixed ret;

  ret.val = this->val - rhs.val;

  return ret;
}

/** Substraction of a fixed-point variable and an integer.
 * @param rhs Right-hand side of the operation.
 */
const Fixed Fixed::operator-(int rhs) const
{
  Fixed ret;

  ret.val = this->val - (rhs << LIBFHI_FIXED_SHIFT);

  return ret;
}

/** Multiplification of a fixed point variable and an integer.
 * @param rhs The integer to multiply with.
 */
const Fixed Fixed::operator*(int rhs) const
{
  Fixed ret;

  ret.val = this->val * rhs;

  return ret;
}
    
/** Division of a fixed point variable and an integer.
 * @param rhs The integer to divide with.
 */
const Fixed Fixed::operator/(int rhs) const
{
  Fixed ret;

  ret.val = this->val / rhs;

  return ret;
}

/** Addition of two fixed-point variables.
 * @param rhs Right-hand side of the operation.
 */
void Fixed::operator+=(const Fixed& rhs)
{
  this->val += rhs.val;
}

/** Addition of a fixed point variable and an integer.
 * @param rhs Right-hand side of the operation.
 */
void Fixed::operator+=(int rhs)
{
  this->val += rhs << LIBFHI_FIXED_SHIFT;
}

/** Substract two fixed point values from each other.
 * @param rhs Right-hand side of the operation.
 */
void Fixed::operator-=(const Fixed& rhs)
{
  this->val -= rhs.val;
}

/** Substraction of a fixed point variable and an integer.
 * @param rhs Integer that will be shifted before addition.
 */
void Fixed::operator-=(int rhs)
{
  this->val -= rhs << LIBFHI_FIXED_SHIFT;
}

/** Multiply this variable by the given integer.
 * @param rhs The integer to multiply with.
 */
void Fixed::operator*=(int rhs)
{
  this->val *= rhs;
}
    
/** Divide this variable by the given integer.
 * @param rhs The integer to divide with.
 */
void Fixed::operator/=(int rhs)
{
  this->val /= rhs;
}

/** Increment this variable by one.
 */
void Fixed::operator++()
{
  this->val += FIXED_ONE;
}

/** Decrement this variable by one.
 */
void Fixed::operator--()
{
  this->val -= FIXED_ONE;
}

//############################################################################
// Conversion ################################################################
//############################################################################

/** Conversion the local native integer type is assumed 32-bit signed.
 * @return The integer value, no rounding.
 */
Fixed::operator int() const
{
  return val >> LIBFHI_FIXED_SHIFT;
}

/** Conversion to an 16-bit unsigned integer.
 * @return The integer value, no rounding.
 */
Fixed::operator uint16_t() const
{
  return static_cast<uint16_t>(val >> LIBFHI_FIXED_SHIFT);
}

//############################################################################
// Comparison ################################################################
//############################################################################

/** The comparison against an integer is actually a comparison against the
 * value directly.
 * @param rhs Integer to compare the value to.
 * @return True if equal, false if not.
 */
bool Fixed::operator==(int32_t rhs) const
{
  return (this->val == rhs);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs == rhs, false otherwise.
 */
bool Fixed::operator==(const Fixed &rhs) const
{
  return (this->val == rhs.val);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs != rhs, false otherwise.
 */
bool Fixed::operator!=(const Fixed &rhs) const
{
  return (this->val != rhs.val);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs <= rhs, false otherwise.
 */
bool Fixed::operator<=(const Fixed& rhs) const
{
  return (this->val <= rhs.val);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs >= rhs, false otherwise.
 */
bool Fixed::operator>=(const Fixed& rhs) const
{
  return (this->val >= rhs.val);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs < rhs, false otherwise.
 */
bool Fixed::operator<(const Fixed& rhs) const
{
  return (this->val < rhs.val);
}

/** Compare two fixed point variables.
 * @param rhs Right-hand side of the comparison.
 * @return True if lhs > rhs, false otherwise.
 */
bool Fixed::operator>(const Fixed &rhs) const
{
  return (this->val > rhs.val);
}

/** Tell if this is zero.
 * @return True if zero, false if not.
 */
bool Fixed::is_null() const
{
  return (this->val == 0);
}

/** Tell if this is nonzero.
 * @return True if nonzero, false if zero.
 */
bool Fixed::not_null() const
{
  return (this->val != 0);
}

//############################################################################
// Misc ######################################################################
//############################################################################

/** Add a half to this variable. This is made to increase performance,
 * since the rounding must be done anyway, it is more advantageous to
 * increment the base variable by half in the beginning, thus resulting
 * in all further conversion operations returning 'rounded' values if
 * the variable is only added and substracted from.
 */
void Fixed::inc_roundable()
{
  val += FIXED_HALF;
}

/** Like the inc_roundable, but assigns an integer value then immediately
 * adds the rounding to it.
 * @param op The integer to set.
 */
void Fixed::set_roundable(int op)
{
  val = (op << LIBFHI_FIXED_SHIFT) + FIXED_HALF;
}

/** Like the inc_roundable, but assigns a floating point value then
 * immediately adds the rounding to it.
 * @param op The integer to set.
 */
void Fixed::set_roundable(float op)
{
  this->val = static_cast<int32_t>(op * FIXED_MUL) + FIXED_HALF;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif
