#include "libfhi_light.h"
#include "libfhi_postmodel.h"
#include "libfhi_texture.h"

#ifdef LIBFHI_OPENGL
#include LIBFHI_GLEW_H
#endif

namespace libfhi {

//############################################################################
// Vakiot ####################################################################
//############################################################################

CVUnit *Light::cvdata = NULL;
size_t Light::cvheap = 0;

const Color4
  LightDirectional::DEFAULT_AMBIENT(0.2f, 0.2f, 0.2f, 1.0f),
  LightDirectional::DEFAULT_DIFFUSE(1.0f, 1.0f, 1.0f, 1.0f);

const Vector3
  LightDirectional::DEFAULT_DV(0.0f, 1.0f, 0.0f);

//############################################################################
// Base class methods ########################################################
//############################################################################

/** Default constructor. */
Light::Light()
{
  // Do nothing.
}

/** Default destructor. */
Light::~Light()
{
  // Do nothing.
}

/** Ensure that the vertex transformation heap is at leas of the given size.
 * @param op New heap size (minimum).
 * @return The reserved vertex array.
 */
CVUnit* Light::ensure_heap(size_t op)
{
  if(op > cvheap)
  {
    if(cvdata)
    {
      delete[] cvdata;
    }
    cvdata = new CVUnit[op];
    cvheap = op;
  }
  return cvdata;
}

//############################################################################
// Luokan funktiot ###########################################################
//############################################################################

/** Default constructor. */
LightDirectional::LightDirectional() :
  ambient(DEFAULT_AMBIENT), diffuse(DEFAULT_DIFFUSE)
{
  // Do nothing.
}

/** Constructor with angles.
 * @param x Initial x rotation.
 * @param y Initial y rotation.
 * @param z Initial z rotation.
 */
LightDirectional::LightDirectional(uint16_t x, uint16_t y, uint16_t z) :
  ambient(DEFAULT_AMBIENT), diffuse(DEFAULT_DIFFUSE)
{
  set_angles(x, y, z);
}

/** Constructor with positions.
 * @param op_pos Initial position.
 */
LightDirectional::LightDirectional(const Vector3& op_pos) :
  ambient(DEFAULT_AMBIENT), diffuse(DEFAULT_DIFFUSE)
{
  set_dir(op_pos);
}

/** Default destructor. */
LightDirectional::~LightDirectional()
{
  // Do nothing.
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** LightDirectional implementation of transform. Color transformation does
 * not take alpha channel into account.
 * @param cm Camera matrix.
 * @param mdl Model.
 * @param mesh Mesh.
 * @return Transformed vertex array in C3F_V3F format.
 */
CVUnit* LightDirectional::transform(const Matrix44& cm, PostModel *mdl,
    Mesh *mesh)
{
  Color4 *col;
  Vector3 lit, *nor, *ver;
  Matrix44 wm;

  // Get arrays from model.
  col = mesh->get_array_color();
  nor = mesh->get_array_vnormal();
  ver = mesh->get_array_vertex();

  // Rotation determines how we get light.
  if(mdl->has_rotation())
  {
    // Use wm as temp for translating light and reference pos.
    wm.transpose(mdl->get_om());
    lit = wm * this->get_dv();
  }
  else
  {
    // If no rotation, light vector is good as-is.
    lit = this->get_dv();
  }
  // World matrix needs multiplification even if it is only a translation.
  wm = cm * mdl->get_om();

  // Transform all vertices and store them in target array in C3F_V3F format.
  size_t i = mesh->get_num_vertex();
  CVUnit *dst = Light::ensure_heap(i);
  do {
    float cosine = lit.product_dot(*nor);

    if(cosine < 0.0f)
    {
      cosine = 0.0f;
    }

    // Color operation.
    float tc[3] =
    {
      (this->mul_ambient.rf + (this->mul_diffuse.rf * cosine)) * col->rf,
      (this->mul_ambient.gf + (this->mul_diffuse.gf * cosine)) * col->gf,
      (this->mul_ambient.bf + (this->mul_diffuse.bf * cosine)) * col->bf
    };

    dst->col[0] = stdmin(tc[0], 255.0f);
    dst->col[1] = stdmin(tc[1], 255.0f);
    dst->col[2] = stdmin(tc[2], 255.0f);
    dst->ver = wm * (*ver);

    ++col;
    ++nor;
    ++ver;
    ++dst;
  } while(--i);

  // Return transformed heap.
  return Light::cvdata;
}

/** Light needs to tick a bit more. In this case, prepare the multiplied
 * ambient and diffuse values for software rendering.
 */
void LightDirectional::tick()
{
  // Prepare for drawing.
  this->mul_ambient.rf = this->ambient.rf * 255.0f;
  this->mul_ambient.gf = this->ambient.gf * 255.0f;
  this->mul_ambient.bf = this->ambient.bf * 255.0f;
  this->mul_diffuse.rf = this->diffuse.rf * 255.0f;
  this->mul_diffuse.gf = this->diffuse.gf * 255.0f;
  this->mul_diffuse.bf = this->diffuse.bf * 255.0f;

  // Call original tick method.
  Orientation::tick();
}

//############################################################################
// OpenGL virtual methods ####################################################
//############################################################################

#ifdef LIBFHI_OPENGL

/** LightDirectional implementation of OpenGL transformations. With OpenGl,
 * the colors are treated as full 4 color component implementations.
 * @param mdl Model.
 * @param mesh Mesh.
 */
void LightDirectional::transform_gl(PostModel *mdl, Mesh *mesh)
{
  // Set pointers to this object's data.
  glEnableClientState(GL_COLOR_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glEnableClientState(GL_VERTEX_ARRAY);
  glColorPointer(4, GL_FLOAT, 0, mesh->get_array_color());
  glNormalPointer(GL_FLOAT, 0, mesh->get_array_vnormal());
  glVertexPointer(3, GL_FLOAT, 0, mesh->get_array_vertex());

  // If mesh has texture, always bind it.
  Texture *tex = mesh->get_texture();
  if(tex)
  {
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, tex->get_texture_name());

    // If mesh has point sprites, disable coordinate arrays.
    if(mesh->has_point_sprites())
    {
      glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    }
    // Otherwise set them to this mesh.
    else
    {
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
      glTexCoordPointer(2, GL_FLOAT, 0, mesh->get_array_texcoord());
    }
  }
  // If no texture, disable textures altogether.
  else
  {
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisable(GL_TEXTURE_2D);
  }

  // Multiply topmost matrix with the object matrix.
  glMultMatrixf(mdl->get_om().f);
}

/** OpenGL tick implementation for directional light.
 */
void LightDirectional::tick_gl()
{
  static float specularfv[] =
  {
    0.0f,
    0.0f,
    0.0f,
    1.0f
  };
  float ambientfv[] =
  {
    this->ambient.rf,
    this->ambient.gf,
    this->ambient.bf,
    this->ambient.af
  };
  float diffusefv[] =
  {
    this->diffuse.rf,
    this->diffuse.gf,
    this->diffuse.bf,
    this->diffuse.af
  };
  Vector3 dir = this->get_dv();
  float directionfv[] =
  {
    dir.xf,
    dir.yf,
    dir.zf,
    0.0f
  };

  // Get the orientation of the light.
  Orientation::tick();

  // Enable OpenGL light 0 as the directional light.
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  // Enable color material for ambient and diffuse.
  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

  // Disable specular, emission and global ambient.
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, specularfv);
  glMaterialfv(GL_FRONT, GL_SPECULAR, specularfv);
  glMaterialfv(GL_FRONT, GL_EMISSION, specularfv);

  // Set light colors and direction.
  glLightfv(GL_LIGHT0, GL_AMBIENT, ambientfv);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, diffusefv);
  glLightfv(GL_LIGHT0, GL_POSITION, directionfv);
}

#endif

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print this light into given stream.
 * @param s Stream to output to.
 * @return Modified stream.
 */
std::ostream& LightDirectional::print(std::ostream &s) const
{
  Orientation::print(s);
  return s << "\nAmbient: " << ambient << "\nDiffuse: " << diffuse;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

}

