#ifndef libfhi_default_include
#define libfhi_default_include

//############################################################################
// Includes and conditional definitions ######################################
//############################################################################

// Needed at least for NULL.
#include <stdio.h>

// Math must be included, also algorithms.
#define _USE_MATH_DEFINES // Enable mathematical constants in VC++.
#ifndef WIN32 // Depend on float_cast.h on VC++, otherwise use math.h.
#define HAVE_LRINT 1
#define HAVE_LRINTF 1
#endif
#include "libfhi_float_cast.h" // Thank you Erik de Castro Lopo.
#include <algorithm>

// Include <windows.h> and do some ANNOYING stuff abut VC++ namespace crap.
#ifdef WIN32
#include <windows.h>
#define stdmin min
#define stdmax max
#else
#define stdmin std::min
#define stdmax std::max
#endif

// Declare standard integer types.
#ifdef HAVE_STDINT_H
#include <stdint.h>
#else
#ifdef HAVE_SYS_INTTYPES_H
#include <sys/inttypes.h>
#else
#ifdef LIBFHI_SDL
#include "SDL.h" // SDL.h contains integer types.
#else
#include "pstdint.h"
#endif
#endif
#endif

// If debug is on, we need printing.
#ifdef LIBFHI_DEBUG
#ifdef HAVE_OSTREAM
#include <ostream>
#else
#include <iostream>
#endif
#endif

// Define OpenGL include files, these change depending on the platform.
#ifdef __APPLE__
#define LIBFHI_GL_H "gl.h"
#define LIBFHI_GLU_H "glu.h"
#define LIBFHI_GLEW_H "GL/glew.h"
#else
// Alternatively, Win32
#ifdef WIN32
#define LIBFHI_GL_H "GL/gl.h"
#define LIBFHI_GLU_H "GL/glu.h"
#define LIBFHI_GLEW_H "GL/glew.h"
// Default is X
#else 
#define LIBFHI_GL_H "GL/gl.h"
#define LIBFHI_GLU_H "GL/glu.h"
#define LIBFHI_GLEW_H "GL/glew.h"
#endif
#endif

// Hash maps.
#ifdef WIN32
#define HASHMAP_INCLUDE <hash_map>
#define HASHMAP stdext::hash_map
#define HASHMULTIMAP stdext::hash_multimap
#define HASHFUNC stdext::hash
// FIXME: stuff here
#else // Assuming we have GCC
#define HASHMAP_INCLUDE <ext/hash_map>
#define HASHMAP __gnu_cxx::hash_map
#define HASHMULTIMAP __gnu_cxx::hash_multimap
#define HASHFUNC __gnu_cxx::hash
#endif

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

#define LIBFHI_HEAP_POLY_C 4 // Quad will be enough.
#define LIBFHI_PNG_BYTE uint8_t

//############################################################################
// Makrot ####################################################################
//############################################################################

/*
 * Create prototypes for debug output functions.
 */
#define DEBUG_PRINT_PROTOTYPE(Type) \
  std::ostream& print(std::ostream&) const; \
  inline friend std::ostream& operator<<(std::ostream &s, const Type &op) \
  { \
    return op.print(s); \
  }

//############################################################################
// Inline-funktiot ###########################################################
//############################################################################

/** Destroy target object if it is not null, and initialize it to zero.
 * @param op Pointer to single object reserved with new.
 */
template <class Type>
static inline void zdel(Type *&op)
{
  delete op;
  op = NULL;
}

/** Destroy target array if it is not null and initialize it to zero.
 * @param op Pointer to an array of objects reserved with new.
 */
template <class Type>
static inline void zdel_array(Type *&op)
{
  delete[] op;
  op = NULL;
}

/** Calculates a new heap value based on the current heap size, desired heap
 * size and minimum heap size. The advancement is done in given steps to
 * ensure that each removal or increment does not force a new reservation.
 * The heap size can be smaller than min and it does not affect anything,
 * however, the size will not be lowered below min.
 * The amortized complexity of this schema is O(1)
 * @param heap [in] The current heap size.
 * @param num [in] The desired heap size.
 * @param min [in] The minimum heap size.
 * @return The new heap size, may be same as heap.
 */  
static inline int calculate_new_heap(int heap, int num, int min)
{
  int ret;

  // If heap is at zero, return the desired value.
  if(heap == 0)
  {
    return num;
  }

  // If the number is smaller than one fourth of the heap value, decrease.
  ret = heap >> 2;
  if(num <= ret)
  {
    return (ret >= min) ? ret : min;
  }
  /* If the number is greater than current heap value, increment by 1.5
   * (do not waste much memory). + 1 is so this will work always. */
  else if(num > heap)
  {
    ret = (((heap << 2) - heap) >> 1) + 1;

    // Still check the num is not bigger than heap.
    return (ret >= num) ? ret : num;
  }

  // Otherwise there is no need for change, return the current value.
  return heap;
}

/** Congruence function, returns the congruential of given parameter by the
 * given modulo. Differs from basic modulo on that, the negative values are
 * handled as they were substracted from the ceiling. Please do not specialize
 * for arguments not of unsigned type.
 * @param op The operand.
 * @param mod The modulo. Must be positive.
 */
template <class Type>
static inline Type congr(Type op, Type mod)
{
  return (op < 0) ? (mod - ((-op) % mod)) : (op % mod);
}

// Specialization of former to float.
template <>
static inline float congr<float>(float op, float mod)
{
  return (op < 0.0f) ? (mod - fmodf((-op), mod)) : fmodf(op, mod);
}

// And for double.
template <>
static inline double congr<double>(double op, double mod)
{
  return (op < 0.0f) ? (mod - fmod((-op), mod)) : fmod(op, mod);
}

/** Compare two types, then exchange them if first is greater than second.
 * @param op1 [in] First parameter.
 * @param op2 [in] Second parameter.
 */
template <class Type>
static inline void cmpswap(Type &op1, Type &op2)
{
  if(op1 > op2)
  {
    std::swap(op1, op2);
  }
}

/** Bound argument into place using C++ standard library min and max.
 * @param op Value.
 * @param op_min Minimum value.
 * @param op_max Maximum value.
 * @return The value or min or max if it was out of bounds.
 */
template <class Type>
static const inline Type& bound(const Type &op, const Type &op_min,
    const Type &op_max)
{
  return stdmax(stdmin(op, op_max), op_min);
}

/** Transform an unsigned integer into a floating point value with minimum
 * value of 0 and maximum value of 2 * pi.
 * @param op The value to convert.
 * @return Rotation value.
 */
template <class Type> static inline float uint2pi(Type op);

/* Specializations of the template. */

template <> static inline float uint2pi<uint16_t>(uint16_t op)
{
  return static_cast<float>(op) * (static_cast<float>(M_PI) / 32768.0f);
}

/** Transform a floating point value to integer.
 * @param op Floating point value to convert.
 * @return Orientation integer.
 */
static inline uint16_t pi2int(float op)
{
  return static_cast<int>(op * 32768.0f / static_cast<float>(M_PI));
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif
