#include "libfhi_surface.h"

#include "libfhi_camera.h"
#include "libfhi_font.h"
#include "libfhi_mesh.h"
#include "libfhi_misc.h"
#include "libfhi_postmodel.h"
#include "libfhi_texture.h"

// Need standard C library
#include <stdlib.h>

// Freetype.
#include <ft2build.h>
#include FT_FREETYPE_H
#include FT_GLYPH_H

// libpng.
#ifdef LIBFHI_PNG
#include "libfhi_png.h"
#endif

using namespace libfhi;

//############################################################################
// Static and global variables ###############################################
//############################################################################

Boundary *Surface::bound;
Point Surface::pointdata[LIBFHI_HEAP_POINTDATA_C],
      *Surface::first, *Surface::next;
int Surface::corners, Surface::xmul;
uint8_t Surface::andmask, Surface::ormask;
uint16_t *Surface::zdata;
void *Surface::cdata;

int (*Surface::makecol3)(int, int, int);
int (*Surface::makecol4)(int, int, int, int);

// Not clipped funcpointers.
void (*Surface::nc_setpixel)(int, int, int);
void (*Surface::nc_hline)(int, int, int, int);
void (*Surface::nc_vline)(int, int, int, int);
void (*Surface::nc_line)(int, int, int, int, int);
void (*Surface::nc_rect)(int, int, int, int, int);
void (*Surface::nc_rect_contour)(int, int, int, int, int);
void (*Surface::nc_triangle_flat)(int);
void (*Surface::nc_triangle_gouraud)();
void (*Surface::nc_poly_flat)(int);
void (*Surface::nc_poly_gouraud)();
void (*Surface::nc_zbuf_spot)(int);
void (*Surface::nc_zbuf_line_flat)(int);
void (*Surface::nc_zbuf_line_gouraud)();
void (*Surface::nc_zbuf_triangle_flat)(int);
void (*Surface::nc_zbuf_triangle_gouraud)();
void (*Surface::nc_zbuf_poly_flat)(int);
void (*Surface::nc_zbuf_poly_gouraud)();

// Clipped funcpointers.
void (*Surface::setpixel)(int, int, int);
void (*Surface::hline)(int, int, int, int);
void (*Surface::vline)(int, int, int, int);
void (*Surface::line)(int, int, int, int, int);
void (*Surface::rect)(int, int, int, int, int);
void (*Surface::rect_contour)(int, int, int, int, int);
void (*Surface::poly_flat)(int);
void (*Surface::poly_gouraud)();
void (*Surface::zbuf_line_flat)(int);
void (*Surface::zbuf_line_gouraud)();
void (*Surface::zbuf_poly_flat)(int);
void (*Surface::zbuf_poly_gouraud)();

// Always clipped funcpointers.
void (*Surface::draw_glyph)(int, int, int, Glyph*);

// High-level funcpointers.
int (*Surface::draw_text)(int, int, int, Font*, const char*);
void (*Surface::draw_model)(PostModel*);
void (*Surface::draw_texture)(int, int, int, int, libfhi::Texture*);

//############################################################################
// Konstruktio ###############################################################
//############################################################################

/** Default constructor.
 */
Surface::Surface()
{
  null();
}

/** Default destructor.
 */
Surface::~Surface()
{
  unreserve();
}

/** Empty this surface.
 */
void Surface::null()
{
  cbuf = NULL;
  zbuf = NULL;
  size_bitmap = size_zbuffer = size_pixels = w = h = 0;
}

/** Reserve the zbuffer.
 * @param lf_flags Flags to determine if reservation is needed.
 * @return size of reserved buffer or 0 if not reserved.
 */
int Surface::reserve_zbuffer(uint8_t lf_flags)
{
  size_zbuffer = 0;

  // Jos zbufferin varaus on päällä
  if(lf_flags & FLAG_ZBUFFER)
  {
    zbuf = new uint16_t[size_pixels];
    size_zbuffer = size_pixels * sizeof(uint16_t);
  }

  return size_zbuffer;
}

/** Requires a destructor function distinctively.
 */
void Surface::unreserve()
{
  delete[] zbuf;
  Surface::null();
}

//############################################################################
// Tyhjennys / asetus ########################################################
//############################################################################

/** Empty this surface.
 */
void Surface::clear()
{
  clear(0, Boundary::ZBUFFER_MAX);
}

/** Empty to color.
 * @param color Clear to this.
 */
void Surface::clear(int color)
{
  if(cbuf != NULL)
  {
    memset(cbuf, color, size_bitmap);
  }
}

/** Empty to depth
 * @param depth Clear to this.
 */
void Surface::clear(uint16_t depth)
{
  if(zbuf != NULL)
  {
    memset(zbuf, depth, size_zbuffer);
  }
}

/** Empty to color and depth.
 * @param color Clear to this.
 * @param depth Clear to this.
 */
void Surface::clear(int color, uint16_t depth)
{
  clear(depth);
  clear(color);
}

//############################################################################
// Piirtoalueen asetus #######################################################
//############################################################################

/** Set the two-dimensional view to span the square given by (x1, y1) ->
 * (x2, y2), cut it inside the screen if need be.
 * @param x1 Left x coordinate.
 * @param y1 Upper y coordinate.
 * @param x2 Right x coordinate.
 * @param y2 Lower y coordinate.
 */
void Surface::set_boundary(int x1, int y1, int x2, int y2)
{
  cmpswap(x1, x2);
  cmpswap(y1, y2);

  x1 = libfhi::bound(x1, 0, this->w - 1);
  y1 = libfhi::bound(y1, 0, this->h - 1);
  x2 = libfhi::bound(x2, 0, this->w - 1);
  y2 = libfhi::bound(y2, 0, this->h - 1);

  boundary.set(x1, y1, x2, y2);
}

/** Sets the three-dimensional picture to span from coordinates (x1, y1) to
 * (x2, y2), sets the center of the display to given sport or to the center of
 * the clip rectangle if it would be outside the screen. Sets the perspective
 * in the manner that the greatest distance from the center of the screen is
 * the ratio diven by d[x|y]/dz.
 * @param x1 Left x coordinate.
 * @param y1 Upper y coordinate.
 * @param x2 Right x coordinate.
 * @param y2 Lower y coordinate.
 * @param ratio The view ratio of largest axis.
 * @param near The distance to near clipping plane.
 * @param far The distance to far clipping plane.
 */
void Surface::set_boundary(int x1, int y1, int x2, int y2, float ratio,
    float znear, float zfar)
{
  float big_x, big_y, perspective, xc, yc;
 
  // Aseta normaalit 2d-rajat
  set_boundary(x1, y1, x2, y2);

  // Laske etäisyys reunoihin.
  big_x = (boundary.xmax_f - boundary.xmin_f) / 2.0f;
  big_y = (boundary.ymax_f - boundary.ymin_f) / 2.0f;

  // Aseta keskus ruudun keskelle.
  xc = boundary.xmin_f + big_x;
  yc = boundary.ymin_f + big_y;

  // Laske isompi erotus, ja aseta ratio sen pohjalta.
  perspective = stdmax(big_x, big_y) / ratio;

  // No real need to check on these, since they have default values.
  boundary.set(xc, yc, perspective, znear, zfar);
}

/** Set the drawing area to the whole screen.
 */
void Surface::set_boundary()
{
  boundary.set(0, 0, w - 1, h - 1);
}

/** Set the 3d drawing area to the whole screen.
 * @param ratio Ratio of the larger on-screen axis to z.
 * @param near The distance to near clipping plane.
 * @param far The distance to far clipping plane.
 */
void Surface::set_boundary(float ratio, float znear, float zfar)
{
  set_boundary(0, 0, w - 1, h - 1, ratio, znear, zfar);
}

//############################################################################
// makecol ###################################################################
//############################################################################

/* All colors are r, g, b triplets, all the parameters of which are between
 * [0, 255]. Makecol transforms a triplet into a color readable by the current
 * buffer. Yhe user needs not know what screen mode is used, as long as (s)he
 * is creating all colors using Surface::makecol. Saving the integer values is
 * of course recommended.
 * @param r Red component.
 * @param g Green component.
 * @param b Blue component.
 */
template <class Type> static inline int inline_makecol3(int r, int g, int b);

// Specialization for 8-bit rgb which would probably look kinda bad.
template <>
static inline int inline_makecol3<SURFACE_TYPEDEF_8>(int r, int g, int b)
{
  // RRGGGBBB || 2R 3G 3B
  return (r & 192) | ((g & 224) >> 2) | (b >> 5);
}

// Specialization for 16-bit rgb.
template <>
static inline int inline_makecol3<SURFACE_TYPEDEF_16>(int r, int g, int b)
{
  // RRRRRGGG | GGGBBBBB || 5R 6G 5B
  return ((r & 248) << 8) | ((g & 252) << 3) | (b >> 3);
}

// Specialization for 32-bit (24 actually) RGB.
template <>
static inline int inline_makecol3<SURFACE_TYPEDEF_32>(int r, int g, int b)
{
  return (r << 16) | (g << 8) | b;
}

/** Make a color from 4 components, including alpha. This one here is actually
 * just a dummy, the fourth color is discarded in software.
 * @param r Red component.
 * @param g Green component.
 * @param b Blue component.
 * @param a Alpha component.
 */
template <class Type>
static inline int inline_makecol4(int r, int g, int b, int a);

// Specialization for 8-bit.
template <>
static inline int inline_makecol4<SURFACE_TYPEDEF_8>(int r, int g, int b, int)
{
  return inline_makecol3<SURFACE_TYPEDEF_8>(r, g, b);
}

// Specialization for 16-bit.
template <>
static inline int inline_makecol4<SURFACE_TYPEDEF_16>(int r, int g, int b, int)
{
  return inline_makecol3<SURFACE_TYPEDEF_16>(r, g, b);
}

// Specialization for 32-bit.
template <>
static inline int inline_makecol4<SURFACE_TYPEDEF_32>(int r, int g, int b, int)
{
  return inline_makecol3<SURFACE_TYPEDEF_32>(r, g, b);
}

/** Transforms a V3F into a color.
 * @param op Vector in floating point format.
 * @return Color as an integer.
 */
static inline int makecol_c3f(float *op)
{
  // Actually important to round correctly here.
  return Surface::makecol3(lrintf(op[0]), lrintf(op[1]), lrintf(op[2]));
}

//############################################################################
// zbuf_setpixel #############################################################
//############################################################################

/*
 * Asettavat zbufferin arvon jonkin arvon mukaan. Eri variaatioita.
 */

template <class Type> static inline void zbuf_setpixel(int offset, int cval,
    uint16_t zval, Type *bitmap, uint16_t *depthmap)
{
  depthmap += offset;
  if(*depthmap > zval)
  {
    *(bitmap + offset) = cval;
    *depthmap = zval;
  }
}

template <class Type> static inline void zbuf_setpixel(int cval, uint16_t zval,
    Type *bitmap, uint16_t *depthmap)
{
  if(*depthmap > zval)
  {
    *bitmap = cval;
    *depthmap = zval;
  }
}

template <class Type> static inline void zbuf_setpixel(int r, int g, int b,
    uint16_t zval, Type *bitmap, uint16_t *depthmap)
{
  if(*depthmap > zval)
  {
    *bitmap = inline_makecol3<Type>(r, g, b);
    *depthmap = zval;
  }
}

//############################################################################
// setpixel ##################################################################
//############################################################################

template <class Type> void Surface::nc_setpixel_template(int x, int y,
    int col)
{
  static_cast<Type*>(cdata)[y * xmul + x] = col;
}

//############################################################################
// hline #####################################################################
//############################################################################

template <class Type> void Surface::nc_hline_template(int x1, int x2, int y,
    int col)
{
  Type *bitmap = static_cast<Type*>(cdata);

  cmpswap(x1, x2);
  bitmap += y * xmul + x1;
  x2 = x2 - x1 + 1;

  do {
    (*bitmap) = col;
    ++bitmap;
  } while(--x2);
}

//############################################################################
// vline #####################################################################
//############################################################################

template <class Type> void Surface::nc_vline_template(int y1, int y2, int x,
    int col)
{
  Type *bitmap;

  cmpswap(y1, y2);
  bitmap = static_cast<Type*>(cdata) + y1 * xmul + x;
  y2 = y2 - y1 + 1;

  do {
    (*bitmap) = col;
    bitmap += xmul;
  } while(--y2);
}
	
//############################################################################
// line ######################################################################
//############################################################################

template <class Type> void Surface::nc_line_template(int x1, int y1, int x2,
    int y2, int col)
{
  int disx, disy, idx1, idx2, loop;
  Fixed div, step1, step2;
  Type *bitmap = static_cast<Type*>(cdata);

  if(abs(x1-x2) > abs(y1-y2))
  {
    if(x1 > x2)
    {
      std::swap(x1, x2);
      std::swap(y1, y2);
    }

    disx = x2 - x1;
    disy = y2 - y1;
    div = Fixed::from(disy);
    div /= disx;
    disx += 1;
    step1.set_roundable(y1);
    step2.set_roundable(y2);

    for(loop = disx / 2; loop; loop--)
    {
      idx1 = static_cast<int>(step1) * xmul + x1;
      idx2 = static_cast<int>(step2) * xmul + x2;
      bitmap[idx1] = col;
      bitmap[idx2] = col;
      step1 += div;
      step2 -= div;
      x1++;
      x2--;
    }

    if(disx & 1)
    {
      idx1 = static_cast<int>(step1) * xmul + x1;
      bitmap[idx1] = col;
    }
  }
  
  else
  {
    if(y1 > y2)
    {
      std::swap(x1, x2);
      std::swap(y1, y2);
    }
    
    disy = y2 - y1;
    disx = x2 - x1;
    div.set_val(0);
    if(disy != 0)
    {
      div = Fixed::from(disx);
      div /= disy;
    }
    disy += 1;
    y1 *= xmul;
    y2 *= xmul;
    step1 = Fixed::from(x1);
    step2 = Fixed::from(x2);
    
    for(loop = disy / 2; loop; loop--)
    {
      idx1 = y1 + static_cast<int>(step1);
      idx2 = y2 + static_cast<int>(step2);
      bitmap[idx1] = col;
      bitmap[idx2] = col;
      step1 += div;
      step2 -= div;
      y1 += xmul;
      y2 -= xmul;
    }

    if(disy & 1)
    {
      idx1 = y1 + static_cast<int>(step1);
      bitmap[idx1] = col;
    }
  }
}

//############################################################################
// rect ######################################################################
//############################################################################

template <class Type> void Surface::nc_rect_template(int x1, int y1, int x2,
    int y2, int col)
{
  Type *bitmap;

  cmpswap(x1, x2);
  cmpswap(y1, y2);
  y2 = y2 - y1 + 1;
  x2 = x2 - x1 + 1;
  bitmap = static_cast<Type*>(cdata) + y1 * xmul + x1;

  do {
    Type *dst = bitmap;
    x1 = x2; // Reuse is obfuscation
    do {
      (*dst) = col;
      ++dst;
    } while(--x1);
    bitmap += xmul;
  } while(--y2);
}

//############################################################################
// rect_contour ##############################################################
//############################################################################

template <class Type> void Surface::nc_rect_contour_template(int x1, int y1,
    int x2, int y2, int col)
{
  nc_hline(x1, x2, y1, col);
  nc_hline(x1, x2, y2, col);
  nc_vline(y1, y2, x1, col);
  nc_vline(y1, y2, x2, col);
}

//############################################################################
// Filleriluokka #############################################################
//############################################################################

/** This integer is used internally for drawing flat-shaded surfaces. */
static int flat_shading_color;

/*
 * Fillerit tarvitsevat apuluokan jossa säilyttää interpolaation tilaa. Tässä
 * on perusluokka, jota periytetään aina eteenpäin.
 *
 * Jokaisesta luokasta löytyy step, loop, setpixel ja hline-funktiot joilla
 * voidaan kirjoittaa fillerit kätevästi templatoimalla (miksi helvetissä en
 * keksinyt tätä aikaisemmin, huoh?).
 */

/** X interpolator.
 */
template <class Type> class FillerDummyX
{
  friend class Surface;
  
  protected:
    Fixed sx, dx;

  public:
    /** End initialization.
     */
    inline void init_end(int diff)
    {
      dx /= diff;
    }

    /** Initialize for x interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_x(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      sx = pos1->xi;
      int diff = static_cast<int>(pos2->yi) - static_cast<int>(pos1->yi);
      if(diff)
      {
	dx = pos2->xi - pos1->xi;
	init_end(diff);
      }
      else
	dx.set_null();
    }

    /** Initialize for y interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_y(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      sx = pos1->yi;
      int diff = static_cast<int>(pos2->xi) - static_cast<int>(pos1->xi);
      if(diff)
      {
	dx = pos2->yi - pos1->yi;
	init_end(diff);
      }
      else
	dx.set_null();
    }

    /** Loop.
     */
    inline void loop()
    {
      sx += dx;
    }

    /** Setpixel.
     * @param idx The index to displace the pointers with.
     * @param bitmap Target bitmap.
     */
    inline void setpixel(int idx, Type *bitmap)
    {
      bitmap[idx] = flat_shading_color;
    }

    /** Horizontal line.
     * @param op The other fillerdummy to draw the line to.
     * @param bitmap Target bitmap.
     */
    void hline(FillerDummyX *op, Type *bitmap)
    {
      FillerDummyX *op1, *op2;
      int col = flat_shading_color;

      if(this->sx <= op->sx)
      {
	op1 = this;
	op2 = op;
      }
      else
      {
	op1 = op;
	op2 = this;
      }

      int x1 = static_cast<int>(op1->sx),
	  x2 = static_cast<int>(op2->sx) - x1;

      bitmap += x1;
      x2++;

      do {
	*bitmap = col;
	++bitmap;
      } while(--x2);
    }
};

/** XZ interpolator.
 */
template <class Type> class FillerDummyXZ
{
  friend class Surface;
  
  protected:
    Fixed sx, sz, dx, dz;

  public:
    /** End initialization.
     */
    inline void init_end(int diff)
    {
      dx /= diff;
      dz /= diff;
    }

    /** Initialize for x interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_x(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      sx = pos1->xi;
      sz = pos1->zi;
      int diff = static_cast<int>(pos2->yi) - static_cast<int>(pos1->yi);
      if(diff)
      {
	dx = pos2->xi - pos1->xi;
	dz = pos2->zi - pos1->zi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dz.set_null();
      }
    }

    /** Initialize for y interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_y(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      sx = pos1->yi;
      sz = pos1->zi;
      int diff = static_cast<int>(pos2->xi) - static_cast<int>(pos1->xi);
      if(diff)
      {
	dx = pos2->yi - pos1->yi;
	dz = pos2->zi - pos1->zi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dz.set_null();
      }
    }

    /** Loop.
     */
    inline void loop()
    {
      sx += dx;
      sz += dz;
    }

    /** Setpixel without index.
     * @param depth Drawing depth.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    inline void setpixel(uint16_t depth, Type *bitmap, uint16_t *depthmap)
    {
      if(*depthmap > depth)
      {
	*depthmap = depth;
	*bitmap = flat_shading_color;
      }
    }

    /** Setpixel.
     * @param idx The index to displace the pointers with.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    inline void setpixel(int idx, Type *bitmap, uint16_t *depthmap)
    {
      uint16_t depth = static_cast<uint16_t>(this->sz);

      depthmap += idx;

      if(*depthmap > depth)
      {
	*depthmap = depth;
	bitmap[idx] = flat_shading_color;
      }
    }

    /** Horizontal line.
     * @param op The other fillerdummy to draw the line to.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    void hline(FillerDummyXZ *op, Type *bitmap, uint16_t *depthmap)
    {
      FillerDummyXZ *op1, *op2;

      if(this->sx <= op->sx)
      {
	op1 = this;
	op2 = op;
      }
      else
      {
	op1 = op;
	op2 = this;
      }

      int x1 = static_cast<int>(op1->sx),
	  x2 = static_cast<int>(op2->sx) - x1;
      Fixed z1 = op1->sz,
	    z2 = op2->sz - z1;
      
      if(x2)
      {
	z2 /= x2;
      }

      bitmap += x1;
      depthmap += x1;
      ++x2;

      do {
	this->setpixel(static_cast<uint16_t>(z1), bitmap, depthmap);
	z1 += z2;
	++bitmap;
	++depthmap;
      } while(--x2);
    }
};

/** XC interpolator.
 */
template <class Type> class FillerDummyXC
{
  friend class Surface;
  
  protected:
    Fixed dx, dr, dg, db, sx, sr, sg, sb;

  public:
    /** End initialization.
     */
    inline void init_end(int diff)
    {
      dx /= diff;
      dr /= diff;
      dg /= diff;
      db /= diff;
    }

    /** Initialize for x interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_x(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      const Color4 *col1 = &(p1->get_col()), *col2 = &(p2->get_col());
      sx = pos1->xi;
      sr = col1->ri;
      sg = col1->gi;
      sb = col1->bi;
      int diff = static_cast<int>(pos2->yi) - static_cast<int>(pos1->yi);
      if(diff)
      {
	dx = pos2->xi - pos1->xi;
	dr = col2->ri - col1->ri;
	dg = col2->gi - col1->gi;
	db = col2->bi - col1->bi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dr.set_null();
	dg.set_null();
	db.set_null();
      }
    }

    /** Initialize for y interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_y(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      const Color4 *col1 = &(p1->get_col()), *col2 = &(p2->get_col());
      sx = pos1->yi;
      sr = col1->ri;
      sg = col1->gi;
      sb = col1->bi;
      int diff = static_cast<int>(pos2->xi) - static_cast<int>(pos1->xi);
      if(diff)
      {
	dx = pos2->yi - pos1->yi;
	dr = col2->ri - col1->ri;
	dg = col2->gi - col1->gi;
	db = col2->bi - col1->bi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dr.set_null();
	dg.set_null();
	db.set_null();
      }
    }

    /** Loop.
     */
    inline void loop()
    {
      sx += dx;
      sr += dr;
      sg += dg;
      sb += db;
    }

    /** Setpixel.
     * @param idx The index to displace the pointers with.
     * @param bitmap Target bitmap.
     */
    inline void setpixel(int idx, Type *bitmap)
    {
      bitmap[idx] = inline_makecol3<Type>(static_cast<int>(this->sr),
	  static_cast<int>(this->sg), static_cast<int>(this->sb));
    }

    /** Horizontal line.
     * @param op The other fillerdummy to draw the line to.
     * @param bitmap Target bitmap.
     */
    void hline(FillerDummyXC *op, Type *bitmap)
    {
      FillerDummyXC *op1, *op2;

      if(this->sx <= op->sx)
      {
	op1 = this;
	op2 = op;
      }
      else
      {
	op1 = op;
	op2 = this;
      }

      int x1 = static_cast<int>(op1->sx),
	  x2 = static_cast<int>(op2->sx) - x1;
      Fixed r1 = op1->sr,
	    g1 = op1->sg,
	    b1 = op1->sb,
	    r2 = op2->sr - r1,
	    g2 = op2->sg - g1,
	    b2 = op2->sb - b1;

      if(x2)
      {
	r2 /= x2;
	g2 /= x2;
	b2 /= x2;
      }
      
      bitmap += x1;
      ++x2;

      do {
	(*bitmap) = inline_makecol3<Type>(static_cast<int>(r1),
	    static_cast<int>(g1), static_cast<int>(b1));
	r1 += r2;
	g1 += g2;
	b1 += b2;
	++bitmap;
      } while(--x2);
    }
};

/** XZC interpolator
 */
template <class Type> class FillerDummyXZC
{
  friend class Surface;
  
  protected:
    Fixed sx, sz, sr, sg, sb, dx, dz, dr, dg, db;

  public:
    /** End initialization.
     */
    inline void init_end(int diff)
    {
      dx /= diff;
      dz /= diff;
      dr /= diff;
      dg /= diff;
      db /= diff;
    }

    /** Initialize for x interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_x(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      const Color4 *col1 = &(p1->get_col()), *col2 = &(p2->get_col());
      sx = pos1->xi;
      sz = pos1->zi;
      sr = col1->ri;
      sg = col1->gi;
      sb = col1->bi;
      int diff = static_cast<int>(pos2->yi) - static_cast<int>(pos1->yi);
      if(diff)
      {
	dx = pos2->xi - pos1->xi;
	dz = pos2->zi - pos1->zi;
	dr = col2->ri - col1->ri;
	dg = col2->gi - col1->gi;
	db = col2->bi - col1->bi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dz.set_null();
	dr.set_null();
	dg.set_null();
	db.set_null();
      }
    }

    /** Initialize for y interpolation.
     * @param p1 Starting point.
     * @param p2 Ending point.
     */
    inline void init_y(Point *p1, Point *p2)
    {
      const Vector3 *pos1 = &(p1->get_pos()), *pos2 = &(p2->get_pos());
      const Color4 *col1 = &(p1->get_col()), *col2 = &(p2->get_col());
      sx = pos1->yi;
      sz = pos1->zi;
      sr = col1->ri;
      sg = col1->gi;
      sb = col1->bi;
      int diff = static_cast<int>(pos2->xi) - static_cast<int>(pos1->xi);
      if(diff)
      {
	dx = pos2->yi - pos1->yi;
	dz = pos2->zi - pos1->zi;
	dr = col2->ri - col1->ri;
	dg = col2->gi - col1->gi;
	db = col2->bi - col1->bi;
	init_end(diff);
      }
      else
      {
	dx.set_null();
	dz.set_null();
	dr.set_null();
	dg.set_null();
	db.set_null();
      }
    }

    /** Loop.
     */
    inline void loop()
    {
      sx += dx;
      sz += dz;
      sr += dr;
      sg += dg;
      sb += db;
    }

    /** Setpixel without index but with color.
     * @param col Drawing color.
     * @param depth Drawing depth.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    inline void setpixel(int col, uint16_t depth, Type *bitmap,
	uint16_t *depthmap)
    {
      if(*depthmap > depth)
      {
	*depthmap = depth;
	*bitmap = col;
      }
    }

    /** Setpixel.
     * @param idx The index to displace the pointers with.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    inline void setpixel(int idx, Type *bitmap, uint16_t *depthmap)
    {
      uint16_t depth = static_cast<uint16_t>(this->sz);

      depthmap += idx;

      if(*depthmap > depth)
      {
	*depthmap = depth;
	*(bitmap + idx) = inline_makecol3<Type>(static_cast<int>(this->sr),
	    static_cast<int>(this->sg), static_cast<int>(this->sb));
      }
    }
    
    /** Horizontal line.
     * @param op The other fillerdummy to draw the line to.
     * @param bitmap Target bitmap.
     * @param depthmap Target depthmap.
     */
    void hline(FillerDummyXZC *op, Type *bitmap, uint16_t *depthmap)
    {
      FillerDummyXZC *op1, *op2;

      if(this->sx <= op->sx)
      {
	op1 = this;
	op2 = op;
      }
      else
      {
	op1 = op;
	op2 = this;
      }

      int x1 = static_cast<int>(op1->sx),
	  x2 = static_cast<int>(op2->sx) - x1;
      Fixed z1 = op1->sz,
	    r1 = op1->sr,
	    g1 = op1->sg,
	    b1 = op1->sb,
	    z2 = op2->sz - z1,
	    r2 = op2->sr - r1,
	    g2 = op2->sg - g1,
	    b2 = op2->sb - b1;

      if(x2)
      {
	z2 /= x2;
	r2 /= x2;
	g2 /= x2;
	b2 /= x2;
      }

      bitmap += x1;
      depthmap += x1;
      ++x2;

      do {
	this->setpixel(inline_makecol3<Type>(static_cast<int>(r1),
  	      static_cast<int>(g1), static_cast<int>(b1)),
	    static_cast<uint16_t>(z1), bitmap, depthmap);
	z1 += z2;
	r1 += r2;
	g1 += g2;
	b1 += b2;
	++bitmap;
	++depthmap;
      } while(--x2);
    }
};

//############################################################################
// triangle ##################################################################
//############################################################################

/** Template for triangle filler.
 */
template <class Type1, class Type2>
inline void Surface::nc_triangle_template()
{
  Type2 f1, f2;
  Point *p1, *p2, *p3;
  Type1 *bitmap = static_cast<Type1*>(cdata);
  int y1, y2, y3;

  p1 = Surface::first;
  p2 = p1->next;
  p3 = p2->next;

  if(p2->pos.yi < p1->pos.yi)
  {
    std::swap(p2, p1);
  }
  if(p3->pos.yi < p1->pos.yi)
  {
    std::swap(p3, p1);
  }
  if(p3->pos.yi < p2->pos.yi)
  {
    std::swap(p3, p2);
  }

  y1 = static_cast<int>(p1->pos.yi);
  y2 = static_cast<int>(p2->pos.yi);
  y3 = static_cast<int>(p3->pos.yi);
  
  f1.init_x(p1, p2);
  f2.init_x(p1, p3);

  bitmap += y1 * xmul;
  for(int i = y2 - y1; i != 0; i--, bitmap += xmul)
  {
    f1.hline(&f2, bitmap);
    f1.loop();
    f2.loop();
  }
  f1.init_x(p2, p3);
  for(int i = y3 - y2; i != 0; i--, bitmap += xmul)
  {
    f1.hline(&f2, bitmap);
    f1.loop();
    f2.loop();
  }
  f1.hline(&f2, bitmap);
}

/** Template for flat triangle filler.
 * @param col Color to draw with.
 */
template <class Type> void Surface::nc_triangle_flat_template(int col)
{
  flat_shading_color = col;
  nc_triangle_template< Type, FillerDummyX<Type> >();
}

/** Template for gouraud triangle filler.
 */
template <class Type> void Surface::nc_triangle_gouraud_template()
{
  nc_triangle_template< Type, FillerDummyXC<Type> >();
}

//############################################################################
// poly_convex_flat ##########################################################
//############################################################################

/** Template for polygon filler.
 */
template <class Type1, class Type2>
inline void Surface::nc_poly_template()
{
  Type2 f1, f2;
  Point *prc = Surface::first, *plc = prc, *prn, *pln;
  Type1 *bitmap = static_cast<Type1*>(cdata);
  Fixed smallest, biggest;
  int curr, end, yr, yl;

  // Search for the smallest and largest value (and startpoint)
  smallest = biggest = prc->pos.yi;
  prc = prc->next;
  do {
    Fixed cmp = prc->pos.yi;
    if(cmp < smallest)
    {
      plc = prc;
      smallest = cmp;
    }
    else if(cmp > biggest)
    {
      biggest = cmp;
    }
    prc = prc->next;
  } while(prc != Surface::first);
  prc = plc;
  curr = static_cast<int>(smallest);
  end = static_cast<int>(biggest);

  // Initialize interpolations, right goes forward, left backward
  prn = prc->next;
  pln = plc->prev;
  f1.init_x(prc, prn);
  f2.init_x(plc, pln);
  yr = static_cast<int>(prn->pos.yi);
  yl = static_cast<int>(pln->pos.yi);

  bitmap += curr * xmul;
  while(curr < end)
  {
    while(yr == curr)
    {
      prc = prn;
      prn = prn->next;
      f1.init_x(prc, prn);
      yr = prn->pos.yi;
    }
    while(yl == curr)
    {
      plc = pln;
      pln = pln->prev;
      f2.init_x(plc, pln);
      yl = pln->pos.yi;
    }

    for(int stop = stdmin(yr, yl);
	(curr < stop);
	++curr, bitmap += xmul)
    {
      f1.hline(&f2, bitmap);
      f1.loop();
      f2.loop();
    }
  }
  f1.hline(&f2, bitmap);
}

/** Draw one flat-shaded polygon from the state machine.
 * @param col [in] Color to draw with.
 */
template <class Type> void Surface::nc_poly_flat_template(int col)
{
  flat_shading_color = col;
  nc_poly_template< Type, FillerDummyX<Type> >();
}

/** Draw one gouraud-shaded polygon from the state machine.
 * @param col [in] Color to draw with.
 */
template <class Type> void Surface::nc_poly_gouraud_template()
{
  nc_poly_template< Type, FillerDummyXC<Type> >();
}

//############################################################################
// zbuf_spot #################################################################
//############################################################################

/** Template for the z-buffered spot.
 * @param col [in] Color to draw with.
 */
template <class Type> void Surface::nc_zbuf_spot_template(int col)
{
  int x, y, index;
  Type *bitmap = static_cast<Type*>(cdata);
  Vector3 *pos = &(pointdata[0].get_pos());
  
  x = pos->xi;
  y = pos->yi;
  index = y * xmul + x;

  zbuf_setpixel(index, col, static_cast<uint16_t>(pos->zi), bitmap, zdata);
}

//############################################################################
// zbuf_line #################################################################
//############################################################################

/** Template for z-buffered line filler.
 * FIXME: Too big, inlining does not work.
 */
template <class Type1, class Type2>
void Surface::nc_zbuf_line_template()
{
  Type2 f;
  Point *p1 = &pointdata[0], *p2 = &pointdata[1];
  Fixed dx, dy;
  int step, cnt;
  Type1 *bitmap = static_cast<Type1*>(Surface::cdata);
  uint16_t *depthmap = Surface::zdata;

  dx = (p1->pos.xi > p2->pos.xi) ? (p1->pos.xi - p2->pos.xi) :
    (p2->pos.xi - p1->pos.xi);
  dy = (p1->pos.yi > p2->pos.yi) ? (p1->pos.yi - p2->pos.yi) :
    (p2->pos.yi - p1->pos.yi);

  if(dx > dy)
  {
    if(p1->pos.xi <= p2->pos.xi)
    {
      f.init_y(p1, p2);
      step = static_cast<int>(p1->pos.xi);
      cnt = static_cast<int>(p2->pos.xi) - step + 1;
    }
    else
    {
      f.init_y(p2, p1);
      step = static_cast<int>(p2->pos.xi);
      cnt = static_cast<int>(p1->pos.xi) - step + 1;
    }
    bitmap += step;
    depthmap += step;
    do {
      f.setpixel(static_cast<int>(f.sx) * xmul, bitmap, depthmap);
      f.loop();
      bitmap++;
      depthmap++;
    } while(--cnt);
  }
  else
  {
    if(p1->pos.yi <= p2->pos.yi)
    {
      f.init_x(p1, p2);
      step = static_cast<int>(p1->pos.yi);
      cnt = static_cast<int>(p2->pos.yi) - step + 1;
    }
    else
    {
      f.init_x(p2, p1);
      step = static_cast<int>(p2->pos.yi);
      cnt = static_cast<int>(p1->pos.yi) - step + 1;
    }
    step *= Surface::xmul;
    bitmap += step;
    depthmap += step;
    do {
      f.setpixel(static_cast<int>(f.sx), bitmap, depthmap);
      f.loop();
      bitmap += xmul;
      depthmap += xmul;
    } while(--cnt);
  }
}

/** Draw one flat z-buffered line.
 * @param col Color to draw with.
 */
template <class Type> void Surface::nc_zbuf_line_flat_template(int col)
{
  flat_shading_color = col;
  nc_zbuf_line_template< Type, FillerDummyXZ<Type> >();
}

/** Draw one gouraud-shaded z-buffered line.
 */
template <class Type> void Surface::nc_zbuf_line_gouraud_template()
{
  nc_zbuf_line_template< Type, FillerDummyXZC<Type> >();
}

//############################################################################
// zbuf_triangle #############################################################
//############################################################################

/** Template for z-buffered triangle filler.
 * NOTE: This is actually pretty much copied from the flat filler with the
 * neccessary z-buffer data added in later on. When modifying this, actually
 * modify the flat version, then port changes back here.
 */
template <class Type1, class Type2>
inline void Surface::nc_zbuf_triangle_template()
{
  Type2 f1, f2;
  Point *p1, *p2, *p3;
  Type1 *bitmap = static_cast<Type1*>(cdata);
  uint16_t *depthmap = zdata;
  int y1, y2, y3;

  p1 = Surface::first;
  p2 = p1->next;
  p3 = p2->next;

  if(p2->pos.yi < p1->pos.yi)
  {
    std::swap(p2, p1);
  }
  if(p3->pos.yi < p1->pos.yi)
  {
    std::swap(p3, p1);
  }
  if(p3->pos.yi < p2->pos.yi)
  {
    std::swap(p3, p2);
  }

  y1 = static_cast<int>(p1->pos.yi);
  y2 = static_cast<int>(p2->pos.yi);
  y3 = static_cast<int>(p3->pos.yi);
  
  f1.init_x(p1, p2);
  f2.init_x(p1, p3);

  bitmap += y1 * xmul;
  depthmap += y1 * xmul;
  for(int i = y2 - y1; i != 0; i--, bitmap += xmul, depthmap += xmul)
  {
    f1.hline(&f2, bitmap, depthmap);
    f1.loop();
    f2.loop();
  }
  f1.init_x(p2, p3);
  for(int i = y3 - y2; i != 0; i--, bitmap += xmul, depthmap += xmul)
  {
    f1.hline(&f2, bitmap, depthmap);
    f1.loop();
    f2.loop();
  }
  f1.hline(&f2, bitmap, depthmap);
}

/** Template for flat z-buffered triangle filler.
 * @param col Color to draw with.
 */
template <class Type> void Surface::nc_zbuf_triangle_flat_template(int col)
{
  flat_shading_color = col;
  nc_zbuf_triangle_template< Type, FillerDummyXZ<Type> >();
}

/** Template for gouraud z-buffered triangle filler.
 */
template <class Type> void Surface::nc_zbuf_triangle_gouraud_template()
{
  nc_zbuf_triangle_template< Type, FillerDummyXZC<Type> >();
}

//############################################################################
// zbuf_poly #################################################################
//############################################################################

/** Z-buffered polygon filler template. Works on the state machine.
 * NOTE: This is actually pretty much copied from the flat filler with the
 * neccessary z-buffer data added in later on. When modifying this, actually
 * modify the flat version, then port changes back here.
 */
template <class Type1, class Type2>
inline void Surface::nc_zbuf_poly_template()
{
  Type2 f1, f2;
  Point *prc = Surface::first, *plc = prc, *prn, *pln;
  Type1 *bitmap = static_cast<Type1*>(cdata);
  uint16_t *depthmap = zdata;
  Fixed smallest, biggest;
  int curr, end, yl, yr;

  // Search for the smallest and largest value (and startpoint)
  smallest = biggest = prc->pos.yi;
  prc = prc->next;
  do {
    Fixed cmp = prc->pos.yi;
    if(cmp < smallest)
    {
      plc = prc;
      smallest = cmp;
    }
    else if(cmp > biggest)
    {
      biggest = cmp;
    }
    prc = prc->next;
  } while(prc != Surface::first);
  prc = plc;
  curr = static_cast<int>(smallest);
  end = static_cast<int>(biggest);

  // Right goes next, left prev
  prn = prc->next;
  pln = plc->prev;
  f1.init_x(prc, prn);
  f2.init_x(plc, pln);

  yr = prn->pos.yi;
  yl = pln->pos.yi;
  bitmap += curr * xmul;
  depthmap += curr * xmul;

  while(curr < end)
  {
    while(yr <= curr)
    {
      prc = prn;
      prn = prn->next;
      f1.init_x(prc, prn);
      yr = prn->pos.yi;
    }
    while(yl <= curr)
    {
      plc = pln;
      pln = pln->prev;
      f2.init_x(plc, pln);
      yl = pln->pos.yi;
    }

    for(int stop = stdmin(yr, yl);
	(curr < stop);
	++curr, bitmap += xmul, depthmap += xmul)
    {
      f1.hline(&f2, bitmap, depthmap);
      f1.loop();
      f2.loop();
    }
  }
  f1.hline(&f2, bitmap, depthmap);
}

/** Draw one flat-shaded z-buffered polygon.
 * @param col [in] Color to draw with.
 */
template <class Type> void Surface::nc_zbuf_poly_flat_template(int col)
{
  flat_shading_color = col;
  nc_zbuf_poly_template< Type, FillerDummyXZ<Type> >();
}

/** Draw one gouraud-shaded z-buffered polygon.
 * @param col [in] Color to draw with.
 */
template <class Type> void Surface::nc_zbuf_poly_gouraud_template()
{
  nc_zbuf_poly_template< Type, FillerDummyXZC<Type> >();
}

//############################################################################
// textout ###################################################################
//############################################################################

/** Template to draw glyph on screen with given tolerance.
 * @param x [in] Left coordinate of glyph.
 * @param y [in] Bottom coordinate of glyph.
 * @param col [in] Color to draw with.
 * @param glp [in] Void pointer to Glyph.
 */
template <class Type> void Surface::draw_glyph_template(int x, int y, int col,
    Glyph *glyph)
{
  uint8_t *srcdata = glyph->get_data();
  int w = glyph->get_w(), h = glyph->get_h(), x1, x2, y1, y2;
  Type *bitmap = static_cast<Type*>(cdata);

  // Safety check.
  if((w <= 0) || (h <= 0))
  {
    return;
  }

  x += glyph->get_left();
  y -= glyph->get_top();

  // Perform xmin clipping.
  x1 = (x < bound->xmin_i) ? (bound->xmin_i - x) : 0;
  if(x1 >= w)
  {
    return;
  }

  // Perform xmax clipping.
  if(x > bound->xmax_i)
  {
    return;
  }
  x2 = ((x + w > bound->xmax_i) ? (bound->xmax_i - x + 1) : w);

  // Perform ymin clipping.
  y1 = (y < bound->ymin_i) ? (bound->ymin_i - y) : 0;
  if(y1 >= h)
  {
    return;
  }

  // Perform ymax clipping.
  if(y > bound->ymax_i)
  {
    return;
  }
  y2 = ((y + h > bound->ymax_i) ? (bound->ymax_i - y + 1) : h) - y1;

  // Increment bitmap
  bitmap += x + (y + y1) * xmul;

  // Get real width in bytes (pitch)
  w = glyph->get_w();
  srcdata += y1 * w;

  do {
    for(int i = x1; i < x2; ++i)
    {
      if(srcdata[i])
      {
	bitmap[i] = col;
      }
    }

    bitmap += xmul;
    srcdata += w;
  } while(--y2);
}

//############################################################################
// setpixel ##################################################################
//############################################################################

/** Clipped setpixel
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param col Color.
 */
void Surface::c_setpixel(int x, int y, int col)
{
  if((bound->xmin_i <= x) && (x <= bound->xmax_i) &&
      (bound->ymin_i <= y) && (y <= bound->ymax_i))
  {
    nc_setpixel(x, y, col);
  }
}

//############################################################################
// hline #####################################################################
//############################################################################

/** Clipped horizontal line.
 * @param x1 First x coordinate.
 * @param x2 Second x coordinate.
 * @param y Y coordinate.
 * @param col Color.
 */
void Surface::c_hline(int x1, int x2, int y, int col)
{
  if((y > bound->ymax_i) || (y < bound->ymin_i))
  {
    return;
  }

  cmpswap(x1, x2);

  if(x2 < bound->xmin_i)
  {
    return;
  }
  if(x1 > bound->xmax_i)
  {
    return;
  }

  nc_hline(
      stdmax(x1, bound->xmin_i),
      stdmin(x2, bound->xmax_i),
      y,
      col);
}

//############################################################################
// vline #####################################################################
//############################################################################

void Surface::c_vline(int y1, int y2, int x, int col)
{
  if((x > bound->xmax_i) || (x < bound->xmin_i))
  {
    return;
  }

  cmpswap(y1, y2);

  if(y2 < bound->ymin_i)
  {
    return;
  }
  if(y1 > bound->ymax_i)
  {
    return;
  }

  nc_vline(
      stdmax(y1, bound->ymin_i),
      stdmin(y2, bound->ymax_i),
      x,
      col);
}

//############################################################################
// line ######################################################################
//############################################################################

void Surface::c_line(int op_x1, int op_y1, int op_x2, int op_y2, int col)
{
  float x1, x2, y1, y2, dist;

  if(op_x1 > op_x2)
  {
    x1 = static_cast<float>(op_x2);
    x2 = static_cast<float>(op_x1);
    y1 = static_cast<float>(op_y2);
    y2 = static_cast<float>(op_y1);
  }
  else
  {
    x1 = static_cast<float>(op_x1);
    x2 = static_cast<float>(op_x2);
    y1 = static_cast<float>(op_y1);
    y2 = static_cast<float>(op_y2);
  }

  if(x1 > bound->xmax_f)
  {
    return;
  }
  if(x2 < bound->xmin_f)
  {
    return;
  }

  if(x1 < bound->xmin_f)
  {
    dist = (bound->xmin_f - x1) / (x2 - x1);
    y1 = dist * (y2 - y1) + y1;
    x1 = bound->xmin_f;
  }
  if(x2 > bound->xmax_f)
  {
    dist = (bound->xmax_f - x1) / (x2 - x1);
    y2 = dist * (y2 - y1) + y1;
    x2 = bound->xmax_f;
  }

  if(y1 > y2)
  {
    std::swap(x1, x2);
    std::swap(y1, y2);
  }

  if(y1 > bound->ymax_f)
  {
    return;
  }
  if(y2 < bound->ymin_f)
  {
    return;
  }

  if(y1 < bound->ymin_f)
  {
    dist = (bound->ymin_f - y1) / (y2 - y1);
    x1 = dist * (x2 - x1) + x1;
    y1 = bound->ymin_f;
  }
  if(y2 > bound->ymax_f)
  {
    dist = (bound->ymax_f - y1) / (y2 - y1);
    x2 = dist * (x2 - x1) + x1;
    y2 = bound->ymax_f;
  }

  nc_line(static_cast<int>(x1),
      static_cast<int>(y1),
      static_cast<int>(x2),
      static_cast<int>(y2),
      col);
}

//############################################################################
// rect ######################################################################
//############################################################################

void Surface::c_rect(int x1, int y1, int x2, int y2, int col)
{
  cmpswap(x1, x2);
  cmpswap(y1, y2);

  if(x1 > bound->xmax_i)
  {
    return;
  }
  if(x2 < bound->xmin_i)
  {
    return;
  }
  if(x1 < bound->xmin_i)
  {
    x1 = bound->xmin_i;
  }
  if(x2 > bound->xmax_i)
  {
    x2 = bound->xmax_i;
  }

  if(y1 > bound->ymax_i)
  {
    return;
  }
  if(y2 < bound->ymin_i)
  {
    return;
  }
  if(y1 < bound->ymin_i)
  {
    y1 = bound->ymin_i;
  }
  if(y2 > bound->ymax_i)
  {
    y2 = bound->ymax_i;
  }

  nc_rect(
      stdmax(x1, bound->xmin_i),
      stdmax(y1, bound->ymin_i),
      stdmin(x2, bound->xmax_i),
      stdmin(y2, bound->ymax_i),
      col);
}

//############################################################################
// rect_contour ##############################################################
//############################################################################

/** Draw the outlines of a rectangle (clipped).
 * @param x1 First x coordinate.
 * @param y1 First y coordinate.
 * @param x2 Second x coordinate.
 * @param y2 Second y coordinate.
 * @param col Contour color.
 */
void Surface::c_rect_contour(int x1, int y1, int x2, int y2, int col)
{
  hline(x1, x2, y1, col);
  hline(x1, x2, y2, col);
  vline(y1, y2, x1, col);
  vline(y1, y2, x2, col);
}

//############################################################################
// Point (non-clipped) #######################################################
//############################################################################

/** Begin non-clipped form.
 */
void Surface::nc_point_begin()
{
  corners = 0;
  next = first = pointdata;
}

/** Begin clipped form.
 */
void Surface::point_begin()
{
  andmask = 0xFF;
  ormask = 0x00;
  nc_point_begin();
}

/** Increment the point data one step ensuring the consistency of the doubly
 * linked list.
 */
void Surface::nc_point_inc()
{
  Point *prev = next - 1;

  if(corners)
  {
    prev->next = next;
    next->prev = prev;
    next->next = pointdata;
    pointdata->prev = next;
  }

  ++next;
  ++corners;
}

/** Increment the point data one step and also append the masks with the
 * bounding data from the latest addition.
 */
void Surface::point_inc()
{
  andmask &= next->mask;
  ormask |= next->mask;
  nc_point_inc();
}

/** Add non-clipped 2d point.
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Surface::nc_point_add(int x, int y)
{
  next->set_nc(x, y);
  nc_point_inc();
}

/** Add non-clipped 2d gouraud point with alpha.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @param a A color component.
 */
void Surface::nc_point_add(int x, int y, int r, int g, int b, int a)
{
  next->set_nc(x, y, r, g, b, a);
  nc_point_inc();
}

/** Add non-clipped 3d point.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param z Z coordinate.
 */
void Surface::nc_point_add(int x, int y, int z)
{
  next->set_nc(x, y, z);
  nc_point_inc();
}

/** Add non-clipped 3d gouraud point with alpha.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param z Z coordinate.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @param a A color component.
 */
void Surface::nc_point_add(int x, int y, int z, int r, int g, int b, int a)
{
  next->set_nc(x, y, z, r, g, b, a);
  nc_point_inc();
}

/** Add clipped 2d point.
 * @param x X coordinate.
 * @param y Y coordinate.
 */
void Surface::point_add(float x, float y)
{
  next->set(x, y);
  next->mask_y(bound);
  point_inc();
}

/** Add clipped 2d gouraud point with alpha.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param col Color in floating point.
 */
void Surface::point_add(float x, float y, const Color4& col)
{
  next->set(x, y, col);
  next->mask_y(bound);
  point_inc();
}

/** Add clipped 3d point.
 * @param pos Position in floating point.
 */
void Surface::point_add(const Vector3& pos)
{
  next->set(pos);
  next->mask_z(bound);
  point_inc();
}

/** Add clipped 3d gouraud point.
 * @param pos Position in floating point.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 */
void Surface::point_add(const Vector3& pos, float r, float g, float b)
{
  next->set(pos, r, g, b);
  next->mask_z(bound);
  point_inc();
}

/** Add clipped 3d gouraud point with alpha.
 * @param pos Position in floating point.
 * @param col Color in floating point.
 */
void Surface::point_add(const Vector3& pos, const Color4 &col)
{
  next->set(pos, col);
  next->mask_z(bound);
  point_inc();
}

//############################################################################
// Clip spot #################################################################
//############################################################################

/** Clips one spot. Please note, this is actually highly suboptimal.
 * @return True if inside, false if outside.
 */
bool Surface::point_clip_spot_zyx()
{
  Point *p1 = &pointdata[0];
  if(p1->outside())
  {
    return false;
  }
  p1->project(bound);
  if(p1->outside())
  {
    return false;
  }
  p1->mask_x(bound);
  if(p1->outside())
  {
    return false;
  }
  static_cast<Point3F*>(p1)->setvi();
  return true;
}

//############################################################################
// Clip line #################################################################
//############################################################################

/** Pass one cylce/boundary of line clipping.
 * Please note, that since this function will NOT be ran if both points are
 * outside the boundary, it can take some freedom in comparisons.
 * @param p1 The first point.
 * @param p2 The second point.
 * @param min Minimum boundary.
 * @param max Maximum boundary.
 * @param clipfunc Clipping function to be called if neccessary.
 */
inline void clip_line_pass(Point *p1, Point *p2, float min, float max,
    void (*clipfunc)(Point*, const Point*, const Point*, float))
{
  if(p1->outside(Point::OUTSIDE_MIN))
  {
    if(p2->outside(Point::OUTSIDE_MAX))
    {
      clipfunc(p1, p1, p2, min);
      clipfunc(p2, p1, p2, max);
    }
    else // p2 inside (wouldn't be ran)
    {
      clipfunc(p1, p1, p2, min);
    }
  }
  else if(p1->outside(Point::OUTSIDE_MAX))
  {
    if(p2->outside(Point::OUTSIDE_MIN))
    {
      clipfunc(p1, p1, p2, max);
      clipfunc(p2, p1, p2, min);
    }
    else // p2 inside (wouldn't be ran)
    {
      clipfunc(p1, p1, p2, max);
    }
  }
  else // p1 inside
  {
    if(p2->outside(Point::OUTSIDE_MIN))
    {
      clipfunc(p2, p1, p2, min);
    }
    else if(p2->outside(Point::OUTSIDE_MAX))
    {
      clipfunc(p2, p1, p2, max);
    }
  }
}

/** Clip one line to all boundaries.
 * @return True if at least something was inside, false if not.
 */
template <class Type> bool Surface::point_clip_line_zyx()
{
  Type *p1 = static_cast<Type*>(&pointdata[0]),
       *p2 = static_cast<Type*>(&pointdata[1]);

  // Z mask is already calculated
  // If both values outside z, just return
  if(Surface::andmask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // If something is outside z, clip.
  if(Surface::andmask != Point::INSIDE)
  {
    clip_line_pass(p1, p2, bound->zmin, bound->zmax, point_clip_z<Type>);
  }

  // Project both, will also calculate y mask.
  p1->project(bound);
  p2->project(bound);

  // If both outside y, just return.
  if(p1->mask & p2->mask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // If something is outside y, clip.
  if((p1->mask | p2->mask) != Point::INSIDE)
  {
    clip_line_pass(p1, p2, bound->ymin_f, bound->ymax_f, point_clip_y<Type>);
  }

  // Calculate x masks.
  p1->mask_x(bound);
  p2->mask_x(bound);
  
  // If both outside x, just return
  if(p1->mask & p2->mask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // If something is outside x, clip
  if((p1->mask | p2->mask) != Point::INSIDE)
  {
    clip_line_pass(p1, p2, bound->xmin_f, bound->xmax_f, point_clip_x<Type>);
  }

  // Round results and exit
  p1->setvi();
  p2->setvi();
  return true;
}

//############################################################################
// Clip poly #################################################################
//############################################################################

/** Perform backface culling on the projected polygon in the state machine.
 * Takes 3 first points, based on the formula:
 * c = (x1 - x2) * (y3 - y2) - (y1 - y2) * (x3 - x2) that is > 0 if the
 * polygon is visible.
 * @return True if visible, false if not.
 */
bool Surface::point_bfc()
{
  Point *p1 = Surface::first,
	*p2 = p1->next,
	*p3 = p2->next;

  return (((p1->pos.xf - p2->pos.xf) * (p3->pos.yf - p2->pos.yf) >
    (p1->pos.yf - p2->pos.yf) * (p3->pos.xf - p2->pos.xf)));
}

/** Pass one cycle/border of polygon clipping.
 * @param out_mask The out mask against which to check.
 * @param value The clipping value which represents the boundary.
 * @param clipfunc the clipping function to use.
 */
void Surface::clip_poly_pass(uint8_t out_mask, float value,
    void (*clipfunc)(Point*, const Point*, const Point*, float))
{
  Point *p1 = Surface::first, *p2;

  // Quickly search first point where p1 is in and the next is out.
  if(p1->outside(out_mask))
  {
    do {
      p2 = p1;
      p1 = p1->prev;
    } while(p1->outside(out_mask));
  }
  else
  {
    p2 = p1->next;
    while(!(p2->outside(out_mask)))
      p1 = p2, p2 = p2->next;
  }

  // Now p1 is in and p2 is not, let's clip.
  Point *new1 = Surface::next,
	*new2 = new1 + 1;
  
  // Create a new point by clipping from p1 to p2 (out).
  clipfunc(new1, p1, p2, value);
  
  // Search until p2 comes back in.
  for(p2 = p2->next; p2->outside(out_mask); p2 = p2->next)
  {
    // For each corner we spend outside the border, decrease count.
    Surface::corners--;
  }
  
  // Create a new point by clippin p2 in.
  clipfunc(new2, p2->prev, p2, value);

  // Set masks correctly.
  new1->mask = new2->mask = Point::INSIDE;

  // Set continuity.
  p1->next = new1;
  new1->prev = p1;
  new1->next = new2;
  new2->prev = new1;
  new2->next = p2;
  p2->prev = new2;
  Surface::next += 2;
  Surface::corners++;
  Surface::first = p1;
}

/** Perform z polygon clipping, then pass control to the x/y clipping
 * @return True if polygon is to be drawn, false if not.
 */
template <class Type> bool Surface::point_clip_poly_zyx()
{
  // If all out of z, quit.
  if(Surface::andmask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // Clip to zmin if neccessary.
  if(ormask & Point::OUTSIDE_MIN)
  {
    clip_poly_pass(Point::OUTSIDE_MIN, bound->zmin, point_clip_z<Type>);
  }

  // Clip to zmax if neccessary.
  if(ormask & Point::OUTSIDE_MAX)
  {
    clip_poly_pass(Point::OUTSIDE_MAX, bound->zmax, point_clip_z<Type>);
  }

  // Project everything and calculate mask.
  uint8_t amask = 0xFF, omask = 0x00;
  Point *p1 = Surface::first;
  do {
    p1->project(bound);
    amask &= p1->mask;
    omask |= p1->mask;
    p1 = p1->next;
  } while(p1 != Surface::first);

  andmask = amask;
  ormask = omask;

  // Perform backface culling and continue to clip-xy if it is ok.
  return point_bfc() ? point_clip_poly_yx<Type>() : false;
}

/** Perform y/x polygon clipping.
 * @return True if polygon is to be drawn, false if not.
 */
template <class Type> bool Surface::point_clip_poly_yx()
{
   // If all outside y, return
  if(andmask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // Clip to ymin if neccessary
  if(ormask & Point::OUTSIDE_MIN)
  {
    clip_poly_pass(Point::OUTSIDE_MIN, bound->ymin_f, point_clip_y<Type>);
  }

  // Clip to ymax if neccessary
  if(ormask & Point::OUTSIDE_MAX)
  {
    clip_poly_pass(Point::OUTSIDE_MAX, bound->ymax_f, point_clip_y<Type>);
  }

  // Calculate new masks after y clipping
  uint8_t amask = 0xFF, omask = 0x00;
  Type *p1 = static_cast<Type*>(Surface::first);
  do {
    p1->mask_x(bound);
    amask &= p1->mask;
    omask |= p1->mask;
    p1 = static_cast<Type*>(p1->next);
  } while(p1 != Surface::first);
 
  // If all outside x, return
  if(amask & Point::OUTSIDE_EITHER)
  {
    return false;
  }

  // Clip to xmin if neccessary
  if(omask & Point::OUTSIDE_MIN)
  {
    clip_poly_pass(Point::OUTSIDE_MIN, bound->xmin_f, point_clip_x<Type>);
  }

  // Clip to xmax if neccessary
  if(omask & Point::OUTSIDE_MAX)
  {
    clip_poly_pass(Point::OUTSIDE_MAX, bound->xmax_f, point_clip_x<Type>);
  }

  // Setvi everything
  p1 = static_cast<Type*>(Surface::first);
  do {
    p1->setvi();
    p1 = static_cast<Type*>(p1->next);
  } while(p1 != Surface::first);

  return true;
}

//############################################################################
// Leikatut fillerit #########################################################
//############################################################################

/** Draw a clipped polygon off the point data.
 * @param col Drawing color.
 */
void Surface::c_poly_flat(int col)
{
  if(point_clip_poly_yx<Point2F>())
  {
    if(corners == 3)
    {
      nc_triangle_flat(col);
    }
    else
    {
      nc_poly_flat(col);
    }
  }
}

/** Draw a clipped groudaud-shaded polygon off the point data.
 */
void Surface::c_poly_gouraud()
{
  if(point_clip_poly_yx<Point2G>())
  {
    if(corners == 3)
    {
      nc_triangle_gouraud();
    }
    else
    {
      nc_poly_gouraud();
    }
  }
}

/** Draw a clipped line with perspective off the point data.
 * @param col Drawing color.
 */
void Surface::c_zbuf_line_flat(int col)
{
  if(point_clip_line_zyx<Point3F>())
  {
    nc_zbuf_line_flat(col);
  }
}

/** Draw a clipped gouraud-shaded line with perspective off the point data.
 */
void Surface::c_zbuf_line_gouraud()
{
  if(point_clip_line_zyx<Point3G>())
  {
    nc_zbuf_line_gouraud();
  }
}

/** Draw a clipped polygon with perspective off the point data.
 * @param col Drawing color.
 */
void Surface::c_zbuf_poly_flat(int col)
{
  if(point_clip_poly_zyx<Point3F>())
  {
    if(corners == 3)
    {
      nc_zbuf_triangle_flat(col);
    }
    else
    {
      nc_zbuf_poly_flat(col);
    }
  }
}

/** Draw a clipped gouraud-shaded polygon with perspective off the point data.
 * @return Zero if not drawn, 1 if yes.
 */
void Surface::c_zbuf_poly_gouraud()
{
  if(point_clip_poly_zyx<Point3G>())
  {
    if(corners == 3)
    {
      nc_zbuf_triangle_gouraud();
    }
    else
    {
      nc_zbuf_poly_gouraud();
    }
  }
}

//############################################################################
// Variantit #################################################################
//############################################################################

/** Template for selecting the drawing primitives.
 */
template <class Type> void Surface::select_template()
{
  // Color creators.
  makecol3 = inline_makecol3<Type>;
  makecol4 = inline_makecol4<Type>;

  // Not clipped.
  nc_setpixel = nc_setpixel_template<Type>;
  nc_hline = nc_hline_template<Type>;
  nc_vline = nc_vline_template<Type>;
  nc_line = nc_line_template<Type>;
  nc_rect = nc_rect_template<Type>;
  nc_rect_contour = nc_rect_contour_template<Type>;
  nc_triangle_flat = nc_triangle_flat_template<Type>;
  nc_triangle_gouraud = nc_triangle_gouraud_template<Type>;
  nc_poly_flat = nc_poly_flat_template<Type>;
  nc_poly_gouraud = nc_poly_gouraud_template<Type>;
  nc_zbuf_spot = nc_zbuf_spot_template<Type>;
  nc_zbuf_line_flat = nc_zbuf_line_flat_template<Type>;
  nc_zbuf_line_gouraud = nc_zbuf_line_gouraud_template<Type>;
  nc_zbuf_triangle_flat = nc_zbuf_triangle_flat_template<Type>;
  nc_zbuf_triangle_gouraud = nc_zbuf_triangle_gouraud_template<Type>;
  nc_zbuf_poly_flat = nc_zbuf_poly_flat_template<Type>;
  nc_zbuf_poly_gouraud = nc_zbuf_poly_gouraud_template<Type>;

  // Clipped.
  setpixel = c_setpixel;
  hline = c_hline;
  vline = c_vline;
  line = c_line;
  rect = c_rect;
  rect_contour = c_rect_contour;
  poly_flat = c_poly_flat;
  poly_gouraud = c_poly_gouraud;
  zbuf_line_flat = c_zbuf_line_flat;
  zbuf_line_gouraud = c_zbuf_line_gouraud;
  zbuf_poly_flat = c_zbuf_poly_flat;
  zbuf_poly_gouraud = c_zbuf_poly_gouraud;

  // Always clipped.
  draw_glyph = draw_glyph_template<Type>;

  // High-level.
  draw_text = c_draw_text;
  draw_model = c_draw_model;
  draw_texture = c_draw_texture;
}

/** Select function selects this surface for all subsequent drawing methods.
 * This must be done first, before any operations on any surface that are not
 * really methods.
 */
void Surface::select()
{
  // Set buffer and control data
  bound = &boundary;
  cdata = cbuf;
  zdata = zbuf;
  xmul = w;

  // Set primitive data
  switch(bpp)
  {
    case 8:
      select_template<SURFACE_TYPEDEF_8>();
      break;
    case 16:
      select_template<SURFACE_TYPEDEF_16>();
      break;
    default: // 32
      select_template<SURFACE_TYPEDEF_32>();
      break;
  }
}

//############################################################################
// 3D-osio ###################################################################
//############################################################################

Camera *Surface::camera;
Light *Surface::light;

//############################################################################
// 3D-varaus- ja asetusfunktiot ##############################################
//############################################################################

/** Select this surface for 3d drawing functions.
 */
void Surface::select_3d()
{
  // Do nothing.
}

/** Select this surface for 2d drawing functions.
 */
void Surface::select_2d()
{
  // Do nothing.
}

/** Set a camera for successive drawing functions.
 * This function must be rerun if the state of the camera changes (note that
 * although it might not apply here, it can apply to derivative
 * implementations).
 * @param op [in] The camera to set.
 */
void Surface::set_camera(Camera *op)
{
  Surface::camera = op;
  op->tick();
}

/** Sets the light to the software surface.
 * This function must be rerun if the state of the light changes.
 * @param op [in] The light to use.
 */
void Surface::set_light(Light *op)
{
  Surface::light = op;
  op->tick();
}

//############################################################################
// High-level drawing methods (these call function pointers) #################
//############################################################################

/** Outputs text to the selected surface. Clipping implemented here since
 * it's hard to 'write in' in the latter one. Because of this, there is no
 * 'clipped' version of textout at all.
 * @param x The left x coordinate.
 * @param y The lower y coordinate of the first line.
 * @param col Color to draw with.
 * @param font Font pointer.
 * @param text UTF-8 text string.
 * @return The width of text drawn in pixels.
 */
int Surface::c_draw_text(int x, int y, int col, Font *font, const char *text)
{
  char *str = const_cast<char*>(text);
  int cx = x,
      ret = 0;

  while(*str)
  {
    int unicode = strdecodeutf8(str);
    Glyph *glyph = font->get_glyph(unicode);

    if(!glyph)
    {
      continue; // Erraneous loads are skipped (no garble, just nothing).
    }

    // FIXME: implement clipping.

    draw_glyph(cx, y, col, glyph);

    int advance = glyph->get_advance_x();

    cx += advance;
    ret += advance;
  }

  return ret;
}


/** Draw one object on the screen according to the state machine.
 * Must be called after set_camera and set_light, should be overwritten by
 * implementation-dependant hardware-accelerated methods. This is just the
 * software renderer version.
 */
void Surface::c_draw_model(PostModel *mdl)
{
  Mesh *mesh;
  CVUnit *cvdata;
  MESH_TYPEDEF_INDEX *elem;

  if(mdl == NULL)
  {
    return;
  }
  mesh = mdl->get_mesh();

  // Check for immediate exit.
  if((mesh == NULL) || (mesh->get_num_vertex() <= 0) || (!mdl->has_draw()))
  {
    return;
  }

  // Perform transformations at the light.
  cvdata = light->transform(camera->get_vm(), mdl, mesh);

  // Point list.
  if((elem = mesh->get_array_elem(MESH_POINTS)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      float *color = feed_vertex_flat_with_color(cvdata, *elem++);
      if(point_clip_spot_zyx())
      {
	nc_zbuf_spot(makecol_c3f(color));
      }
    } while(i -= 1);
  }

  // Line list.
  if((elem = mesh->get_array_elem(MESH_LINES)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      c_zbuf_line_gouraud();
    } while(i -= 2);
  }

  // Triangle list.
  if((elem = mesh->get_array_elem(MESH_TRIANGLES)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      c_zbuf_poly_gouraud();
    } while(i -= 3);
  }

  // Quad list.
  if((elem = mesh->get_array_elem(MESH_QUADS)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      feed_vertex_gouraud(cvdata, *elem++);
      c_zbuf_poly_gouraud();
    } while(i -= 4);
  }

  // Flat line list.
  if((elem = mesh->get_array_elem(MESH_FLAT_LINES)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      float *color = feed_vertex_flat_with_color(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      if(point_clip_line_zyx<Point3F>())
      {
	nc_zbuf_line_flat(makecol_c3f(color));
      }
    } while(i -= 2);
  }

  // Flat triangle list.
  if((elem = mesh->get_array_elem(MESH_FLAT_TRIANGLES)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      float *color = feed_vertex_flat_with_color(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      if(point_clip_poly_zyx<Point3F>())
      {
	nc_zbuf_poly_flat(makecol_c3f(color));
      }
    } while(i -= 3);
  }

  // Flat quad list.
  if((elem = mesh->get_array_elem(MESH_FLAT_QUADS)) != NULL)
  {
    MESH_TYPEDEF_INDEX i = *(elem++);
    do {
      point_begin();
      float *color = feed_vertex_flat_with_color(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      feed_vertex_flat(cvdata, *elem++);
      if(point_clip_poly_zyx<Point3F>())
      {
	nc_zbuf_poly_flat(makecol_c3f(color));
      }
    } while(i -= 4);
  }
}

/** Draw a texture from texture file to a confined space given by the four
 * coordinates. To draw the texture without stretching, specify coordinates
 * by using the get width / height information from the texture.
 * @param x1 Rectangle x1 coordinate.
 * @param y1 Rectangle y1 coordinate.
 * @param x2 Rectangle x2 coordinate.
 * @param y2 Rectangle y2 coordinate.
 * @param tex Texture to update.
 */
void Surface::c_draw_texture(int x1, int y1, int x2, int y2, Texture *tex)
{
  cmpswap(x1, y1);
  cmpswap(x2, y2);

  // TODO: implement.
}

//############################################################################
// Empty functions (for compatimility / inheritance) #########################
//############################################################################

/*
 * A note of information here. The following methods should be overwritten in
 * surface implementations requiring and/or offering them. They should be
 * called in overlying code that is not aware of the implementation it is
 * using.
 */

/** Lock this surface.
 * Not implemented in base surface class.
 */
void Surface::lock()
{
  return;
}

/** Unlock this surface.
 * Not implemented in base surface class.
 */
void Surface::unlock()
{
  return;
}

//############################################################################
// Save methods ##############################################################
//############################################################################

#ifdef LIBFHI_PNG
/** Save this surface as a png file.
 * @param filename Name of the file to create.
 */
bool Surface::save_png(const char *filename)
{
  return write_png(this->w, this->h, this->bpp, this->cbuf, filename);
}
#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

