#include "entityarchtype.h"

#include "data.h"
#include "entity.h"
#include "sound.h"
#include "thurster.h"
#include "turret.h"
#include "weaponslot.h"

#include "effect.h"

#include <sstream>

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor. Does nothing except to null position and movement.
 */
EntityArchtype::EntityArchtype() :
  Attributable(ATTR_DEFAULT), longest_radius(0.0f),
  ai_aim_proximity(AI_DEFAULT_AIM_PROXIMITY), bleed(NULL), die(NULL),
  die_sample(NULL)
{
  // Do nothing.
}

/** Load constructor.
 * @param filename The filename to load from.
 */
EntityArchtype::EntityArchtype(const char *filename) :
  Attributable(ATTR_DEFAULT), longest_radius(0.0f),
  ai_aim_proximity(AI_DEFAULT_AIM_PROXIMITY), bleed(NULL), die(NULL),
  die_sample(NULL)
{
  float scale[3] = { 1.0f, 1.0f, 1.0f },
	cr = 1.0f,
	sr = 0.0f;
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cout << "ERROR: could not model archtype data: \"" << filename
      << "\"\n";
    return;
  }

  while(filu.advance())
  {
    // Scale
    if(filu.has_id("scale"))
    {
      switch(filu.get_num_arg())
      {
	case 3:
    	  scale[0] = filu.get_float(0);
    	  scale[1] = filu.get_float(1);
    	  scale[2] = filu.get_float(2);
	  break;
	case 1:
	  scale[0] = scale[1] = scale[2] = filu.get_float(0);
	  break;
	default:
	  filu.warn("needs 1 or 3 arguments");
	  break;
      }
    }
    // Rotation.
    else if(filu.has_id_arg("rotate", 1))
    {
      float pi = libfhi::uint2pi(static_cast<uint16_t>(filu.get_int(0)));
      cr = cosf(pi);
      sr = sinf(pi);
    }
    // Name.
    else if(filu.has_id("name"))
    {
      std::stringstream fullname;

      fullname << filu.get_string(0);

      for(size_t i = 1; (i < filu.get_num_arg()); ++i)
      {
	fullname << " " << filu.get_string(i);
      }

      this->name = fullname.str();
    }
    // Mass.
    else if(filu.has_id_arg("mass", 1))
    {
      this->mass = filu.get_float(0);
    }
    // Thurst.
    else if(filu.has_id_arg("thurst", 4))
    {
      this->thurst[0] = filu.get_float(0);
      this->thurst[1] = filu.get_float(1);
      this->thurst[2] = filu.get_float(2);
      this->thurst[3] = filu.get_float(3);
    }
    // Drag.
    else if(filu.has_id_arg("drag", 2))
    {
      this->drag_vertical = filu.get_float(0);
      this->drag_horizontal = filu.get_float(1);
    }
    // Rotation.
    else if(filu.has_id_arg("rot", 1))
    {
      this->rotspeed = filu.get_int(0);
    }
    // Aiming proximity.
    else if(filu.has_id_arg("aim_proximity", 1))
    {
      this->ai_aim_proximity = filu.get_float(0);
    }
    // Life.
    else if(filu.has_id_arg("life", 1))
    {
      this->life_maximum = filu.get_int(0);
    }
    // Attributes.
    else if(filu.has_id_arg("attr", 1))
    {
      this->set_attr(filu.get_int(0));
    }
    // Sample
    else if(filu.has_id_arg("die_sample", 1))
    {
      this->die_sample = Data::load_sample(filu.get_string(0).c_str());
    }
    // Explosion.
    else if(filu.has_id_arg("die", 2))
    {
      this->die = Data::load_eff(filu.get_string(0).c_str());
      this->die_multitude = filu.get_int(1);
    }
    // Small explosion.
    else if(filu.has_id_arg("bleed", 1))
    {
      this->bleed = Data::load_eff(filu.get_string(0).c_str());
    }
    // Thurster forward.
    else if(filu.has_id_arg("thurster", 6))
    {
      libfhi::Vector2
	pvec(filu.get_float(1) * scale[0], filu.get_float(2) * scale[1]),
	rvec;
      rvec.rotate(pvec, cr, sr);

      this->thursters[filu.get_int(0)].push_back(new Thurster(
	    rvec,
	    filu.get_float(3),
	    filu.get_int(4),
	    filu.get_string(5).c_str()));
    }
    // Collision map.
    else if(filu.has_id_arg("cmap", 2))
    {
      int attr = filu.get_int(0);
      CollisionMap *ld = Data::load_cmap(filu.get_string(1).c_str());

      if(attr & 1)
      {
	ld->attr_or(CollisionMap::ATTR_NORMAL);
      }
      if(attr & 2)
      {
	ld->attr_or(CollisionMap::ATTR_ABSORB);
      }

      ld->section(&(this->maps));
      // Do NOT delete the loaded collisionmap. It is saved in the canonical
      // database and will be fetched from there on successive loads.
    }
    // Model.
    else if(filu.has_id_arg("msh", 2))
    {
      int attr = filu.get_int(0);
      libfhi::Mesh* ld = Data::load_msh(filu.get_string(1).c_str());

      // Allow all meshes to be cached, probably have to turn this off later
      // on, but for now it gives a nice speed boost.
      ld->set_cache();

      if(attr & 1)
      {
	ld->attr_or(libfhi::Mesh::ATTR_USER_1);
      }
      if(attr & 2)
      {
	ld->attr_or(libfhi::Mesh::ATTR_USER_2);
      }

      // Okay, we added a new mesh, update the longest radius if neccessary.
      this->longest_radius = stdmax(this->longest_radius,
	  ld->get_maxdist());

      this->meshes.push_back(ld);
      // Do not delete the loaded mesh, since it is saved to the canonical
      // database and will be removed from there in a time.
    }
    // Weapon slot.
    else if(filu.has_id_arg("weap", 4))
    {
      libfhi::Vector2
	pvec(filu.get_float(0) * scale[0], filu.get_float(1) * scale[1]),
	rvec;
      rvec.rotate(pvec, cr, sr);

      WeaponMount *mount = Data::load_mount(filu.get_string(3).c_str());
      this->weaponslots.push_back(new WeaponSlot(
	    rvec,
	    filu.get_int(2),
	    mount));
    }
    // Turret.
    else if(filu.has_id_arg("turret", 7))
    {
      libfhi::Vector2
	pvec(filu.get_float(0) * scale[0], filu.get_float(1) * scale[1]),
	rvec;
      rvec.rotate(pvec, cr, sr);

      this->turrets.push_back(new Turret(
	    rvec,
	    scale[2] * filu.get_float(2),
	    filu.get_int(3),
	    filu.get_int(4),
	    filu.get_int(5),
	    Data::load_ent(filu.get_string(6).c_str())));
    }
    else
    {
      filu.warn_empty();
    }
  }

  std::cout << "Loaded entity archtype \"" << filename << "\" with M(" <<
    this->mass << ") T(" << this->thurst[0] << "," << this->thurst[1] << ","
    << this->thurst[2] << "," << this->thurst[3] << ") D(" <<
    this->drag_vertical << "," << this->drag_horizontal << ") R(" <<
    this->rotspeed << ")\n";

  // Physics engine has a different scale from real life. We must apply this
  // scale to allow realistic and sensible input of values. Note that even
  // though the scaling power for lenght to mass is three, we use four here to
  // counter the last notch of 1:8 length disparity. The drag coefficient
  // seems to work as-is, as fixing the others fixes it too.
  this->mass = (mass * 0.5f) / powf(PHYSICS_STEP_SPACE, 4.0f);
  for(int i = 0; (i < static_cast<int>(THURSTER_COUNT)); ++i)
  {
    this->thurst[i] = (this->thurst[i] * 0.5f) /
      powf(PHYSICS_STEP_SPACE, 4.0f);
  }
  this->drag_vertical /= powf(PHYSICS_STEP_SPACE, 2.0f);
  this->drag_horizontal /= powf(PHYSICS_STEP_SPACE, 2.0f);

  // Calculate minimum and maximum points of this mesh (for bleeding and
  // dying).
  this->corner_ll.set(FLT_MAX, FLT_MAX);
  this->corner_ur.set(-FLT_MAX, -FLT_MAX);
  for(std::vector<libfhi::Mesh*>::iterator i = this->meshes.begin(),
      e = this->meshes.end(); (i != e); ++i)
  {
    this->corner_ll.set(
	stdmin(this->corner_ll.xf, (*i)->get_minimum_in_axis(0)),
	stdmin(this->corner_ll.yf, (*i)->get_minimum_in_axis(1)));
    this->corner_ur.set(
    	stdmax(this->corner_ur.xf, (*i)->get_maximum_in_axis(0)),
    	stdmax(this->corner_ur.yf, (*i)->get_maximum_in_axis(1)));
  }

  // Okay, about the maxdist, if we have turrets, it's essential to walk
  // through them to get actual maximum distances.
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    this->longest_radius = stdmax(this->longest_radius,
  	(*i)->get_pos().length() +
	(*i)->get_base_class()->get_longest_radius());
  }

  // Square the AI aim proximity.
  this->ai_aim_proximity *= this->ai_aim_proximity;
}

/** Default destructor. Not deleting the meshes, they are in their own
 * canondb.
 */
EntityArchtype::~EntityArchtype()
{
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    delete *i;
  }
  for(std::list<Turret*>::iterator i = this->turrets.begin(),
      e = this->turrets.end(); (i != e); ++i)
  {
    delete *i;
  }
  for(std::vector<WeaponSlot*>::iterator i = this->weaponslots.begin(),
      e = this->weaponslots.end(); (i != e); ++i)
  {
    delete *i;
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Fires one set of thursters.
 * @param hpos Host position.
 * @param hmov Host movement.
 * @param hdir Host direction.
 * @param vec Vector pointer to thursters.
 */
void EntityArchtype::fire_thursters(ThursterEnum thid,
    const libfhi::Vector2 &hpos, const libfhi::Vector2 &hmov, uint16_t hdir)
{
  std::vector<Thurster*> *thvec = this->thursters + thid;

  for(std::vector<Thurster*>::iterator i = thvec->begin(),
      e = thvec->end(); (i != e); ++i)
  {
    (*i)->fire(hpos, hmov, hdir);
  }
}

/** Invokes some sparks within this entity archtype.
 * @param op The particle generation parameters to use.
 * @param pos Actual position.
 * @param dir Actual direction.
 */
void EntityArchtype::invoke_sparks(const Effect *op,
    const libfhi::Vector2 &pos, uint16_t dir) const
{
  float w = (this->corner_ur.xf - this->corner_ll.xf) * 0.001f,
	h = (this->corner_ur.yf - this->corner_ll.yf) * 0.001f;

  // Randomize a point within the boundaries of this.
  libfhi::Vector2 invoke_pos(
      corner_ll.xf + static_cast<float>(rand() % 1001) * w,
      corner_ll.yf + static_cast<float>(rand() % 1001) * h);

  // Get rotation of this.
  float pi = libfhi::uint2pi(dir);
  libfhi::Vector2 dirvec(cosf(pi), sinf(pi));

  // Translate position.
  libfhi::Vector2 tpos;
  tpos.rotate_translate(invoke_pos, dirvec.xf, dirvec.yf, pos);

  // Generate batch in the rotated position-
  op->execute(&tpos, &dirvec);
}

//############################################################################
// End #######################################################################
//############################################################################

