#ifndef weaponslot_h
#define weaponslot_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

#include "weaponmount.h"

//############################################################################
// Class #####################################################################
//############################################################################

/** Weaponslot specifies one weapon an entity can fire. This encompsses the
 * weapon's type (model created on firing), recharge time and start position,
 * and direction.
 */
class WeaponSlot
{
  private:
    /** The relative position to the host wherefrom new bullets spawn. */
    libfhi::Vector2 pos;

    /** Direction into which bullets are fired (0 == forward). */
    uint16_t dir;

    /** Reload time left or charge time used. */
    int reload;

    /** Current ammo count. Increments by 1 each frame. */
    int ammo;

    /** Burst shots to fire left. */
    int queue_cnt;

    /** Pointer to the mountable weapon used here. */
    WeaponMount *mount;

  public:
    WeaponSlot(const libfhi::Vector2&, uint16_t, WeaponMount*);
    WeaponSlot(WeaponSlot*);
    ~WeaponSlot();

  public:
    void fire(int, const libfhi::Vector2&, const libfhi::Vector2&, uint16_t,
	int, int);
    int draw(int, int, libfhi::Surface*);

    // Inline methods.
  public:
    inline WeaponMount* get_mount();
    inline float get_speed();
    inline bool has_ammo() const;
    inline bool has_charge(int) const;
    inline void tick();
};

//############################################################################
// Inline ####################################################################
//############################################################################

/** Tell if this is a reloadable weapon that has ammo.
 * @return True if yes, false if no.
 */
bool WeaponSlot::has_ammo() const
{
  return (!this->mount->has_charge() &&
      (this->ammo >= this->mount->get_drain()));
}

/** Tell if this is a charging weapon that is ready to fire with given charge.
 * @param op Charge to test against.
 * @return True if yes, false if no.
 */
bool WeaponSlot::has_charge(int op) const
{
  return (this->mount->has_charge() &&
      (op >= this->mount->get_charge()));
}

/** Return the mount used by this particular weapon.
 * @return Pointer to the mount.
 */
WeaponMount* WeaponSlot::get_mount()
{
  return this->mount;
}

/** Get the speed of bullets this slot generates.
 * @return speed as float.
 */
float WeaponSlot::get_speed()
{
  return this->mount->get_speed();
}

/** Tick this slot one step forward. Basically, it increases ammo by one and
 * decrements reload time by one.
 */
void WeaponSlot::tick()
{
  this->ammo = stdmin(this->ammo + 1, this->mount->get_max_ammo());
  this->reload = stdmax(this->reload - 1, 0);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

