using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace StarifficEditor
{
    public partial class EditorForm : Form
    {
        private LevelObject m_selected_obj;

        private RunGameForm m_run_form = new RunGameForm();

        private String m_path_levels;
        private String m_path_export;
        private String m_path_meshes;

        private Meshes m_meshes;

        public Level level
        {
            get { return editorControlMain.level; }
        }

        public EditorForm()
        {
            InitializeComponent();

            // Set the default directories
            m_path_levels = Path.Combine(Application.StartupPath, "Levels");
            m_path_export = Path.Combine(Application.StartupPath, "Export");
            m_path_meshes = Path.Combine(Application.StartupPath, "Meshes");

            openFileDialog.InitialDirectory = m_path_levels;
            saveFileDialog.InitialDirectory = m_path_levels;
            exportFileDialog.InitialDirectory = m_path_export;

            // Read the meshes
            m_meshes = new Meshes(m_path_meshes);
            Meshes.global_meshes = m_meshes;

            // Init the creatable types list
            foreach (LevelObjectType val in Enum.GetValues(typeof(LevelObjectType)))
            {
                if (val == LevelObjectType.Start ||
                    val == LevelObjectType.End)
                    continue;
                comboBoxNewType.Items.Add(val);
            }
            comboBoxNewType.SelectedIndex = 0;

            // Add the level listener
            level.level_changed_event += new LevelChanged(LevelChanged);
            UpdateFromLevel();

            SelectObject(null);
        }

        private void UpdateFromLevel()
        {
            // Test for changing object lists
            bool objs_changed = listBoxObjects.Items.Count != level.objects.Count;
            int new_obj_idx = -1;
            if (objs_changed == false)
            {
                LevelObject sel_item = (LevelObject) listBoxObjects.SelectedItem;
                new_obj_idx = level.objects.IndexOf(sel_item);

                for (int i = 0; i < listBoxObjects.Items.Count; ++i)
                {
                    if (listBoxObjects.Items[i] != level.objects[i])
                    {
                        objs_changed = true;
                        break;
                    }
                }
            }

            if (objs_changed)
            {
                listBoxObjects.Items.Clear();
                listBoxObjects.Items.AddRange(level.objects.ToArray());
                listBoxObjects.SelectedIndex = new_obj_idx;
            }

            // Update the property grid
            if (propertyGrid.SelectedObject != null)
            {
                if (level.objects.Contains((LevelObject)propertyGrid.SelectedObject) == false)
                    propertyGrid.SelectedObject = null;
                else
                    propertyGrid.SelectedObject = propertyGrid.SelectedObject;
            }
            editorControlMain.Invalidate();
        }

        private void LevelChanged()
        {
            UpdateFromLevel();
        }

        private void SelectObject(LevelObject obj)
        {
            int obj_idx = level.objects.IndexOf(obj);
            if (obj_idx == -1)
                obj = null;

            m_selected_obj = obj;

            if (obj == null)
            {
                listBoxObjects.SelectedIndex = -1;
                propertyGrid.SelectedObject = null;
                buttonDelete.Enabled = false;
            }
            else
            {
                listBoxObjects.SelectedIndex = obj_idx;
                propertyGrid.SelectedObject = obj;
                buttonDelete.Enabled = obj.Deletable;
            }
            buttonUp.Enabled = (obj != null);
            buttonDown.Enabled = (obj != null);

            editorControlMain.SelectedObject = obj;
            editorControlMain.Invalidate();
        }

        private void listBoxObjects_SelectedIndexChanged(object sender, EventArgs e)
        {
            LevelObject obj = null;

            int idx = listBoxObjects.SelectedIndex;
            if (idx >= 0 && idx < level.objects.Count)
                obj = level.objects[idx];

            SelectObject(obj);
        }

        private void buttonNew_Click(object sender, EventArgs e)
        {
            LevelObject obj_to_add = LevelObjectFactory.CreateObject((LevelObjectType)comboBoxNewType.Items[comboBoxNewType.SelectedIndex]);
            if (obj_to_add == null)
                return;

            PointF pos = editorControlMain.GetCenterMapCoord();
            obj_to_add.X = pos.X;
            obj_to_add.Y = pos.Y;

            level.objects.Insert(0, obj_to_add);
            listBoxObjects.Items.Insert(0, obj_to_add);
            SelectObject(obj_to_add);
        }

        private void propertyGrid_PropertyValueChanged(object s, PropertyValueChangedEventArgs e)
        {
            level.NotifyChanges();
            listBoxObjects.Items[listBoxObjects.SelectedIndex] = listBoxObjects.Items[listBoxObjects.SelectedIndex];
        }

        private void propertyGrid_SelectedObjectsChanged(object sender, EventArgs e)
        {
            editorControlMain.Invalidate();
        }

        private void RunAt(int w, int h, bool use_exported_data)
        {
            String export_path = null;
            if (use_exported_data)
                export_path = m_path_export;

            Form f = new GameForm(level, m_meshes, w, h, export_path);
            f.ShowDialog(this);
        }

        private void RunAt(int w, int h)
        {
            RunAt(w, h, false);
        }

        private void runAt256x256ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            RunAt(256, 256);
        }

        private void runAt176x208ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            RunAt(176, 208);
        }

        private void runAt640x480ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            RunAt(640, 480);
        }

        private void buttonDelete_Click(object sender, EventArgs e)
        {
            if (m_selected_obj == null)
                return;

            level.objects.Remove(m_selected_obj);
            level.NotifyChanges();
        }

        private void loadToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (openFileDialog.ShowDialog(this) != DialogResult.OK)
                return;

            level.Load(openFileDialog.FileName);
            saveFileDialog.FileName = openFileDialog.FileName;

            UpdateFromLevel();
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (saveFileDialog.ShowDialog(this) != DialogResult.OK)
                return;

            level.Save(saveFileDialog.FileName);
        }

        private void newToolStripMenuItem_Click(object sender, EventArgs e)
        {
            level.Clear();
            saveFileDialog.FileName = "";
        }

        private void exportToolStripMenuItem_Click(object sender, EventArgs e)
        {
/*          if (exportFileDialog.ShowDialog(this) != DialogResult.OK)
                return;

            level.Export(exportFileDialog.FileName);*/
            ExportForm form = new ExportForm(m_path_levels, m_path_export, m_meshes);
            form.ShowDialog(this);
        }

        private void checkBoxWireframe_CheckedChanged(object sender, EventArgs e)
        {
            editorControlMain.Wireframe = checkBoxWireframe.Checked;
        }

        private void checkBoxPreview_CheckedChanged(object sender, EventArgs e)
        {
            editorControlMain.Preview = checkBoxPreview.Checked;
        }

        private void buttonUp_Click(object sender, EventArgs e)
        {
            LevelObject obj = m_selected_obj;
            if (obj == null)
                return;

            int idx = level.objects.IndexOf(obj);
            if (idx == -1)
                return;
            if (idx <= 0)
                return;

            level.objects.RemoveAt(idx);
            level.objects.Insert(idx - 1, obj);
            listBoxObjects.Items.Clear();
            UpdateFromLevel();
            SelectObject(obj);
        }

        private void buttonDown_Click(object sender, EventArgs e)
        {
            LevelObject obj = m_selected_obj;
            if (obj == null)
                return;

            int idx = level.objects.IndexOf(obj);
            if (idx == -1)
                return;
            if (idx >= (level.objects.Count-1))
                return;

            level.objects.RemoveAt(idx);
            level.objects.Insert(idx + 1, obj);
            listBoxObjects.Items.Clear();
            UpdateFromLevel();
            SelectObject(obj);
        }

        private void buttonLevelProps_Click(object sender, EventArgs e)
        {
            Form f = new LevelPropertiesForm(level);
            f.ShowDialog(this);
        }

        private void runToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (m_run_form.ShowDialog(this) != DialogResult.OK)
                return;

            RunAt(m_run_form.width, m_run_form.height, m_run_form.run_exported_data);
        }

        private void editorControlMain_SelectedObjectChangedEvent(LevelObject obj)
        {
            SelectObject(obj);
        }

        private void checkBoxShowScreen_CheckedChanged(object sender, EventArgs e)
        {
            editorControlMain.ShowScreen = checkBoxShowScreen.Checked;
        }

        private void quitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }
    }
}