///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_IMAGE_H
#define MOIRA_IMAGE_H
///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

/*! @brief Rectangular image area.
 */
class ImageArea
{
public:
  ImageArea(void);
  ImageArea(unsigned int x,
            unsigned int y,
	    unsigned int width,
	    unsigned int height);
  void set(unsigned int newX,
           unsigned int newY,
	   unsigned int newWidth,
	   unsigned int newHeight);
  unsigned int x;
  unsigned int y;
  unsigned int width;
  unsigned int height;
};

///////////////////////////////////////////////////////////////////////

/*! @brief %Image format descriptor.
 *
 *  All formats are at least byte-aligned, although their channels may not be.
 */
class ImageFormat
{
public:
  /*! %Image format type enumeration.
   */
  enum Type
  {
    /*! The invalid format. You cannot create images of this format.
     */
    INVALID,
    /*! Each pixel is 8 bits of alpha.
     */
    ALPHA8,
    /*! Each pixel is 8 bits of luminance.
     */
    GREY8,
    /*! Each pixel is 8 bits of luminance and 8 bits of alpha.
     */
    GREYALPHA88,
    /*! Each pixel is 24 bits of RGB color.
     */
    RGB888,
    /*! Each pixel is 32 bits, with 8 bits for each of red, green and blue and
     *  8 bits of padding.
     */
    RGBX8888,
    /*! Each pixel is 32 bits, with 8 bits for each of red, green and blue and
     *  8 bits of alpha.
     */
    RGBA8888,
    /*! Each pixel is 32 bits floating point of alpha.
     */
    ALPHA32F,
    /*! Each pixel is 32 bits floating point of luminance.
     */
    GREY32F,
    /*! Each pixel is 64 bits, with 32 bits floating point of luminance and 32
     *  bits floating point of alpha.
     */
    GREYALPHA32F,
    /*! Each pixel is 96 bits, with 32 bits floating point for each of red,
     *  green and blue.
     */
    RGB32F,
    /*! Each pixel is 128 bits, with 32 bits floating point for each of red,
     *  green and blue and 32 bits floating point of padding.
     */
    RGBX32F,
    /*! Each pixel is 128 bits, with 32 bits floating point for each of red,
     *  green and blue and 32 bits floating point of alpha.
     */
    RGBA32F,
  };
  enum Channel
  {
    GREY,
    RED,
    GREEN,
    BLUE,
    ALPHA,
  };
  /*! Default constructor.
   *  @param type The type of this pixel format.
   */
  ImageFormat(Type type = INVALID);
  /*! Decodes a pixel of this format to a ColorRGBA object.
   *  @param result The decoded color.
   *  @param source The address of the pixel to read.
   */
  void decode(ColorRGBA& result, const void* source) const;
  /*! Encodes a ColorRGBA object to a pixel of this format.
   *  @param result The address of the pixel to write.
   *  @param source The ColorRGBA object to encode.
   */
  void encode(void* result, const ColorRGBA& source) const;
  /*! @return @c true if this pixel format has an alpha channel,
   *  otherwise @c false.
   */
  bool hasAlpha(void) const;
  /*! @return The size, in bytes, of a pixel in this pixel format.
   */
  size_t getSize(void) const;
  /*! @return The number of components in this pixel format.
   */
  unsigned int getChannelCount(void) const;
  /*! @return The number of bits in a pixel of this format.
   */
  unsigned int getChannelBitCount(Channel channel) const;
  /*! @return The type of this pixel format.
   */
  operator Type (void) const;
private:
  Type type;
};

///////////////////////////////////////////////////////////////////////

/*! @brief %Image data object.
 *
 *  This is the standard container for all forms of 1D and 2D image data.
 */
class Image : public Resource<Image>, public RefObject<Image>
{
public:
  enum Method
  {
    /*! Use nearest-neighbour sampling.
     */
    SAMPLE_NEAREST,
    /*! Use bilinear sampling.
     */
    SAMPLE_LINEAR,
    /*! Default sampling method.
     */
    SAMPLE_DEFAULT = SAMPLE_LINEAR,
  };
  /*! Constructor.
   *  @param format The desired format of the image.
   *  @param width The desired width of the image. This cannot be zero.
   *  @param height The desired height of the image. This cannot be zero.
   *  @param data The pixel data to initialize the image with, or @c NULL to
   *  initialize it with zeros.
   *  @param pitch The pitch, in bytes, between consecutive scanlines, or zero
   *  if the scanlines are contiguous in memory. If @c data is @c NULL, then
   *  this parameter is ignored.
   *  @param name The name of the image, or the empty string to automatically
   *  generate a name.
   *
   *  @remarks No, you cannot create an empty image object.
   */
  Image(const ImageFormat& format,
        unsigned int width,
        unsigned int height,
        const void* data = NULL,
        size_t pitch = 0,
        const String& name = "");
  /*! Copy constructor.
   */
  Image(const Image& source);
  /*! Fills this image with the specified color.
   *  @param color The color with which to fill.
   *  @return @c true if successful, otherwise @c false.
   *  @remarks Depending on the formats involved, not all channels may affect
   *  the resulting fill color.
   */
  void fill(const ColorRGBA& color);
  /*! Changes the size of this image, resampling its pixel data using
   *  the specified filter method.
   *  @return @c true if successful, otherwise @c false.
   */
  bool resize(unsigned int targetWidth,
              unsigned int targetHeight,
              Method method = SAMPLE_DEFAULT);
  /*! Converts this image to the specified format.
   *  @param targetFormat The format to convert to.
   *  @return @c true if successful, otherwise @c false.
   *  @remarks Depending on the format, not all channels may affect
   *  the resulting image data.
   */
  bool convert(const ImageFormat& targetFormat);
  /*! Sets this image to the specified area of the current image data.
   *  @param area The desired area.
   *  @return @c true if successful, otherwise @c false.
   *  @remarks This method fails if the desired area is completely
   *  outside the current image data.
   *  @remarks If the desired area is partially outside of the current
   *  image data, the area (and the resulting image) is cropped to fit
   *  the current (existing) image data.
   */
  bool crop(const ImageArea& area);
  /*! Flips this image along the x axis.
   */
  void flipHorizontal(void);
  /*! Flips this image along the y axis.
   */
  void flipVertical(void);
  /*! Assignment operator.
   */
  Image& operator = (const Image& source);
  /*! @return The width, in pixels, of this image.
   */
  unsigned int getWidth(void) const;
  /*! @return The height, in pixels, of this image.
   */
  unsigned int getHeight(void) const;
  /*! @return The base address of the pixel data for this image.
   */
  void* getPixels(void);
  /*! @return The base address of the pixel data for this image.
   */
  const void* getPixels(void) const;
  /*! Helper method to calculate the address of the specified pixel.
   *  @param x The x coordinate of the desired pixel.
   *  @param y The y coordinate of the desired pixel.
   *  @return The address of the desired pixel, or @c NULL if the
   *  specified coordinates are outside of the current image data.
   */
  void* getPixel(unsigned int x, unsigned int y);
  /*! Helper method to calculate the address of the specified pixel.
   *  @param x The x coordinate of the desired pixel.
   *  @param y The y coordinate of the desired pixel.
   *  @return The address of the desired pixel, or @c NULL if the
   *  specified coordinates are outside of the current image data.
   */
  const void* getPixel(unsigned int x, unsigned int y) const;
  /*! @return The pixel format of this image.
   */
  const ImageFormat& getFormat(void) const;
  /*! @return The number of dimensions (that differ from 1) in this image.
   */
  unsigned int getDimensionCount(void) const;
  /*! Returns an image containing the specified area of this image.
   *  @param area The desired area of this image.
   */
  Image* getArea(const ImageArea& area);
  /*! Creates an image object from the specified image file.
   *  @param path The path of the image file to use.
   *  @param name The desired name of the image object.
   *  @return The newly created image.
   */
  static bool copyPixels(void* target,
                         size_t targetPitch,
                         const ImageFormat& targetFormat,
                         const ImageArea& targetArea,
                         const void* source,
                         size_t sourcePitch,
                         const ImageFormat& sourceFormat,
                         const ImageArea& sourceArea);
private:
  unsigned int width;
  unsigned int height;
  unsigned int locks;
  ImageFormat format;
  Block data;
};

///////////////////////////////////////////////////////////////////////

class ImageCube
{
public:
  enum Face
  {
    POSITIVE_X,
    NEGATIVE_X,
    POSITIVE_Y,
    NEGATIVE_Y,
    POSITIVE_Z,
    NEGATIVE_Z,
  };
  ImageCube(void);
  void setImage(Face face, Image& image);
  Image& getImage(Face face);
private:
  Ref<Image> images[6];
};

///////////////////////////////////////////////////////////////////////

/*! @brief Image generator interface.
 */
class ImageGenerator
{
public:
  ImageGenerator(void);
  virtual ~ImageGenerator(void);
  const ColorRGBA& getDefaultColor(void) const;
  void setDefaultColor(const ColorRGBA& newColor);
  virtual Image* generate(const ImageFormat& format,
                          unsigned int width,
                          unsigned int height) = 0;
private:
  ColorRGBA defaultColor;
};

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_IMAGE_H*/
///////////////////////////////////////////////////////////////////////
