///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_MATRIX_H
#define MOIRA_MATRIX_H
///////////////////////////////////////////////////////////////////////

#include <stack>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

class Quaternion;

///////////////////////////////////////////////////////////////////////

class Matrix2
{
public:
  Matrix2(void);
  Matrix2(const Vector2& sx, const Vector2& sy);
  void invert(void);
  void transpose(void);
  void normalize(void);
  void concatenate(const Matrix2& matrix);
  void rotateVector(Vector2& vector) const;
  operator float* (void);
  operator const float* (void) const;
  float operator () (unsigned int column, unsigned int row) const;
  float& operator () (unsigned int column, unsigned int row);
  Matrix2 operator + (const Matrix2& matrix) const;
  Matrix2 operator - (const Matrix2& matrix) const;
  Matrix2 operator * (const Matrix2& matrix) const;
  Matrix2& operator += (const Matrix2& matrix);
  Matrix2& operator -= (const Matrix2& matrix);
  Matrix2& operator *= (const Matrix2& matrix);
  bool operator == (const Matrix2& matrix) const;
  bool operator != (const Matrix2& matrix) const;
  void setIdentity(void);
  void set(const Vector2& sx, const Vector2& sy);
  void setRotation(const float angle);
  Vector2 x;
  Vector2 y;
};

///////////////////////////////////////////////////////////////////////

class Matrix3
{
public:
  Matrix3(void);
  Matrix3(const Vector3& sx, const Vector3& sy, const Vector3& sz);
  void invert(void);
  void transpose(void);
  void normalize(void);
  void concatenate(const Matrix3& matrix);
  void rotateVector(Vector3& vector) const;
  operator float* (void);
  operator const float* (void) const;
  float operator () (unsigned int column, unsigned int row) const;
  float& operator () (unsigned int column, unsigned int row);
  Matrix3 operator + (const Matrix3& matrix) const;
  Matrix3 operator - (const Matrix3& matrix) const;
  Matrix3 operator * (const Matrix3& matrix) const;
  Matrix3& operator += (const Matrix3& matrix);
  Matrix3& operator -= (const Matrix3& matrix);
  Matrix3& operator *= (const Matrix3& matrix);
  bool operator == (const Matrix3& matrix) const;
  bool operator != (const Matrix3& matrix) const;
  void setIdentity(void);
  void set(const Vector3& sx, const Vector3& sy, const Vector3& sz);
  void setQuatRotation(const Quaternion& quat);
  void setEulerRotation(const Vector3& angles);
  void setVectorRotation(const Vector3& vector);
  void setAxisRotation(const Vector3& axis, float angle);
  Vector3 x;
  Vector3 y;
  Vector3 z;
};

///////////////////////////////////////////////////////////////////////

class Matrix4
{
public:
  Matrix4(void);
  Matrix4(const Vector4& sx, const Vector4& sy, const Vector4& sz, const Vector4& sw);
  void invert(void);
  void transpose(void);
  void normalize(void);
  void concatenate(const Matrix4& matrix);
  void transformVector(Vector3& vector) const;
  void transformVector(Vector4& vector) const;
  void rotateVector(Vector3& vector) const;
  void translateVector(Vector3& vector) const;
  operator float* (void);
  operator const float* (void) const;
  float operator () (unsigned int column, unsigned int row) const;
  float& operator () (unsigned int column, unsigned int row);
  Matrix4 operator + (const Matrix4& matrix) const;
  Matrix4 operator - (const Matrix4& matrix) const;
  Matrix4 operator * (const Matrix4& matrix) const;
  Matrix4& operator += (const Matrix4& matrix);
  Matrix4& operator -= (const Matrix4& matrix);
  Matrix4& operator *= (const Matrix4& matrix);
  bool operator == (const Matrix4& matrix) const;
  bool operator != (const Matrix4& matrix) const;
  void setIdentity(void);
  void set(const Vector4& sx, const Vector4& sy, const Vector4& sz, const Vector4& sw);
  void setEulerRotation(const Vector3& angles);
  void setQuatRotation(const Quaternion& quat);
  void setVectorRotation(const Vector3& vector);
  void setAxisRotation(const Vector3& axis, float angle);
  void getMatrixRotation(Matrix3& matrix) const;
  void setMatrixRotation(const Matrix3& matrix);
  void getScaling(Vector3& scaling) const;
  void setScaling(const Vector3& scaling);
  void getTranslation(Vector3& vector) const;
  void setTranslation(const Vector3& vector);
  Vector4 x;
  Vector4 y;
  Vector4 z;
  Vector4 w;
};

///////////////////////////////////////////////////////////////////////

template <typename T>
class MatrixStack
{
public:
  inline MatrixStack(void);
  inline void push(const T& matrix);
  inline void pop(void);
  inline bool isEmpty(void) const;
  inline unsigned int getCount(void) const;
  inline const T& getTop(void) const;
  inline const T& getTotal(void) const;
private:
  struct Entry
  {
    T local;
    T total;
  };
  std::stack<Entry> stack;
};

///////////////////////////////////////////////////////////////////////

typedef MatrixStack<Matrix2> MatrixStack2;
typedef MatrixStack<Matrix3> MatrixStack3;
typedef MatrixStack<Matrix4> MatrixStack4;

///////////////////////////////////////////////////////////////////////

template <typename T>
inline MatrixStack<T>::MatrixStack(void)
{
  Entry entry;
  entry.local.setIdentity();
  entry.total.setIdentity();
  stack.push(entry);
}

template <typename T>
inline void MatrixStack<T>::push(const T& matrix)
{
  // TODO: Verify correctness.

  Entry entry;
  entry.local = matrix;
  entry.total = matrix;
  entry.total.concatenate(stack.top().total);
  stack.push(entry);
}

template <typename T>
inline void MatrixStack<T>::pop(void)
{
  if (stack.size() > 1)
    stack.pop();
}

template <typename T>
inline bool MatrixStack<T>::isEmpty(void) const
{
  return stack.size() == 1;
}

template <typename T>
inline unsigned int MatrixStack<T>::getCount(void) const
{
  return stack.size();
}

template <typename T>
inline const T& MatrixStack<T>::getTop(void) const
{
  return stack.top().local;
}

template <typename T>
inline const T& MatrixStack<T>::getTotal(void) const
{
  return stack.top().total;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_MATRIX_H*/
///////////////////////////////////////////////////////////////////////
