///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2006 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_SIGNAL_H
#define MOIRA_SIGNAL_H
///////////////////////////////////////////////////////////////////////

#include <list>

///////////////////////////////////////////////////////////////////////
// NOTE: This file is generated by makesignal.rb. DO NOT EDIT.
///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

class SignalSlot;

template <typename R>
class Signal0;
template <typename R, typename A1>
class Signal1;
template <typename R, typename A1, typename A2>
class Signal2;
template <typename R, typename A1, typename A2, typename A3>
class Signal3;
template <typename R, typename A1, typename A2, typename A3, typename A4>
class Signal4;

////////////////////////////////////////////////////////////////////////

/*! @brief Mixin for automatic slot destruction.
 *
 *  @remarks Inherit from this to automatically be disconnected from all
 *  attached signals on destruction.
 */
class Trackable
{
  friend class SignalSlot;
public:
  virtual ~Trackable(void);
private:
  typedef std::list<SignalSlot*> SlotList;
  SlotList slots;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Abstract base class for signal slots.
 *
 *  @remarks If you wish to keep track of a connection for explicit
 *  disconnection, you usually want to do it by keeping a pointer to a
 *  SignalSlot object.
 */
class SignalSlot
{
public:
  /*! Constructor.
   *  @param[in] object The trackable object this slot connects to, or @c
   *  NULL if it is connected to a global function or a non-trackable object.
   */
  SignalSlot(Trackable* object = NULL);
  /*! Destructor.
   *  If this slot is attached to a trackable object, it will sever the
   *  connection to it.
   */
  virtual ~SignalSlot(void) = 0;
private:
  Trackable* object;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Base class for typed signal slots.
 */
template <typename R>
class SignalSlot0 : public SignalSlot
{
  friend class Signal0<R>;
public:
  /*! Destructor.
   */
  ~SignalSlot0(void);
  /*! Calls the target for this slot.
   */
  virtual R emit(void) = 0;
protected:
  /*! Constructor.
   */
  SignalSlot0(Trackable* object = NULL);
private:
  Signal0<R>* signal;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for global functions.
 */
template <typename R>
class GlobalSlot0 : public SignalSlot0<R>
{
public:
  /*! Function pointer type for this slot.
   */
  typedef R (*Function)(void);
  /*! Constructor.
   */
  inline GlobalSlot0(Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(void);
private:
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for member functions.
 */
template <typename T, typename R>
class MethodSlot0 : public SignalSlot0<R>
{
public:
  /*! Method pointer type for this slot.
   */
  typedef R (T::*Function)(void);
  /*! Constructor.
   */
  inline MethodSlot0(T& object, Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(void);
private:
  T& object;
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal object.
 */
template <typename R>
class Signal0
{
  friend class SignalSlot0<R>;
public:
  /*! Slot base class for this signal.
   */
  typedef SignalSlot0<R> Slot;
  /*! Destructor.
   */
  ~Signal0(void);
  /*! Adds the specified generic slot to this signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified instance and member function to this signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot0<T,R>::Function function)
  { return connect(new MethodSlot0<T,R>(object, function)); }
  /*! Connects the specified global function to this signal.
   */
  inline Slot* connect(typename GlobalSlot0<R>::Function function);
  /*! Calls the targets for all slots in this signal.
   */
  inline void emit(void);
  /*! Calls the targets for all slots in this signal and stores the return values.
   */
  inline void emit(std::list<R>& results);
protected:
  typedef std::list<Slot*> SlotList;
  SlotList slots;
};

///////////////////////////////////////////////////////////////////////

/*! Proxy object for signal objects.
 *  Allows public connections to private signal objects.
 *  Concept taken from gtkmm.
 */
template <typename R>
class SignalProxy0
{
public:
  /*! Signal type.
   */
  typedef Signal0<R> Signal;
  /*! Slot type.
   */
  typedef SignalSlot0<R> Slot;
  /*! Constructor.
   */
  inline SignalProxy0(Signal& signal);
  /*! Connects the specified slot to the proxied signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified global function to the proxied signal.
   */
  inline Slot* connect(typename GlobalSlot0<R>::Function function);
  /*! Connects the specified instance and member function to the proxied signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot0<T,R>::Function function)
  { return signal.connect(new MethodSlot0<T,R>(object, function)); }
private:
  Signal& signal;
};

////////////////////////////////////////////////////////////////////////

template <typename R>
SignalSlot0<R>::~SignalSlot0(void)
{
  if (signal)
    signal->slots.remove(this);
}

template <typename R>
SignalSlot0<R>::SignalSlot0(Trackable* object):
  SignalSlot(object)
{
}

////////////////////////////////////////////////////////////////////////

template <typename R>
inline GlobalSlot0<R>::GlobalSlot0(Function initFunction):
  function(initFunction)
{
}

template <typename R>
inline R GlobalSlot0<R>::emit(void)
{
  return (*function)();
}

////////////////////////////////////////////////////////////////////////

template <typename T, typename R>
inline MethodSlot0<T,R>::MethodSlot0(T& initObject, Function initFunction):
  SignalSlot0<R>(dynamic_cast<Trackable*>(&initObject)),
  object(initObject),
  function(initFunction)
{
}

template <typename T, typename R>
inline R MethodSlot0<T,R>::emit(void)
{
  return (object.*function)();
}

////////////////////////////////////////////////////////////////////////

template <typename R>
Signal0<R>::~Signal0(void)
{
  while (!slots.empty())
    delete slots.front();
}

template <typename R>
inline SignalSlot0<R>* Signal0<R>::connect(Slot* slot)
{
  slots.push_front(slot);
  slot->signal = this;
  return slot;
}

template <typename R>
inline SignalSlot0<R>* Signal0<R>::connect(typename GlobalSlot0<R>::Function function)
{
  return connect(new GlobalSlot0<R>(function));
}

template <typename R>
inline void Signal0<R>::emit(void)
{
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    (*i)->emit();
}

template <typename R>
inline void Signal0<R>::emit(std::list<R>& results)
{
  results.clear();
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    results.push_back((*i)->emit());
}

///////////////////////////////////////////////////////////////////////

template <typename R>
inline SignalProxy0<R>::SignalProxy0(Signal& initSignal):
  signal(initSignal)
{
}

template <typename R>
inline SignalSlot0<R>*
SignalProxy0<R>::connect(Slot* slot)
{
  return signal.connect(slot);
}

template <typename R>
inline SignalSlot0<R>*
SignalProxy0<R>::connect(typename GlobalSlot0<R>::Function function)
{
  return signal.connect(function);
}

////////////////////////////////////////////////////////////////////////

/*! @brief Base class for typed signal slots.
 */
template <typename R, typename A1>
class SignalSlot1 : public SignalSlot
{
  friend class Signal1<R,A1>;
public:
  /*! Destructor.
   */
  ~SignalSlot1(void);
  /*! Calls the target for this slot.
   */
  virtual R emit(A1 a1) = 0;
protected:
  /*! Constructor.
   */
  SignalSlot1(Trackable* object = NULL);
private:
  Signal1<R,A1>* signal;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for global functions.
 */
template <typename R, typename A1>
class GlobalSlot1 : public SignalSlot1<R,A1>
{
public:
  /*! Function pointer type for this slot.
   */
  typedef R (*Function)(A1 a1);
  /*! Constructor.
   */
  inline GlobalSlot1(Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1);
private:
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for member functions.
 */
template <typename T, typename R, typename A1>
class MethodSlot1 : public SignalSlot1<R,A1>
{
public:
  /*! Method pointer type for this slot.
   */
  typedef R (T::*Function)(A1 a1);
  /*! Constructor.
   */
  inline MethodSlot1(T& object, Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1);
private:
  T& object;
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal object.
 */
template <typename R, typename A1>
class Signal1
{
  friend class SignalSlot1<R,A1>;
public:
  /*! Slot base class for this signal.
   */
  typedef SignalSlot1<R,A1> Slot;
  /*! Destructor.
   */
  ~Signal1(void);
  /*! Adds the specified generic slot to this signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified instance and member function to this signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot1<T,R,A1>::Function function)
  { return connect(new MethodSlot1<T,R,A1>(object, function)); }
  /*! Connects the specified global function to this signal.
   */
  inline Slot* connect(typename GlobalSlot1<R,A1>::Function function);
  /*! Calls the targets for all slots in this signal.
   */
  inline void emit(A1 a1);
  /*! Calls the targets for all slots in this signal and stores the return values.
   */
  inline void emit(std::list<R>& results, A1 a1);
protected:
  typedef std::list<Slot*> SlotList;
  SlotList slots;
};

///////////////////////////////////////////////////////////////////////

/*! Proxy object for signal objects.
 *  Allows public connections to private signal objects.
 *  Concept taken from gtkmm.
 */
template <typename R, typename A1>
class SignalProxy1
{
public:
  /*! Signal type.
   */
  typedef Signal1<R,A1> Signal;
  /*! Slot type.
   */
  typedef SignalSlot1<R,A1> Slot;
  /*! Constructor.
   */
  inline SignalProxy1(Signal& signal);
  /*! Connects the specified slot to the proxied signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified global function to the proxied signal.
   */
  inline Slot* connect(typename GlobalSlot1<R,A1>::Function function);
  /*! Connects the specified instance and member function to the proxied signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot1<T,R,A1>::Function function)
  { return signal.connect(new MethodSlot1<T,R,A1>(object, function)); }
private:
  Signal& signal;
};

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1>
SignalSlot1<R,A1>::~SignalSlot1(void)
{
  if (signal)
    signal->slots.remove(this);
}

template <typename R, typename A1>
SignalSlot1<R,A1>::SignalSlot1(Trackable* object):
  SignalSlot(object)
{
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1>
inline GlobalSlot1<R,A1>::GlobalSlot1(Function initFunction):
  function(initFunction)
{
}

template <typename R, typename A1>
inline R GlobalSlot1<R,A1>::emit(A1 a1)
{
  return (*function)(a1);
}

////////////////////////////////////////////////////////////////////////

template <typename T, typename R, typename A1>
inline MethodSlot1<T,R,A1>::MethodSlot1(T& initObject, Function initFunction):
  SignalSlot1<R,A1>(dynamic_cast<Trackable*>(&initObject)),
  object(initObject),
  function(initFunction)
{
}

template <typename T, typename R, typename A1>
inline R MethodSlot1<T,R,A1>::emit(A1 a1)
{
  return (object.*function)(a1);
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1>
Signal1<R,A1>::~Signal1(void)
{
  while (!slots.empty())
    delete slots.front();
}

template <typename R, typename A1>
inline SignalSlot1<R,A1>* Signal1<R,A1>::connect(Slot* slot)
{
  slots.push_front(slot);
  slot->signal = this;
  return slot;
}

template <typename R, typename A1>
inline SignalSlot1<R,A1>* Signal1<R,A1>::connect(typename GlobalSlot1<R,A1>::Function function)
{
  return connect(new GlobalSlot1<R,A1>(function));
}

template <typename R, typename A1>
inline void Signal1<R,A1>::emit(A1 a1)
{
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    (*i)->emit(a1);
}

template <typename R, typename A1>
inline void Signal1<R,A1>::emit(std::list<R>& results, A1 a1)
{
  results.clear();
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    results.push_back((*i)->emit(a1));
}

///////////////////////////////////////////////////////////////////////

template <typename R, typename A1>
inline SignalProxy1<R,A1>::SignalProxy1(Signal& initSignal):
  signal(initSignal)
{
}

template <typename R, typename A1>
inline SignalSlot1<R,A1>*
SignalProxy1<R,A1>::connect(Slot* slot)
{
  return signal.connect(slot);
}

template <typename R, typename A1>
inline SignalSlot1<R,A1>*
SignalProxy1<R,A1>::connect(typename GlobalSlot1<R,A1>::Function function)
{
  return signal.connect(function);
}

////////////////////////////////////////////////////////////////////////

/*! @brief Base class for typed signal slots.
 */
template <typename R, typename A1, typename A2>
class SignalSlot2 : public SignalSlot
{
  friend class Signal2<R,A1,A2>;
public:
  /*! Destructor.
   */
  ~SignalSlot2(void);
  /*! Calls the target for this slot.
   */
  virtual R emit(A1 a1, A2 a2) = 0;
protected:
  /*! Constructor.
   */
  SignalSlot2(Trackable* object = NULL);
private:
  Signal2<R,A1,A2>* signal;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for global functions.
 */
template <typename R, typename A1, typename A2>
class GlobalSlot2 : public SignalSlot2<R,A1,A2>
{
public:
  /*! Function pointer type for this slot.
   */
  typedef R (*Function)(A1 a1, A2 a2);
  /*! Constructor.
   */
  inline GlobalSlot2(Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2);
private:
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for member functions.
 */
template <typename T, typename R, typename A1, typename A2>
class MethodSlot2 : public SignalSlot2<R,A1,A2>
{
public:
  /*! Method pointer type for this slot.
   */
  typedef R (T::*Function)(A1 a1, A2 a2);
  /*! Constructor.
   */
  inline MethodSlot2(T& object, Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2);
private:
  T& object;
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal object.
 */
template <typename R, typename A1, typename A2>
class Signal2
{
  friend class SignalSlot2<R,A1,A2>;
public:
  /*! Slot base class for this signal.
   */
  typedef SignalSlot2<R,A1,A2> Slot;
  /*! Destructor.
   */
  ~Signal2(void);
  /*! Adds the specified generic slot to this signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified instance and member function to this signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot2<T,R,A1,A2>::Function function)
  { return connect(new MethodSlot2<T,R,A1,A2>(object, function)); }
  /*! Connects the specified global function to this signal.
   */
  inline Slot* connect(typename GlobalSlot2<R,A1,A2>::Function function);
  /*! Calls the targets for all slots in this signal.
   */
  inline void emit(A1 a1, A2 a2);
  /*! Calls the targets for all slots in this signal and stores the return values.
   */
  inline void emit(std::list<R>& results, A1 a1, A2 a2);
protected:
  typedef std::list<Slot*> SlotList;
  SlotList slots;
};

///////////////////////////////////////////////////////////////////////

/*! Proxy object for signal objects.
 *  Allows public connections to private signal objects.
 *  Concept taken from gtkmm.
 */
template <typename R, typename A1, typename A2>
class SignalProxy2
{
public:
  /*! Signal type.
   */
  typedef Signal2<R,A1,A2> Signal;
  /*! Slot type.
   */
  typedef SignalSlot2<R,A1,A2> Slot;
  /*! Constructor.
   */
  inline SignalProxy2(Signal& signal);
  /*! Connects the specified slot to the proxied signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified global function to the proxied signal.
   */
  inline Slot* connect(typename GlobalSlot2<R,A1,A2>::Function function);
  /*! Connects the specified instance and member function to the proxied signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot2<T,R,A1,A2>::Function function)
  { return signal.connect(new MethodSlot2<T,R,A1,A2>(object, function)); }
private:
  Signal& signal;
};

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2>
SignalSlot2<R,A1,A2>::~SignalSlot2(void)
{
  if (signal)
    signal->slots.remove(this);
}

template <typename R, typename A1, typename A2>
SignalSlot2<R,A1,A2>::SignalSlot2(Trackable* object):
  SignalSlot(object)
{
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2>
inline GlobalSlot2<R,A1,A2>::GlobalSlot2(Function initFunction):
  function(initFunction)
{
}

template <typename R, typename A1, typename A2>
inline R GlobalSlot2<R,A1,A2>::emit(A1 a1, A2 a2)
{
  return (*function)(a1, a2);
}

////////////////////////////////////////////////////////////////////////

template <typename T, typename R, typename A1, typename A2>
inline MethodSlot2<T,R,A1,A2>::MethodSlot2(T& initObject, Function initFunction):
  SignalSlot2<R,A1,A2>(dynamic_cast<Trackable*>(&initObject)),
  object(initObject),
  function(initFunction)
{
}

template <typename T, typename R, typename A1, typename A2>
inline R MethodSlot2<T,R,A1,A2>::emit(A1 a1, A2 a2)
{
  return (object.*function)(a1, a2);
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2>
Signal2<R,A1,A2>::~Signal2(void)
{
  while (!slots.empty())
    delete slots.front();
}

template <typename R, typename A1, typename A2>
inline SignalSlot2<R,A1,A2>* Signal2<R,A1,A2>::connect(Slot* slot)
{
  slots.push_front(slot);
  slot->signal = this;
  return slot;
}

template <typename R, typename A1, typename A2>
inline SignalSlot2<R,A1,A2>* Signal2<R,A1,A2>::connect(typename GlobalSlot2<R,A1,A2>::Function function)
{
  return connect(new GlobalSlot2<R,A1,A2>(function));
}

template <typename R, typename A1, typename A2>
inline void Signal2<R,A1,A2>::emit(A1 a1, A2 a2)
{
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    (*i)->emit(a1, a2);
}

template <typename R, typename A1, typename A2>
inline void Signal2<R,A1,A2>::emit(std::list<R>& results, A1 a1, A2 a2)
{
  results.clear();
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    results.push_back((*i)->emit(a1, a2));
}

///////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2>
inline SignalProxy2<R,A1,A2>::SignalProxy2(Signal& initSignal):
  signal(initSignal)
{
}

template <typename R, typename A1, typename A2>
inline SignalSlot2<R,A1,A2>*
SignalProxy2<R,A1,A2>::connect(Slot* slot)
{
  return signal.connect(slot);
}

template <typename R, typename A1, typename A2>
inline SignalSlot2<R,A1,A2>*
SignalProxy2<R,A1,A2>::connect(typename GlobalSlot2<R,A1,A2>::Function function)
{
  return signal.connect(function);
}

////////////////////////////////////////////////////////////////////////

/*! @brief Base class for typed signal slots.
 */
template <typename R, typename A1, typename A2, typename A3>
class SignalSlot3 : public SignalSlot
{
  friend class Signal3<R,A1,A2,A3>;
public:
  /*! Destructor.
   */
  ~SignalSlot3(void);
  /*! Calls the target for this slot.
   */
  virtual R emit(A1 a1, A2 a2, A3 a3) = 0;
protected:
  /*! Constructor.
   */
  SignalSlot3(Trackable* object = NULL);
private:
  Signal3<R,A1,A2,A3>* signal;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for global functions.
 */
template <typename R, typename A1, typename A2, typename A3>
class GlobalSlot3 : public SignalSlot3<R,A1,A2,A3>
{
public:
  /*! Function pointer type for this slot.
   */
  typedef R (*Function)(A1 a1, A2 a2, A3 a3);
  /*! Constructor.
   */
  inline GlobalSlot3(Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2, A3 a3);
private:
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for member functions.
 */
template <typename T, typename R, typename A1, typename A2, typename A3>
class MethodSlot3 : public SignalSlot3<R,A1,A2,A3>
{
public:
  /*! Method pointer type for this slot.
   */
  typedef R (T::*Function)(A1 a1, A2 a2, A3 a3);
  /*! Constructor.
   */
  inline MethodSlot3(T& object, Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2, A3 a3);
private:
  T& object;
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal object.
 */
template <typename R, typename A1, typename A2, typename A3>
class Signal3
{
  friend class SignalSlot3<R,A1,A2,A3>;
public:
  /*! Slot base class for this signal.
   */
  typedef SignalSlot3<R,A1,A2,A3> Slot;
  /*! Destructor.
   */
  ~Signal3(void);
  /*! Adds the specified generic slot to this signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified instance and member function to this signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot3<T,R,A1,A2,A3>::Function function)
  { return connect(new MethodSlot3<T,R,A1,A2,A3>(object, function)); }
  /*! Connects the specified global function to this signal.
   */
  inline Slot* connect(typename GlobalSlot3<R,A1,A2,A3>::Function function);
  /*! Calls the targets for all slots in this signal.
   */
  inline void emit(A1 a1, A2 a2, A3 a3);
  /*! Calls the targets for all slots in this signal and stores the return values.
   */
  inline void emit(std::list<R>& results, A1 a1, A2 a2, A3 a3);
protected:
  typedef std::list<Slot*> SlotList;
  SlotList slots;
};

///////////////////////////////////////////////////////////////////////

/*! Proxy object for signal objects.
 *  Allows public connections to private signal objects.
 *  Concept taken from gtkmm.
 */
template <typename R, typename A1, typename A2, typename A3>
class SignalProxy3
{
public:
  /*! Signal type.
   */
  typedef Signal3<R,A1,A2,A3> Signal;
  /*! Slot type.
   */
  typedef SignalSlot3<R,A1,A2,A3> Slot;
  /*! Constructor.
   */
  inline SignalProxy3(Signal& signal);
  /*! Connects the specified slot to the proxied signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified global function to the proxied signal.
   */
  inline Slot* connect(typename GlobalSlot3<R,A1,A2,A3>::Function function);
  /*! Connects the specified instance and member function to the proxied signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot3<T,R,A1,A2,A3>::Function function)
  { return signal.connect(new MethodSlot3<T,R,A1,A2,A3>(object, function)); }
private:
  Signal& signal;
};

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3>
SignalSlot3<R,A1,A2,A3>::~SignalSlot3(void)
{
  if (signal)
    signal->slots.remove(this);
}

template <typename R, typename A1, typename A2, typename A3>
SignalSlot3<R,A1,A2,A3>::SignalSlot3(Trackable* object):
  SignalSlot(object)
{
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3>
inline GlobalSlot3<R,A1,A2,A3>::GlobalSlot3(Function initFunction):
  function(initFunction)
{
}

template <typename R, typename A1, typename A2, typename A3>
inline R GlobalSlot3<R,A1,A2,A3>::emit(A1 a1, A2 a2, A3 a3)
{
  return (*function)(a1, a2, a3);
}

////////////////////////////////////////////////////////////////////////

template <typename T, typename R, typename A1, typename A2, typename A3>
inline MethodSlot3<T,R,A1,A2,A3>::MethodSlot3(T& initObject, Function initFunction):
  SignalSlot3<R,A1,A2,A3>(dynamic_cast<Trackable*>(&initObject)),
  object(initObject),
  function(initFunction)
{
}

template <typename T, typename R, typename A1, typename A2, typename A3>
inline R MethodSlot3<T,R,A1,A2,A3>::emit(A1 a1, A2 a2, A3 a3)
{
  return (object.*function)(a1, a2, a3);
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3>
Signal3<R,A1,A2,A3>::~Signal3(void)
{
  while (!slots.empty())
    delete slots.front();
}

template <typename R, typename A1, typename A2, typename A3>
inline SignalSlot3<R,A1,A2,A3>* Signal3<R,A1,A2,A3>::connect(Slot* slot)
{
  slots.push_front(slot);
  slot->signal = this;
  return slot;
}

template <typename R, typename A1, typename A2, typename A3>
inline SignalSlot3<R,A1,A2,A3>* Signal3<R,A1,A2,A3>::connect(typename GlobalSlot3<R,A1,A2,A3>::Function function)
{
  return connect(new GlobalSlot3<R,A1,A2,A3>(function));
}

template <typename R, typename A1, typename A2, typename A3>
inline void Signal3<R,A1,A2,A3>::emit(A1 a1, A2 a2, A3 a3)
{
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    (*i)->emit(a1, a2, a3);
}

template <typename R, typename A1, typename A2, typename A3>
inline void Signal3<R,A1,A2,A3>::emit(std::list<R>& results, A1 a1, A2 a2, A3 a3)
{
  results.clear();
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    results.push_back((*i)->emit(a1, a2, a3));
}

///////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3>
inline SignalProxy3<R,A1,A2,A3>::SignalProxy3(Signal& initSignal):
  signal(initSignal)
{
}

template <typename R, typename A1, typename A2, typename A3>
inline SignalSlot3<R,A1,A2,A3>*
SignalProxy3<R,A1,A2,A3>::connect(Slot* slot)
{
  return signal.connect(slot);
}

template <typename R, typename A1, typename A2, typename A3>
inline SignalSlot3<R,A1,A2,A3>*
SignalProxy3<R,A1,A2,A3>::connect(typename GlobalSlot3<R,A1,A2,A3>::Function function)
{
  return signal.connect(function);
}

////////////////////////////////////////////////////////////////////////

/*! @brief Base class for typed signal slots.
 */
template <typename R, typename A1, typename A2, typename A3, typename A4>
class SignalSlot4 : public SignalSlot
{
  friend class Signal4<R,A1,A2,A3,A4>;
public:
  /*! Destructor.
   */
  ~SignalSlot4(void);
  /*! Calls the target for this slot.
   */
  virtual R emit(A1 a1, A2 a2, A3 a3, A4 a4) = 0;
protected:
  /*! Constructor.
   */
  SignalSlot4(Trackable* object = NULL);
private:
  Signal4<R,A1,A2,A3,A4>* signal;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for global functions.
 */
template <typename R, typename A1, typename A2, typename A3, typename A4>
class GlobalSlot4 : public SignalSlot4<R,A1,A2,A3,A4>
{
public:
  /*! Function pointer type for this slot.
   */
  typedef R (*Function)(A1 a1, A2 a2, A3 a3, A4 a4);
  /*! Constructor.
   */
  inline GlobalSlot4(Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2, A3 a3, A4 a4);
private:
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal slot for member functions.
 */
template <typename T, typename R, typename A1, typename A2, typename A3, typename A4>
class MethodSlot4 : public SignalSlot4<R,A1,A2,A3,A4>
{
public:
  /*! Method pointer type for this slot.
   */
  typedef R (T::*Function)(A1 a1, A2 a2, A3 a3, A4 a4);
  /*! Constructor.
   */
  inline MethodSlot4(T& object, Function function);
  /*! Calls the target for this slot.
   */
  inline R emit(A1 a1, A2 a2, A3 a3, A4 a4);
private:
  T& object;
  Function function;
};

////////////////////////////////////////////////////////////////////////

/*! @brief Typed signal object.
 */
template <typename R, typename A1, typename A2, typename A3, typename A4>
class Signal4
{
  friend class SignalSlot4<R,A1,A2,A3,A4>;
public:
  /*! Slot base class for this signal.
   */
  typedef SignalSlot4<R,A1,A2,A3,A4> Slot;
  /*! Destructor.
   */
  ~Signal4(void);
  /*! Adds the specified generic slot to this signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified instance and member function to this signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot4<T,R,A1,A2,A3,A4>::Function function)
  { return connect(new MethodSlot4<T,R,A1,A2,A3,A4>(object, function)); }
  /*! Connects the specified global function to this signal.
   */
  inline Slot* connect(typename GlobalSlot4<R,A1,A2,A3,A4>::Function function);
  /*! Calls the targets for all slots in this signal.
   */
  inline void emit(A1 a1, A2 a2, A3 a3, A4 a4);
  /*! Calls the targets for all slots in this signal and stores the return values.
   */
  inline void emit(std::list<R>& results, A1 a1, A2 a2, A3 a3, A4 a4);
protected:
  typedef std::list<Slot*> SlotList;
  SlotList slots;
};

///////////////////////////////////////////////////////////////////////

/*! Proxy object for signal objects.
 *  Allows public connections to private signal objects.
 *  Concept taken from gtkmm.
 */
template <typename R, typename A1, typename A2, typename A3, typename A4>
class SignalProxy4
{
public:
  /*! Signal type.
   */
  typedef Signal4<R,A1,A2,A3,A4> Signal;
  /*! Slot type.
   */
  typedef SignalSlot4<R,A1,A2,A3,A4> Slot;
  /*! Constructor.
   */
  inline SignalProxy4(Signal& signal);
  /*! Connects the specified slot to the proxied signal.
   */
  inline Slot* connect(Slot* slot);
  /*! Connects the specified global function to the proxied signal.
   */
  inline Slot* connect(typename GlobalSlot4<R,A1,A2,A3,A4>::Function function);
  /*! Connects the specified instance and member function to the proxied signal.
   */
  template <typename T>
  inline Slot* connect(T& object, typename MethodSlot4<T,R,A1,A2,A3,A4>::Function function)
  { return signal.connect(new MethodSlot4<T,R,A1,A2,A3,A4>(object, function)); }
private:
  Signal& signal;
};

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3, typename A4>
SignalSlot4<R,A1,A2,A3,A4>::~SignalSlot4(void)
{
  if (signal)
    signal->slots.remove(this);
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
SignalSlot4<R,A1,A2,A3,A4>::SignalSlot4(Trackable* object):
  SignalSlot(object)
{
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline GlobalSlot4<R,A1,A2,A3,A4>::GlobalSlot4(Function initFunction):
  function(initFunction)
{
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline R GlobalSlot4<R,A1,A2,A3,A4>::emit(A1 a1, A2 a2, A3 a3, A4 a4)
{
  return (*function)(a1, a2, a3, a4);
}

////////////////////////////////////////////////////////////////////////

template <typename T, typename R, typename A1, typename A2, typename A3, typename A4>
inline MethodSlot4<T,R,A1,A2,A3,A4>::MethodSlot4(T& initObject, Function initFunction):
  SignalSlot4<R,A1,A2,A3,A4>(dynamic_cast<Trackable*>(&initObject)),
  object(initObject),
  function(initFunction)
{
}

template <typename T, typename R, typename A1, typename A2, typename A3, typename A4>
inline R MethodSlot4<T,R,A1,A2,A3,A4>::emit(A1 a1, A2 a2, A3 a3, A4 a4)
{
  return (object.*function)(a1, a2, a3, a4);
}

////////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3, typename A4>
Signal4<R,A1,A2,A3,A4>::~Signal4(void)
{
  while (!slots.empty())
    delete slots.front();
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline SignalSlot4<R,A1,A2,A3,A4>* Signal4<R,A1,A2,A3,A4>::connect(Slot* slot)
{
  slots.push_front(slot);
  slot->signal = this;
  return slot;
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline SignalSlot4<R,A1,A2,A3,A4>* Signal4<R,A1,A2,A3,A4>::connect(typename GlobalSlot4<R,A1,A2,A3,A4>::Function function)
{
  return connect(new GlobalSlot4<R,A1,A2,A3,A4>(function));
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline void Signal4<R,A1,A2,A3,A4>::emit(A1 a1, A2 a2, A3 a3, A4 a4)
{
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    (*i)->emit(a1, a2, a3, a4);
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline void Signal4<R,A1,A2,A3,A4>::emit(std::list<R>& results, A1 a1, A2 a2, A3 a3, A4 a4)
{
  results.clear();
  for (typename SlotList::iterator i = slots.begin();  i != slots.end();  i++)
    results.push_back((*i)->emit(a1, a2, a3, a4));
}

///////////////////////////////////////////////////////////////////////

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline SignalProxy4<R,A1,A2,A3,A4>::SignalProxy4(Signal& initSignal):
  signal(initSignal)
{
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline SignalSlot4<R,A1,A2,A3,A4>*
SignalProxy4<R,A1,A2,A3,A4>::connect(Slot* slot)
{
  return signal.connect(slot);
}

template <typename R, typename A1, typename A2, typename A3, typename A4>
inline SignalSlot4<R,A1,A2,A3,A4>*
SignalProxy4<R,A1,A2,A3,A4>::connect(typename GlobalSlot4<R,A1,A2,A3,A4>::Function function)
{
  return signal.connect(function);
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_SIGNAL_H*/
///////////////////////////////////////////////////////////////////////
