///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Vector.h>
#include <moira/Matrix.h>
#include <moira/Quaternion.h>
#include <moira/Transform.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

Transform2::Transform2(void)
{
  setIdentity();
}

Transform2::Transform2(const Vector2& initPosition, float initAngle):
  position(initPosition),
  angle(initAngle)
{
}

void Transform2::invert(void)
{
  angle = -angle;
  position.negate();
  rotateVector(position);
}

void Transform2::concatenate(const Transform2& parent)
{
  parent.rotateVector(position);
  position += parent.position;
  angle += parent.angle;
}

void Transform2::rotateVector(Vector2& vector) const
{
  const float sina = sinf(angle);
  const float cosa = cosf(angle);

  Vector2 result;

  result.x = vector.x * cosa - vector.y * sina;
  result.y = vector.x * sina + vector.y * cosa;
  vector = result;
}

void Transform2::translateVector(Vector2& vector) const
{
  vector += position;
}

void Transform2::transformVector(Vector2& vector) const
{
  rotateVector(vector);
  vector += position;
}

Transform2::operator Matrix3 (void) const
{
  const float sina = sinf(angle);
  const float cosa = cosf(angle);

  Matrix3 result;
  result.x.x = cosa;
  result.x.y = sina;
  result.y.x = -sina;
  result.y.y = cosa;
  result.z.x = position.x;
  result.z.y = position.y;

  return result;
}

Transform2 Transform2::operator * (const Transform2& parent) const
{
  Transform2 result(*this);
  result.concatenate(parent);
  return result;
}

Transform2& Transform2::operator *= (const Transform2& parent)
{
  concatenate(parent);
  return *this;
}

void Transform2::setIdentity(void)
{
  position.set(0.f, 0.f);
  angle = 0.f;
}

void Transform2::set(const Vector2& newPosition, float newAngle)
{
  position = newPosition;
  angle = newAngle;
}

///////////////////////////////////////////////////////////////////////

Transform3::Transform3(void)
{
  setIdentity();
}

Transform3::Transform3(const Vector3& initPosition, const Quaternion& initRotation):
  position(initPosition),
  rotation(initRotation)
{
}

void Transform3::invert(void)
{
  rotation.invert();
  position.negate();
  rotation.rotateVector(position);
}

void Transform3::concatenate(const Transform3& parent)
{
  parent.rotateVector(position);
  position += parent.position;
  rotation = parent.rotation * rotation;
}

void Transform3::rotateVector(Vector3& vector) const
{
  rotation.rotateVector(vector);
}

void Transform3::translateVector(Vector3& vector) const
{
  vector += position;
}

void Transform3::transformVector(Vector3& vector) const
{
  rotation.rotateVector(vector);
  vector += position;
}

Transform3::operator Matrix4 (void) const
{
  Matrix4 result;
  result.setIdentity();
  result.setQuatRotation(rotation);
  result.setTranslation(position);
  return result;
}

Transform3 Transform3::operator * (const Transform3& parent) const
{
  Transform3 result(*this);
  result.concatenate(parent);
  return result;
}

Transform3& Transform3::operator *= (const Transform3& parent)
{
  concatenate(parent);
  return *this;
}

void Transform3::setIdentity(void)
{
  rotation.set(1.f, 0.f, 0.f, 0.f);
  position.set(0.f, 0.f, 0.f);
}

void Transform3::set(const Vector3& newPosition, const Quaternion& newRotation)
{
  position = newPosition;
  rotation = newRotation;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
