// this is the function where all the server communication is done.

function Communication()
{
    var self = this;

    this._minTimeBetweenUpdates = 100;
    this._lastUpdateTime = new Date();
    
    this._nextMessageId = 1;
    /*
     * Received commands are buffered here and synchronized to game state by syncGameState().
     * This is always implicitly ordered by the command's message ID,
     * but not necessarily time.
     */
    this._commandBuffer = [];
    this._latestTimeLimit = null;

    /*
     * Applies one buffered command if there is one and the current time limit
     * is set to the time of that buffered command.
     * 
     * Sets the time limit to the next buffered command or the latest
     * lockstep time limit sent by the server.
     * 
     * Only commands timed for before the new time limit will be considered.
     */
    this.applyBufferedCommand = function()
    {
        if (this._latestTimeLimit === null)
            return; // Nothing received yet
        
        if (this._commandBuffer.length > 0) {
            var command = this._commandBuffer[0];
            assert(command.time >= time.getTimeLimit(), 'Got command ' + command.messageId + ' for the past (' + command.time + ') at ' + actions.getTime() + '/' + time.getTime() + '|' + time.getTimeLimit());
            
            if (command.time == time.getTimeLimit()) {
                debugLog('Applying command ' + command.messageId + ' for ' + command.time + ' at ' + actions.getTime() + '/' + time.getTime() + '|' + time.getTimeLimit());
                game.commands.handleCommand(command);
                this._commandBuffer.shift();
            }
        }
        
        
        /*
         * The new time limit is the minimum of _latestTimeLimit and
         * the time of the next command, regardless of whether a command
         * was just executed.
         */
        var newTimeLimit = this._latestTimeLimit; // Default
        
        if (this._commandBuffer.length > 0) {
            var nextCommand = this._commandBuffer[0];
            assert(nextCommand.time >= time.getTimeLimit(), 'Time limit of next command in the past!');
            if (nextCommand.time < this._latestTimeLimit)
                newTimeLimit = nextCommand.time;
        }
        
        time.setTimeLimit(newTimeLimit);
    }
    
    this.update = function()
    {
        if (settings.masterKey)
            this.advanceLockstepCommand();
        var msg = this.messageToSend;
        this.messageToSend = "";
        if (settings.singlePlayer)
        {
            // convert messge to server message, to emulate the server.
            setTimeout(function () {
                    var convertedMessage = (time.getTime()+1000)+":";
                    var messageId = self._nextMessageId;
                    var commandArray = msg.split("|");
                    for (var c in commandArray)
                    {
                        if (commandArray[c])
                        {
                            convertedMessage += 
                            messageId+"&"+
                            (time.getTime()+1000)+"&"+
                            commandArray[c]+"|";
                            messageId ++;
                        }
                    }
                    // callback.
                    self.serverCallback(convertedMessage); 
            }, 500);
        }
        else
        {
            // multiplayer
            var messageId = this._nextMessageId;

            var req = {
                'C': settings.clientId,
                'RT': 'STEP',
                'RM': (messageId - 1) + "\n" + Math.round(ping.getPing()) + "\n" + msg
            }
            var retries = 0;

            function sendRequest() {
                new Ajax(settings.serverUrl,
                         function(response) { self.serverCallback(response); },
                         'POST',
                         req,
                         function() {
                             debugLog('STEP call failed', true);
                             if (retries <= 10) {
                                 ++retries;
                                 debugLog('Retry ' + retries + '/' + 10, true);
                                 sendRequest();
                             } else {
                                 alert('Connection to server lost');
                             }
                         });
            }
            sendRequest();
        }
        
        this._lastUpdateTime = new Date();
    }
    
    // add command to a string that will be sent on the next update.
    this.enqueueCommand = function(id, cmd, flags)
    {
        cmd = id+"&"+cmd+"&"+flags+"|";
        this.messageToSend += cmd;
    }
    
    this.step = 0;
    
    // Called with the server's response to a STEP request
    this.serverCallback = function(serverResponse)
    {
        if (serverResponse != undefined)
        {
            var splittedResp = serverResponse.split(":");
            this._latestTimeLimit = parseInt(splittedResp[0]);
            this.bufferCommands(splittedResp[1]);
        }

        //var waitTime = (new Date() - this._lastUpdateTime) + this._minTimeBetweenUpdates;

        //if (waitTime > 0) {
            //setTimeout(function() { self.update(); }, waitTime);
        //} else {
            this.update();
        //}
    }
    
    // unserialize the commands in saved file
    this.bufferCommands = function(serial)
    {
        if (!serial)
            return;
        
        var commands = serial.split("|");
        for (var i = 0; i < commands.length; ++i)
        {
            if (commands[i])
            {
                var parts = commands[i].split("&");
                // 0 = startTime, 1 = object id, 2 = command 3 = flags/parameters
                var messageId = parseInt(parts[0]);
                if (messageId != this._nextMessageId) {
                    debug("Warning: message id was: " + messageId + ", expected: " + this._nextMessageId);
                    debug("Duplicate server response?");
                    return;
                }
                this._nextMessageId++;
                
                var time = parseInt(parts[1]);
                var objectId = parseInt(parts[2]);
                var command = parts[3];
                var parameters = parts[4];
                
                this._commandBuffer.push({
                    messageId: messageId,
                    time: time,
                    objectId: objectId,
                    command: command,
                    parameters: parameters
                });
            }
        }
    }
    // id&time&objectid
    this.messageToSend = "";


    // Handle master client tasks
    if (settings.masterKey) {
        // Have master client advance the lockstep end of time limit
        //var lockstepAdvanceInterval = 500;
        var lockstepAdvanceAmount = 1000;

        this.advanceLockstepCommand = function() {
                self.enqueueCommand(settings.masterKey,"advancelockstep",
                    (time.getTime() + lockstepAdvanceAmount));
        };
        
        // Checksum game state periodically to debug out of sync errors.
        if (settings.outOfSyncChecks) {
            var checksumInterval = 5000;
            var checksummer = setInterval(function() {
                self.enqueueCommand(0, "checksum");
            }, checksumInterval);
        }
    }
    this.update();
}
