/*
 * Game simulation time.
 *
 * This object stores the current simulation time relative to
 * the start time of the game (called "game time").
 * The game time can be limited to a certain value, as needs to be done
 * in lockstep synchronization.
 * 
 * Methods:
 * - getGameTime() Returns the current game time,
 *                i.e. time relative to the game start time.
 *                This returns a constant value until updateToRealTime()
 *                is called.
 * - setGameTimeLimt() Sets a limit to the relative game time.
 *                     The limit must not be lower than any previous limit.
 *                     The initial limit is 0.
 * - updateToRealTime() Increments game time to match the elapsed real time.
 *                      If the game time would go past the game time limit then
 *                      the game time will be paused and any "overflowing" time
 *                      will not count towards further game time.
 *
 * Note: updateToRealTime() should be called quite often.
 *       Consider the following scenario:
 *       - The game is paused because the game time limit has been reached.
 *       - updateToRealTime() has not been called for, say, one second.
 *       - setGameTimeLimit() extends the time limit by, say, five seconds.
 *       - updateToRealTime() is called and the game time jumps ahead by
 *         one second, which will probably cause it to be poorly synchronized
 *         with other players.
 */
function GameTime()
{
    this._startTime = new Date();
    this._currentAbsTime = this._startTime;
    this._totalWaitTime = 0; // The time we've had to wait because of _gameTimeLimit
    this._currentGameTime = 0;
    this._gameTimeLimit = 0;

    this.setTimeLimit = function(newGameTimeLimit) {
        assert(newGameTimeLimit >= this._gameTimeLimit, "New time limit is less than current one");
        this._gameTimeLimit = newGameTimeLimit;
    }
    
    this.getTimeLimit = function() {
        return this._gameTimeLimit;
    }

    this.update = function() {
        var oldAbsTime = this._currentAbsTime;
        var newAbsTime = new Date();

        var increment = newAbsTime - oldAbsTime;

        var newGameTime = this._currentGameTime + increment;
        if (newGameTime > this._gameTimeLimit) {
            this._totalWaitTime += newGameTime - this._gameTimeLimit;
            this._currentGameTime = this._gameTimeLimit;
//debug("TIME LIMIT REACHED. TOTAL WAIT TIME: " + this._totalWaitTime);
        } else {
            this._currentGameTime = newGameTime;
        }
        
        this.showTime();
        this._currentAbsTime = newAbsTime;
    }

    this.getTime = function() {
        return this._currentGameTime;
    }
    
    this.showTime = function()
    {
        document.getElementById("ping").innerHTML = this._currentGameTime;
    }

}

