#include <stdio.h>
#include <string.h>

#include "external/glfw/include/GL/glfw.h"
#include "external/pnglite/pnglite.h"
#include "external/portaudio/include/portaudio.h"

extern "C"
{
	#include "external/wavpack/wavpack.h"
}

#include "base/system.h"
#include "base/vmath.hpp"

int gfx_fullscreen = 1;
int gfx_width = 640;
int gfx_height = 480;
int gfx_fsaa = 0;
const float gfx_aspect = gfx_width/(float)gfx_height;

float last_demo_time = 0;
float demo_time = 0;
float frame_time = 0;

GLuint texture_overlay = -1;
GLuint texture_blob = -1;
GLuint texture_nameplate = -1;
GLuint texture_debug_font = -1;
GLuint texture_endscreen = -1;

GLuint texture_plates[20] = {-1};
int plate_offset = 0;

GLuint texture_opening[4] = {-1};

const int audio_bpm = 114.0f;
const float audio_bps = audio_bpm/60.0f;
unsigned long audio_num_samples = 0;


// effect parameters
float local_time = 0;
float local_amount = 0;
float local_duration = 0;
float local_timeleft = 0;

// triggers
float trigger_kick = 0;
float trigger_bass = 0;
float trigger_blipp[10] = {0};

float trigger_blipp_high = 0;
float trigger_blipp_low = 0;

float times_kick[] =
{8.42f, 8.42f, 12.63f, 13.68f, 14.62f, 15.00f, 15.79f, 16.84f, 17.89f, 18.83f, 19.21f, 20.00f, 21.05f, 22.11f, 23.04f, 
23.42f, 24.21f, 25.26f, 26.32f, 27.25f, 27.63f, 28.42f, 29.47f, 30.53f, 31.46f, 31.84f, 32.63f, 33.68f, 34.74f, 35.67f, 
36.05f, 36.84f, 37.89f, 38.95f, 39.88f, 40.26f, 40.94f, 42.11f, 43.16f, 44.09f, 44.47f, 46.32f, 47.37f, 48.42f, 48.68f, 
49.47f, 50.53f, 51.58f, 52.63f, 52.89f, 55.79f, 56.73f, 57.11f, 57.89f, 58.95f, 60.00f, 60.94f, 61.32f, 62.25f, 63.05f, -1.0f};

float times_bass[] = 
{15.81f, 15.89f, 15.94f, 16.08f, 16.17f, 16.20f, 16.24f, 16.28f, 16.30f, 16.32f, 16.37f, 16.43f, 16.46f, 16.50f, 16.53f, 
16.56f, 16.60f, 16.62f, 16.67f, 16.84f, 17.63f, 17.89f, 18.68f, 18.83f, 19.09f, 19.21f, 19.74f, 20.00f, 21.05f, 21.84f, 
22.11f, 22.89f, 23.04f, 23.30f, 23.42f, 23.83f, 23.95f, 24.21f, 24.47f, 24.74f, 24.88f, 25.00f, 25.26f, 25.53f, 25.67f, 
25.79f, 25.94f, 26.05f, 26.20f, 26.32f, 26.73f, 26.99f, 27.25f, 27.52f, 27.63f, 27.78f, 28.04f, 28.30f, 28.42f, 28.68f, 
28.95f, 29.09f, 29.21f, 29.47f, 29.74f, 29.88f, 30.00f, 30.15f, 30.26f, 30.41f, 30.53f, 30.94f, 31.20f, 31.46f, 31.73f, 
31.84f, 31.99f, 32.25f, 32.52f, 32.63f, 32.89f, 33.16f, 33.30f, 33.42f, 33.68f, 34.47f, 34.74f, 35.26f, 35.67f, 35.94f, 
36.46f, 36.58f, 36.73f, 36.84f, 36.92f, 36.99f, 37.11f, 37.26f, 37.38f, 37.52f, 37.63f, 37.71f, 37.78f, 37.89f, 38.68f, 
38.95f, 39.47f, 39.87f, 40.94f, 46.32f, 47.11f, 47.37f, 48.30f, 48.42f, 48.68f, 49.09f, 49.47f, 50.15f, 50.41f, 50.53f, 
51.32f, 51.58f, 52.52f, 52.63f, 52.89f, 53.30f, 53.83f, 54.10f, 54.36f, 54.50f, 55.53f, 55.67f, 55.79f, 56.20f, 56.46f, 
56.73f, 56.99f, 57.11f, 57.25f, 57.52f, 57.78f, 57.89f, 58.16f, 58.42f, 58.57f, 58.68f, 58.95f, 59.21f, 59.36f, 59.47f, 
59.62f, 59.74f, 59.88f, 60.00f, 60.41f, 60.67f, 60.94f, 61.20f, 61.32f, 61.46f, 61.73f, 61.99f, 62.11f, 62.37f, 62.63f, 
62.78f, 63.04f, -1.0f};

float times_blipp[] =
{0.15f, 0.26f, 0.53f, 0.94f, 1.20f, 1.32f, 1.58f, 1.73f, 1.84f, 1.99f, 2.25f, 2.63f, 3.04f, 3.30f, 3.42f, 
3.68f, 3.83f, 3.95f, 4.36f, 4.47f, 4.74f, 5.15f, 5.41f, 5.53f, 5.79f, 5.94f, 6.05f, 6.20f, 6.46f, 6.84f, 
7.25f, 7.52f, 7.63f, 7.89f, 8.04f, 8.16f, 8.57f, 8.68f, 8.95f, 9.36f, 9.62f, 9.74f, 10.00f, 10.15f, 10.26f, 
10.41f, 10.67f, 11.05f, 11.46f, 11.73f, 11.84f, 12.11f, 12.25f, 12.37f, 12.78f, 12.89f, 13.16f, 13.57f, 13.83f, 13.95f, 
14.21f, 14.36f, 14.47f, 14.62f, 14.88f, 15.26f, 15.67f, 15.94f, 16.05f, 16.32f, 16.46f, 16.58f, 16.99f, 17.11f, 17.37f, 
17.78f, 18.04f, 18.16f, 18.42f, 18.57f, 18.68f, 18.83f, 19.09f, 19.47f, 19.88f, 20.15f, 20.26f, 20.53f, 20.67f, 20.79f, 
21.20f, 21.32f, 21.58f, 21.99f, 22.25f, 22.37f, 22.63f, 22.78f, 22.89f, 23.04f, 23.30f, 23.68f, 24.09f, 24.36f, 24.47f, 
24.74f, 24.88f, 25.00f, 25.53f, 25.79f, 25.94f, 26.32f, 26.58f, 26.73f, 27.11f, 27.37f, 27.52f, 27.89f, 28.16f, 28.30f, 
29.74f, 30.00f, 30.15f, 30.53f, 30.79f, 30.94f, 31.32f, 31.58f, 31.73f, 32.11f, 32.37f, 32.52f, 42.25f, 42.37f, 42.63f, 
43.04f, 43.30f, 43.42f, 43.68f, 43.83f, 43.95f, 44.09f, 44.36f, 44.74f, 45.15f, 45.41f, 45.53f, 45.79f, 45.94f, 46.05f, 
46.32f, 46.46f, 46.73f, 46.84f, 47.11f, 47.25f, 47.52f, 47.63f, 47.89f, 48.04f, 48.16f, 48.30f, 48.42f, 48.57f, 48.83f, 
48.95f, 49.21f, 49.36f, 49.62f, 49.74f, 50.00f, 50.15f, 50.26f, 50.41f, 50.53f, 50.67f, 50.94f, 51.05f, 51.32f, 51.46f, 
51.73f, 51.84f, 52.11f, 52.25f, 52.37f, 52.52f, 52.63f, 52.78f, 53.04f, 53.16f, 53.42f, 53.57f, 55.00f, 55.26f, 55.41f, 
55.79f, 56.05f, 56.20f, 56.58f, 56.84f, 56.99f, 57.37f, 57.63f, 57.78f, 59.21f, 59.47f, 59.62f, 60.00f, 60.26f, 60.41f, 
60.79f, 61.05f, 61.20f, 61.58f, 61.84f, 61.99f, -1.0f};

int notes_blipp[] =
{65, 77, 65, 77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 
77, 77, 65, 65, 77, 65, 77, 65, 77, 77, 77, 65, 65, 77, 65, 
77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 77, 77, 65, 
65, 77, 65, 77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 
77, 77, 65, 65, 77, 65, 77, 65, 77, 77, 77, 65, 65, 77, 65, 
77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 77, 77, 65, 
65, 77, 65, 77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 
77, 77, 65, 101, 101, 89, 89, 89, 77, 77, 77, 65, 65, 65, 53, 
101, 101, 89, 89, 89, 77, 77, 77, 65, 65, 65, 53, 65, 77, 65, 
77, 65, 77, 77, 77, 65, 65, 77, 65, 77, 65, 77, 77, 77, 65, 
80, 92, 68, 75, 84, 75, 68, 78, 84, 68, 78, 75, 77, 89, 68, 
77, 85, 80, 68, 77, 85, 68, 80, 75, 80, 92, 68, 75, 84, 75, 
68, 78, 84, 68, 78, 75, 77, 89, 68, 77, 85, 80, 101, 101, 89, 
89, 89, 77, 77, 77, 65, 65, 65, 53, 101, 101, 89, 89, 89, 77, 
77, 77, 65, 65, 65, 53};


/*
 
	basis2
               .--
             .'
            /
           /
          /
         /
       .'
	--'
*/

float hermite_basis2(float s) { return -2*s*s*s + 3*s*s; }
float hermite_basis3(float s) { return s*s*s - 2*s*s + s; }


static FILE *file = NULL;

static short *audio_data = 0;
static PaStream *audio_stream = 0;
static int current_sample = 0;
static int sample_rate = 0;


static int pacallback(const void *in, void *out_, unsigned long frames, const PaStreamCallbackTimeInfo *time, PaStreamCallbackFlags status, void *user)
{
	short *out = (short *)out_;
	if(current_sample/2 + frames > audio_num_samples)
	{
		for(unsigned long s = 0; s < frames; s++)
		{
			out[s*2] = 0;
			out[s*2+1] = 0;
		}
	}
	else
	{
		for(unsigned long s = 0; s < frames; s++)
		{
			out[s*2] = audio_data[current_sample];
			out[s*2+1] = audio_data[current_sample+1];
			current_sample+=2;
			
		}
	}
	return 0;
}

static int snd_init()
{
	//PaStreamParameters params;
	//PaError err =
	Pa_Initialize();

	printf("pa open: %d\n", Pa_OpenDefaultStream(&audio_stream, 0, 2, paInt16, sample_rate, 128, pacallback, 0));
	Pa_StartStream(audio_stream);

#if 0
	params.device = Pa_GetDefaultOutputDevice();
	params.channelCount = 2;
	params.sampleFormat = paInt16;
	params.suggestedLatency = Pa_GetDeviceInfo(params.device)->defaultLowOutputLatency;
	params.hostApiSpecificStreamInfo = 0x0;

	err = Pa_OpenStream(
			&audio_stream,        /* passes back stream pointer */
			0,              /* no input channels */
			&params,                /* pointer to parameters */
			sample_rate,          /* sample rate */
			128,            /* frames per buffer */
			paClipOff,              /* no clamping */
			pacallback,             /* specify our custom callback */
			0x0); /* pass our data through to callback */
	err = Pa_StartStream(audio_stream);
#endif
	return 0;
}

static int read_data(void *buffer, int size)
{
	return fread(buffer, 1, size, file);	
}

int snd_load_wv(const char *filename)
{
	char error[100];
	WavpackContext *context;
	
	/* don't waste memory on sound when we are stress testing */
	file = fopen(filename, "rb"); /* TODO: use system.h stuff for this */
	if(!file)
		return -1;

	context = WavpackOpenFileInput(read_data, error);
	if (context)
	{
		audio_num_samples = WavpackGetNumSamples(context);
		//int bitspersample = WavpackGetBitsPerSample(context);
		sample_rate = WavpackGetSampleRate(context);
		int channels = WavpackGetNumChannels(context);
		int *data;
		int *src;
		short *dst;
		//printf("%d %d %d %d\n", sample_rate, bitspersample, channels, audio_num_samples);

		data = (int *)mem_alloc(4*audio_num_samples*channels, 1);
		WavpackUnpackSamples(context, data, audio_num_samples); /* TODO: check return value */
		src = data;
		
		dst = (short *)mem_alloc(2*audio_num_samples*channels, 1);
		audio_data = dst;

		for(unsigned long i = 0; i < audio_num_samples*channels; i++)
			*dst++ = (short)*src++;

		mem_free(data);
	}

	fclose(file);
	file = NULL;
	return 0;
}

GLuint gfx_load_png(const char *filename)
{
	unsigned char *buffer;
	png_t png;
	
	/* open file for reading */
	png_init(0,0);

	if(png_open_file(&png, filename) != PNG_NO_ERROR)
	{
		dbg_msg("game/png", "failed to open file. filename='%s'", filename);
		return 0;
	}
	
	if(png.depth != 8 || (png.color_type != PNG_TRUECOLOR && png.color_type != PNG_TRUECOLOR_ALPHA))
	{
		dbg_msg("game/png", "invalid format. filename='%s'", filename);
		png_close_file(&png);
        return 0;
	}
		
	buffer = (unsigned char *)mem_alloc(png.width * png.height * png.bpp, 1);
	png_get_data(&png, buffer);
	png_close_file(&png);
	
	GLuint ogltexture = -1;
	glGenTextures(1, &ogltexture);
	glBindTexture(GL_TEXTURE_2D, ogltexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA, png.width, png.height, GL_RGBA, GL_UNSIGNED_BYTE, buffer);	
	
	return ogltexture;
}

void gfx_blend_add() { glEnable(GL_BLEND); glBlendFunc(GL_SRC_ALPHA, GL_ONE); }
void gfx_blend_normal() { glEnable(GL_BLEND); glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); }

void gfx_ortho()
{
	/* set ortho projection */	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, gfx_width, gfx_height, 0, 1.0f, 10.f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(0,0,-5);
	glDisable(GL_DEPTH_TEST);
}

void gfx_projection()
{
	/* set ortho projection */	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(80.0f, 4/3.0f, 1, 1000);
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
}

static float hue_to_rgb(float v1, float v2, float h)
{
   if(h < 0) h += 1;
   if(h > 1) h -= 1;
   if((6 * h) < 1) return v1 + ( v2 - v1 ) * 6 * h;
   if((2 * h) < 1) return v2;
   if((3 * h) < 2) return v1 + ( v2 - v1 ) * ((2.0f/3.0f) - h) * 6;
   return v1;
}

vec3 hsl_to_rgb(vec3 in)
{
	float v1, v2;
	vec3 out;

	if(in.s == 0)
	{
		out.r = in.l;
		out.g = in.l;
		out.b = in.l;
	}
	else
	{
		if(in.l < 0.5f) 
			v2 = in.l * (1 + in.s);
		else           
			v2 = (in.l+in.s) - (in.s*in.l);

		v1 = 2 * in.l - v2;

		out.r = hue_to_rgb(v1, v2, in.h + (1.0f/3.0f));
		out.g = hue_to_rgb(v1, v2, in.h);
		out.b = hue_to_rgb(v1, v2, in.h - (1.0f/3.0f));
	} 

	return out;
}


/* open window */	
int init()
{
	glfwInit();

	if(gfx_fsaa)
		glfwOpenWindowHint(GLFW_FSAA_SAMPLES, gfx_fsaa);
	
	if(gfx_fullscreen)
	{
		int result = glfwOpenWindow(gfx_width, gfx_height, 8, 8, 8, 0, 24, 0, GLFW_FULLSCREEN);
		if(result != GL_TRUE)
			return -1;
	}
	else
	{
		int result = glfwOpenWindow(gfx_width, gfx_height, 0, 0, 0, 0, 24, 0, GLFW_WINDOW);
		if(result != GL_TRUE)
			return -1;
	}

	/* set some default settings */	
	gfx_blend_add();
	glDisable(GL_CULL_FACE);
	
	gfx_ortho();
	
	return 0;
}

int shutdown()
{
	glfwCloseWindow();
	glfwTerminate();
	return 0;
}

static inline void glVertex(vec2 v) { return glVertex2f(v.x, v.y); }
static inline void glVertex(vec3 v) { return glVertex3f(v.x, v.y, v.z); }

void render_thick_line(vec2 p0, vec2 p1, float thickness)
{
	vec2 dir = normalize(p1-p0);
	vec2 p = perp(dir);
	
	glTexCoord2f(0,0); glVertex(p0+p*thickness);
	glTexCoord2f(0,1); glVertex(p0-p*thickness);
	glTexCoord2f(1,1); glVertex(p1-p*thickness);
	glTexCoord2f(1,0); glVertex(p1+p*thickness);
}

typedef void (*EFFECT)();

void render_effect(float start_time, float end_time, EFFECT effect)
{
	if(demo_time < start_time || demo_time > end_time)
		return;
	
	local_time = demo_time-start_time;
	local_duration = end_time-start_time;
	local_amount = local_time/local_duration;
	local_timeleft = end_time - demo_time;
	effect();
}

inline float fdrandom() { return (frandom()*2-1); }

void effect_flum()
{
	int i;

	gfx_blend_add();

	glBegin(GL_QUADS);
	glColor4f(0.3f,0.3f,0.3f,0.10f);
	
	float size = gfx_width*0.15f;
	srand(0);
	for(i = 0; i < 500; i++)
	{
		vec2 dir(sinf(fdrandom()*local_time), cosf(fdrandom()*local_time));
		vec2 center(frandom()*gfx_width, fdrandom()*gfx_height+local_time*20);
		render_thick_line(center + dir*size, center - dir*size, size*0.1f+frandom()*size*0.1f);
	}
	glEnd();
}

void effect_bizarro()
{
	gfx_blend_add();

	float offset_amount = hermite_basis2(clamp(local_time-(4*4)/audio_bps, 0.0f, 1.0f));
	vec3 offset(offset_amount*-gfx_width,0,0);
	glTranslatef(offset.x, offset.y, offset.z);
	
	glBegin(GL_TRIANGLE_FAN);
	
	glColor4f(1,1,1,1);
	glVertex2f(gfx_width/2, gfx_height/2);
	
	float beep = trigger_bass;

	const int resolusion = 500;
	for(int i = 0; i <= resolusion; i++)
	{
		float a = i/(float)resolusion;
		float r = a*pi*2;
		float o = sinf(r*8+local_time*5)* sinf(local_time) + cosf(r*35)*beep*0.30f;
		
		float y = cosf(r);
		float x = sinf(r);
		
		float s = 200 + sinf(local_time*2)*25+o*100.0f;

		vec3 c = hsl_to_rgb(vec3(fmod(a+local_time, 1.0f), 0.5f, 0.5f));
		glColor4f(c.r,c.g,c.b,0.9+sinf(local_time*45+a*5)*0.1f);
	
		glVertex2f(gfx_width/2 + x*s, gfx_height/2 + y*s);
	}
	
	glEnd();
}

inline float hermite_basis1(float v) { return 2*v*v*v - 3*v*v+1; }

void effect_overlay()
{
	gfx_blend_normal();
	
	glBindTexture(GL_TEXTURE_2D, texture_overlay);
	glEnable(GL_TEXTURE_2D);
	glBegin(GL_QUADS);
	glColor4f(0.0f,0.0f,0.0f,0.890f+sinf(sinf(local_time*5)*10)*0.1f);
	for(int i = 0; i < 1; i++)
	{
		glTexCoord2f(0,0); glVertex2f(0,0);
		glTexCoord2f(1,0); glVertex2f(gfx_width,0);
		glTexCoord2f(1,1); glVertex2f(gfx_width,gfx_height);
		glTexCoord2f(0,1); glVertex2f(0,gfx_height);
	}
	glEnd();
	glDisable(GL_TEXTURE_2D);
}

void effect_boxes()
{
	gfx_blend_normal();

	for(int i = 0; i < 5; i++)
	{
		glBindTexture(GL_TEXTURE_2D, texture_plates[plate_offset+i]);
		glEnable(GL_TEXTURE_2D);

		glBegin(GL_QUADS);
		glColor4f(1,1,1,1);
	
		float amount;

		amount = max(min((local_time-i*0.25f)/0.75f, 1.0f), 0.0f);

		float xsize = gfx_width*0.15f;
		float ysize = xsize*0.5f;

		vec2 start_pos(gfx_width/2-i*gfx_width*0.1f, -gfx_height*0.1f);
		vec2 end_pos(gfx_width/2, gfx_height*0.85f-i*ysize*1.25f);
		vec2 center = mix(start_pos, end_pos, amount*amount);
		float rotation = 4*(1-amount);
		
		xsize *= amount;

		if(local_time > 3.25f)
		{
			amount = max(min((local_time-3.25f-i*0.05f)/0.75f, 1.0f), 0.0f);
			center = end_pos + vec2(0,1)*gfx_height*1.25f*amount*amount;
			rotation = amount*2;
		}
		
		vec2 dir(cosf(rotation), sinf(rotation));
		render_thick_line(center-dir*xsize, center+dir*xsize, ysize);

		glEnd();
		
		glDisable(GL_TEXTURE_2D);

	}
}

void effect_hexfloor()
{
	gfx_projection();
	gfx_blend_normal();
	
	glEnable(GL_DEPTH_TEST);
	
	float rotatetime = local_time * 0.4f;

	float upstuff = sinf(local_time*2)*1.5f + powf(sinf(local_time*1),3)*2.0f;

	vec3 p0(sinf(rotatetime)*0.1f,1.2f+upstuff,cosf(rotatetime)*0.1f);
	vec3 u0(0,1,0);
	vec3 l0(0,-10,0);

	vec3 p1(sinf(rotatetime)*13,10+sinf(rotatetime*3)*2,cosf(rotatetime)*13);
	vec3 u1(0,1,0);
	vec3 l1(0,0,0);
	
	float blendtime = clamp(local_time * 0.2f, 0.0f, 1.0f);
	
	vec3 p = mix(p0, p1, hermite_basis2(blendtime)) + vec3(0,10,0) * hermite_basis3(clamp(blendtime, 0.0f, 1.0f));
	vec3 u = u0; //normalize(mix(u0, u1, blendtime));
	vec3 l = mix(l0, l1, hermite_basis2(blendtime));

	float offset_amount = hermite_basis2(clamp(local_time-(4*3+2)/audio_bps, 0.0f, 1.0f));
	vec3 offset(offset_amount*-12,offset_amount*-5,0);
	
	//printf("%f\n", offset.x);
	glTranslatef(offset.x, offset.y, offset.z);
	glRotatef(offset_amount*-20.0f, 1, 0, 0);
	
	gluLookAt(p.x,p.y,p.z, l.x,l.y,l.z, u.x,u.y,u.z);
	//gluLookAt(sinf(rotatetime)*13,10+sinf(rotatetime*3)*2,cosf(rotatetime)*13, 0,0,0, 0,1.0f,0);


	static GLuint lists = 0;
	bool do_lists = false;
	
	if(lists == 0)
	{
		lists = glGenLists(20*20);
		do_lists = true;
	}
	
	for(int y = -10, c = 0; y < 10; y++)
		for(int x = -10; x < 10; x++, c++)
		{
			float fog = clamp(1 - length(vec2(x,y)) / 10.0f, 0.0f, 1.0f)*0.9f;
			float up = sinf(sqrtf(x*x+y*y)+local_time*2)*1.5f + powf(sinf(sqrtf(x*x+y*y)+local_time*1),3)*2.0f;
			
			glPushMatrix();
			
			const float ystep = 3.05f;
			const float xstep = 3.5f;
			if(y&1)
				glTranslatef(x*xstep-xstep/2,0,y*ystep);
			else
				glTranslatef(x*xstep,0,y*ystep);
			
			glTranslatef(0,up,0);
			
			if(!do_lists)
			{
				glCallList(c);
			}
			else
			{
				glNewList(c, GL_COMPILE_AND_EXECUTE);
				glBegin(GL_QUADS);

				const float size = 2.0f;
				const int sides = 6;
				for(int i = 0; i < sides; i += 2)
				{
					float r0 = (i)/(float)sides * pi * 2;
					float r1 = (i+1)/(float)sides * pi * 2;
					float r2 = (i+2)/(float)sides * pi * 2;
					
					float c0 = fog;
					float c1 = ((i+1)/(float)(sides) + 0.4f) * 0.8f * fog;
					float c2 = ((i+2)/(float)(sides)+ 0.4f) * 0.8f * fog;
					
					glColor4f(c0,c0,c0,1);

					glVertex(vec3(0,0,0));
					glVertex(vec3(sinf(r0), 0, cosf(r0))*size);
					glVertex(vec3(sinf(r1), 0, cosf(r1))*size);
					glVertex(vec3(sinf(r2), 0, cosf(r2))*size);

					glColor4f(c1,c1,c1,1);
					glVertex(vec3(sinf(r0), 0, cosf(r0))*size);
					glVertex(vec3(sinf(r1), 0, cosf(r1))*size);
					glColor4f(0,0,0,1);
					glVertex(vec3(sinf(r1), -5, cosf(r1))*size);
					glVertex(vec3(sinf(r0), -5, cosf(r0))*size);

					glColor4f(c2,c2,c2,1);
					glVertex(vec3(sinf(r1), 0, cosf(r1))*size);
					glVertex(vec3(sinf(r2), 0, cosf(r2))*size);
					glColor4f(0,0,0,1);
					glVertex(vec3(sinf(r2), -5, cosf(r2))*size);
					glVertex(vec3(sinf(r1), -5, cosf(r1))*size);
				}
				
				glEnd();
				glEndList();
			}
			glPopMatrix();
			
		}
}

void effect_shutter()
{
	float time = min(max(local_time/3.0f, 0.0f), 1.0f);

	gfx_blend_normal();
	glBegin(GL_QUADS);
	const int num_shutters = 25;
	
	for(int i = 0; i < num_shutters; i++)
	{
		float size = 1.0f/(float)num_shutters * gfx_width;
		float start = i*size;
		float offset = pow((1-time), (num_shutters-i)*0.5f+8)*gfx_height;
		if(i&1)
			offset = -offset;
		glColor4f(1,1,1,1);
		glVertex2f(start, offset);
		glVertex2f(start+size, offset);
		glVertex2f(start+size, offset+gfx_height);
		glVertex2f(start, offset+gfx_height);
	}
	
	glEnd();
}

void effect_circles()
{
	gfx_blend_normal();
	float cx = gfx_width/2;
	float cy = gfx_height/2;

	//float inner_rad = length(vec2(gfx_width, gfx_height)) * (1-local_amount);
	//float outer_rad = 10000;

	float inner_rad = 0;
	float outer_rad = length(vec2(gfx_width, gfx_height)) * powf(local_amount,2);
	
	const int num_segments = 100;

	glBegin(GL_QUADS);
	for(int s = 0; s < num_segments; s++)
	{
		float r0 = s/(float)num_segments * pi*2;
		float r1 = (s+1)/(float)num_segments * pi*2;

		glColor4f(1,1,1,1);
		glVertex2f(cx + cosf(r0)*inner_rad, cy + sinf(r0)*inner_rad);
		glVertex2f(cx + cosf(r0)*outer_rad, cy + sinf(r0)*outer_rad);
		glVertex2f(cx + cosf(r1)*outer_rad, cy + sinf(r1)*outer_rad);
		glVertex2f(cx + cosf(r1)*inner_rad, cy + sinf(r1)*inner_rad);
	}

	glEnd();
}

void effect_bricks()
{
	gfx_blend_normal();
	glBegin(GL_QUADS);
	const int w = 15;
	const int h = (int)(w/gfx_aspect);
	const float bw = gfx_width/(float)w;
	const float bh = gfx_height/(float)h;
	
	for(int y = 0; y < h; y++)
		for(int x = 0; x < w; x++)
		{
			float offset = 1-(sqrtf(x*x+y*y)/sqrtf(25*25+20*20));
			float amount = clamp(local_time - offset, 0.0f, 1.0f);
			float ia = 1-amount;
			float rot = amount*15;
			vec2 center(x*bw+bw/2, y*bh + bh/2);
			vec2 up(sinf(rot), cosf(rot));
			vec2 right = perp(up);
			float size = mix(1.0f, 0.5f, amount);
			
			center += vec2(0, amount*amount*gfx_height*2);
			
			up *= bh/2 * size;
			right *= bw/2 * size;
			
			glColor4f(ia,ia,ia,1);
			glVertex(center-up-right);
			glVertex(center-up+right);
			glVertex(center+up+right);
			glVertex(center+up-right);
		}
	glEnd();	
}

void effect_starfield()
{
	gfx_blend_normal();

	glBindTexture(GL_TEXTURE_2D, texture_blob);
	glEnable(GL_TEXTURE_2D);
	
	glBegin(GL_QUADS);
	const int num_stars = 250;
	
	srand(0);
	
	for(int i = 0; i < num_stars; i++)
	{
		float speed = (100+frandom()*1500.0f);
		vec2 start(frandom()*gfx_width*10-local_time*speed, frandom()*gfx_height);
		float length = 50.0f+speed/10.0f;
		float height = 5.0f;
		
		glColor4f(1,1,1,1);
		glTexCoord2f(0,0); glVertex2f(start.x, start.y);
		glTexCoord2f(1,0); glVertex2f(start.x+length, start.y+height/2);
		glTexCoord2f(1,1); glVertex2f(start.x+length, start.y+height/2);
		glTexCoord2f(0,1); glVertex2f(start.x, start.y+height);
	}
	
	glEnd();
	glDisable(GL_TEXTURE_2D);
}

void effect_selection()
{
	glBegin(GL_QUADS);
	float amount = min(local_amount*5, 1.0f);
	if(local_amount > 0.8f)
		amount = max(1-(local_amount-0.8f)*5, 0.0f);
		
	float size_amount = (0.75f + 0.25f*trigger_blipp_high - 0.25f*trigger_blipp_low);
	
	float total_height = gfx_height*0.10f;
	float size = gfx_height*0.10f*size_amount;
	float border_size = 5.0f*size_amount;
	
	glColor4f(0.4f,0.6f,1,1);
	glVertex2f(-gfx_width, gfx_height/2-size);
	glVertex2f(gfx_width*2, gfx_height/2-size);
	glVertex2f(gfx_width*2, gfx_height/2+size);
	glVertex2f(-gfx_width, gfx_height/2+size);

	glColor4f(0.2f,0.4f,0.8f,1);
	glVertex2f(-gfx_width, gfx_height/2-size);
	glVertex2f(gfx_width*2, gfx_height/2-size);
	glVertex2f(gfx_width*2, gfx_height/2-size+border_size);
	glVertex2f(-gfx_width, gfx_height/2-size+border_size);

	glVertex2f(-gfx_width, gfx_height/2+size);
	glVertex2f(gfx_width*2, gfx_height/2+size);
	glVertex2f(gfx_width*2, gfx_height/2+size-border_size);
	glVertex2f(-gfx_width, gfx_height/2+size-border_size);
	
	glEnd();
	
	
	glBindTexture(GL_TEXTURE_2D, texture_opening[plate_offset]);
	glEnable(GL_TEXTURE_2D);
	
	glBegin(GL_QUADS);
	glColor4f(1,1,1,amount);
	glTexCoord2f(0,0); glVertex2f(gfx_width/2-total_height*4, gfx_height/2-total_height);
	glTexCoord2f(1,0); glVertex2f(gfx_width/2+total_height*4, gfx_height/2-total_height);
	glTexCoord2f(1,1); glVertex2f(gfx_width/2+total_height*4, gfx_height/2+total_height);
	glTexCoord2f(0,1); glVertex2f(gfx_width/2-total_height*4, gfx_height/2+total_height);
	glEnd();
	glDisable(GL_TEXTURE_2D);
	
}

void effect_hexagons()
{
	gfx_projection();
	gfx_blend_normal();
	
	glDisable(GL_DEPTH_TEST);
	
	gluLookAt(0,0,15, 0,0,0, 0,1.0f,0);

	glLineWidth(1.2f);
	
	float fade = clamp(min(local_time, local_timeleft),0.0f,1.0f);

	glColor4f(1,0,0,fade);

	for(int p = 0; p < 2; p++)
	{
		if(p && trigger_kick<0.01f)
			return;
	
		srand(0);
		const int num_hexes = 500;
		for(int s = 0; s < num_hexes; s++)
		{
			if(s&1)
				continue;
			
			glPushMatrix();
			glTranslatef(fdrandom()*30,fdrandom()*30,-200+frandom()*200+local_time*15);
			glRotatef(90*local_time, fdrandom(), fdrandom(), fdrandom());
			
			if(p)
			{
				glBegin(GL_TRIANGLE_FAN);
				glColor4f(1,0,0,fade*trigger_kick);
				glVertex(vec2(0,0));
			}
			else
				glBegin(GL_LINES);
			
			const int sides = 6;
			for(int i = 0; i < sides; i++)
			{
				float r = i/(float)sides * pi * 2;
				float r2 = (i+1)/(float)sides * pi * 2;
				
				glVertex(vec2(sinf(r), cosf(r))*2);
				glVertex(vec2(sinf(r2), cosf(r2))*2);
			}
			
			glEnd();	
			glPopMatrix();
		}
	}
}


void effect_nameplates()
{
	glEnable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);
	
	const int num_plates = 10;
	for(int i = 0; i < num_plates; i++)
	{
		glPushMatrix();
		float amount = clamp(local_time*0.30f-i*0.10f, 0.0f, 1.0f);
		
		gluLookAt(0,0,20, 0,0,0, 0,1.0f,0);

		float dir = -1;
		if(i&1)
			dir = 1;
		
		glTranslatef((1-hermite_basis2(clamp(amount*2,0.0f,1.0f)))*dir*-4, -20+amount*40, 0);
		
		if(i&1)
		{
			glCullFace(GL_BACK);
			glRotatef(-90 + hermite_basis2(clamp(amount*2,0.0f,1.0f))*80, 0, 1, 0);
		}
		else
		{
			glCullFace(GL_FRONT);
			glRotatef(-90 - hermite_basis2(clamp(amount*2,0.0f,1.0f))*80, 0, 1, 0);
		}
		glTranslatef(-5, 0, 0);


		glBindTexture(GL_TEXTURE_2D, texture_plates[plate_offset+i]);
		glEnable(GL_TEXTURE_2D);
		
		glBegin(GL_QUADS);
		
		float w = 16;
		float h = 8;
			
		glColor4f(0,0,0,1-powf(local_amount,32));
		glTexCoord2f(0,1); glVertex3f(0, 0, 0);
		glTexCoord2f(dir,1); glVertex3f(w, 0, 0);
		glTexCoord2f(dir,0); glVertex3f(w, h, 0);
		glTexCoord2f(0,0); glVertex3f(0, h, 0);
		
		glEnd();
		
		glPopMatrix();
	}
	
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glEnable(GL_DEPTH_TEST);
}

void render_grid()
{
	gfx_blend_normal();
	glBegin(GL_LINES);
	
	glColor4f(1,1,1,1);
	
	for(int z = -10; z <= 10; z++)
	{
		glVertex3f(-10, 0, z);
		glVertex3f(10, 0, z);
	}
	
	for(int x = -10; x <= 10; x++)
	{
		glVertex3f(x, 0, -10);
		glVertex3f(x, 0, 10);
	}
	
	glEnd();
}

float cursor_x = 0;
float cursor_y = 0;

float mouse_x = 0;
float mouse_y = 0;
float mouse_z = 0;

float mouse2_x = 0;
float mouse2_y = 0;
float mouse2_z = 0;

void set_mouse_stuff(float a, float b, float c, float d, float e, float f)
{
	mouse_x = a;
	mouse_y = b;
	mouse_z = c;
	mouse2_x = d;
	mouse2_y = e;
	mouse2_z = f;
}

void mouse_look()
{
	gluLookAt(mouse_x, mouse_y, mouse_z, mouse2_x, mouse2_y, mouse2_z, 0, 1.0f, 0);
}

void effect_spiral()
{
	const float spiralspeed = 100.0f;
	float spiralamount = local_time/5.0f;
	
	gfx_projection();
	gfx_blend_normal();

	gluLookAt(mix(6.300158f, 10.200165f, spiralamount), 7.599995f, mix(16.499973f, 17.499977f, spiralamount),-7.399996f,0.000000f,8.099996f, 0, 1.0f, 0);
	
	vec3 start(0,0,-30);
	vec3 end(0,0,30);
	float start_rot = 0;
	float end_rot = 20;
	
	for(int p = 0; p < 2; p++)
	{
		glLineWidth(1.25f);
		glBegin(GL_LINES);
		
		const int num_steps = 150;
		for(int s = 0; s < num_steps; s++)
		{
			if(s < (local_time-1)*spiralspeed)
				continue;
			if(s > local_time*spiralspeed)
				break;
			
			float a = s/(float)num_steps;
			float a2 = (s+1)/(float)num_steps;
			vec3 center = mix(start, end, a)+vec3(sinf(a*5), cosf(a*5),0)*2;
			vec3 center2 = mix(start, end, a2)+vec3(sinf(a2*5), cosf(a2*5),0)*2;
			float rot = mix(start_rot, end_rot, a);
			float rot2 = mix(start_rot, end_rot, a2);
			
			vec3 outer = vec3(sinf(rot), cosf(rot), 0)*2;
			vec3 outer2 = vec3(sinf(rot2), cosf(rot2), 0)*2;
			
			const int num_sections = 10;
			for(int k = 0; k < num_sections; k++)
			{
				float m = k/(float)num_sections;
				float m2 = (k+1)/(float)num_sections;
				if(0)
					glColor4f(0,0,0,1);
				else
					glColor4f(a,m,0,1);
				glColor4f(1,1,1,1);
				
				vec3 c = hsl_to_rgb(vec3(a*2, 0.5f, 0.5f));
				glColor4f(c.r,c.g,c.b,0.9+sinf(local_time*45+a*5)*0.1f);
				glColor4f(0,0,0,1);
				
				vec3 u1 = mix(center-outer, center+outer, m);
				vec3 u2 = mix(center2-outer2, center2+outer2, m);
				vec3 v1 = mix(center-outer, center+outer, m2);
				vec3 v2 = mix(center2-outer2, center2+outer2, m2);

				if(1)
				{
					glVertex(u1);
					glVertex(u2);

					glVertex(u1);
					glVertex(v1);

					glVertex(u2);
					glVertex(v2);

					glVertex(v1);
					glVertex(v2);
				}
				else
				{
					glVertex(u1);
					glVertex(u2);
					glVertex(v2);
					glVertex(v1);
				}
			}
		}
		glEnd();
	}
}

void effect_cylinder()
{
	//env3 env = curl;

	gfx_projection();
	gfx_blend_add();
	
	glBindTexture(GL_TEXTURE_2D, texture_blob);
	glEnable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	
	//gluLookAt(15,15,15, 15,15,0, 0,1.0f,0);
	
	
	//mouse_look();
	gluLookAt(7.000002f,-1.600002f,26.299980f,-7.899993f,-4.799998f,6.199997f, 0,1.0f,0);
	
	glRotatef(sinf(local_time*0.5f)*360,0,0,1);

	glBegin(GL_QUADS);
	const int num_steps = 20;
	for(int s = 0; s < num_steps; s++)
	{
		if(s&1)
			continue;
			
		float a = s/(float)num_steps;
		float a2 = (s+1)/(float)num_steps;
		float time = min(local_time/2.5f, 1.0f);
		vec3 center = vec3(cosf(a*pi*2*time), sinf(a*pi*2*time), 0)*10;
		vec3 center2 = vec3(cosf(a2*pi*2*time), sinf(a2*pi*2*time), 0)*10;
		vec3 dir = vec3(0, 0, 1)*20;
		
		float trigger = trigger_blipp[s/4];
		vec3 c = hsl_to_rgb(vec3(a, 0.25f+0.75f*trigger, 0.25f+0.75f*trigger));
		glColor4f(c.r,c.g,c.b,0.9+sinf(local_time*45+a*5)*0.1f + 0.2f*trigger);

		glTexCoord2f(0,1); glVertex(center-dir);
		glTexCoord2f(0,0); glVertex(center+dir);
		glTexCoord2f(1,0); glVertex(center2+dir);
		glTexCoord2f(1,1); glVertex(center2-dir);

		if(trigger > 0.01f)
		{
			glTexCoord2f(0,1); glVertex(center-dir);
			glTexCoord2f(0,0); glVertex(center+dir);
			glTexCoord2f(1,0); glVertex(center2+dir);
			glTexCoord2f(1,1); glVertex(center2-dir);
		}
	}
	glEnd();
	
	glDisable(GL_TEXTURE_2D);
}

void render_debugtext(float x, float y, const char *text)
{
	gfx_ortho();
	gfx_blend_normal();
	
	glBindTexture(GL_TEXTURE_2D, texture_debug_font);
	glEnable(GL_TEXTURE_2D);	
	glColor4f(1,1,1,1);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);
	glBegin(GL_QUADS);
	
	while(*text)
	{
		float s = 32;
		float u = ((*text)%16) / 16.0f;
		float v = ((*text)/16) / 16.0f;
		float f = 1/16.0f;
		
		glTexCoord2f(u,v); glVertex2f(x, y);
		glTexCoord2f(u+f,v); glVertex2f(x+s, y);
		glTexCoord2f(u+f,v+f); glVertex2f(x+s, y+s);
		glTexCoord2f(u,v+f); glVertex2f(x, y+s);
		x += s/2;
		
		text++;
	}
	
	glEnd();	
	
	glDisable(GL_TEXTURE_2D);
}

void effect_whiteout()
{
	gfx_blend_normal();
	
	glBegin(GL_QUADS);
	glColor4f(1,1,1,1-local_amount);
	vec2 center(gfx_width/2,gfx_height/2);
	vec2 dir(gfx_width/4,gfx_width/8);

	glTexCoord2f(0,0); glVertex2f(0,0);
	glTexCoord2f(1,0); glVertex2f(gfx_width, 0);
	glTexCoord2f(1,1); glVertex2f(gfx_width, gfx_height);
	glTexCoord2f(0,1); glVertex2f(0, gfx_height);
	glEnd();
	
	glDisable(GL_TEXTURE_2D);
}

void effect_endscreen()
{
	gfx_blend_normal();
	
	glBindTexture(GL_TEXTURE_2D, texture_endscreen);
	glEnable(GL_TEXTURE_2D);

	glBegin(GL_QUADS);
	glColor4f(1,1,1,1-powf(local_amount, 5));
	vec2 center(gfx_width/2,gfx_height/2);
	vec2 dir(gfx_width/4,gfx_width/8);

	glTexCoord2f(0,0); glVertex2f(center.x-dir.x, center.y-dir.y);
	glTexCoord2f(1,0); glVertex2f(center.x+dir.x, center.y-dir.y);
	glTexCoord2f(1,1); glVertex2f(center.x+dir.x, center.y+dir.y);
	glTexCoord2f(0,1); glVertex2f(center.x-dir.x, center.y+dir.y);
	glEnd();
	
	glDisable(GL_TEXTURE_2D);
}

int load()
{
	texture_overlay = gfx_load_png("data/overlay.png");
	texture_blob = gfx_load_png("data/blob.png");
	texture_nameplate = gfx_load_png("data/nameplate.png");
	texture_debug_font = gfx_load_png("data/debug_font.png");
	texture_endscreen = gfx_load_png("data/endscreen.png");
	
	int p = 0;
	texture_plates[p++] = gfx_load_png("data/p_stalvark80.png");
	texture_plates[p++] = gfx_load_png("data/p_fairlight.png");
	texture_plates[p++] = gfx_load_png("data/p_hypercube.png");
	texture_plates[p++] = gfx_load_png("data/p_tbl.png");
	texture_plates[p++] = gfx_load_png("data/p_birdie.png");

	texture_plates[p++] = gfx_load_png("data/p_psytroll.png");
	texture_plates[p++] = gfx_load_png("data/p_sphr.png");
	texture_plates[p++] = gfx_load_png("data/p_conceptp.png");
	texture_plates[p++] = gfx_load_png("data/p_cloaked.png");
	texture_plates[p++] = gfx_load_png("data/p_lyra.png");

	texture_plates[p++] = gfx_load_png("data/p_qaz.png");
	texture_plates[p++] = gfx_load_png("data/p_wixgf.png");
	texture_plates[p++] = gfx_load_png("data/p_redcom.png");
	texture_plates[p++] = gfx_load_png("data/p_monixa.png");
	texture_plates[p++] = gfx_load_png("data/p_jarlzor.png");
	
	texture_plates[p++] = gfx_load_png("data/p_althaeria.png");
	texture_plates[p++] = gfx_load_png("data/p_malmat.png");
	texture_plates[p++] = gfx_load_png("data/p_moh.png");
	texture_plates[p++] = gfx_load_png("data/p_andersa.png");
	texture_plates[p++] = gfx_load_png("data/p_allsceners.png");

	p = 0;
	texture_opening[p++] = gfx_load_png("data/opening1.png");
	texture_opening[p++] = gfx_load_png("data/opening2.png");
	texture_opening[p++] = gfx_load_png("data/opening3.png");
	texture_opening[p++] = gfx_load_png("data/opening4.png");

	return 0;
}

static void key_callback(int key, int action)
{
	if(action == GLFW_PRESS)
	{
		if(key == GLFW_KEY_RIGHT)
			current_sample += sample_rate*3;
		else if(key == GLFW_KEY_LEFT)
			current_sample -= sample_rate*3;
	}
}

void effect_white()
{
	glClearColor(1,1,1,1);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

float get_time_beat(int measure, int beat)
{
	return (measure-1)*4/audio_bps+(beat-1)/audio_bps;
}

int get_audio_measure()
{
	return (int)(demo_time * audio_bps / 4) + 1;
}

int get_audio_beat()
{
	return (int)(demo_time * audio_bps)%4 + 1;
}

float trigger(float value, float time)
{
	if(last_demo_time < time && demo_time >= time)
		return 1.0f;
	return value;
}

float reduce(float variable, float mod)
{
	return max(variable-mod*frame_time, 0.0f);
}

void print_help()
{
	printf(
	"arranging matches by matricks with music by teetow\n"
	"-w X    screen width\n"
	"-h X    screen height\n"
	"-fsaa X fsaa quality\n"
	"-win    windowed mode\n");
}

int main(int argc, const char **argv)
{
	argc--;
	argv++;
	while(argc)
	{
		if(strcmp(*argv, "-w") == 0)
		{
			if(argc < 2)
			{
				printf("expected argument for %s\n", *argv);
				return -1;
			}
			
			argc--; argv++;
			gfx_width = atol(*argv);
			argc--; argv++;
		}
		else if(strcmp(*argv, "-h") == 0)
		{
			if(argc < 2)
			{
				printf("expected argument for %s\n", *argv);
				return -1;
			}
			
			argc--; argv++;
			gfx_height = atol(*argv);
			argc--; argv++;
		}
		else if(strcmp(*argv, "-fsaa") == 0)
		{
			if(argc < 2)
			{
				printf("expected argument for %s\n", *argv);
				return -1;
			}
			
			argc--; argv++;
			gfx_fsaa = atol(*argv);
			argc--; argv++;
		}
		else if(strcmp(*argv, "-win") == 0)
		{
			gfx_fullscreen = 0;
			argc--; argv++;
		}
		else
		{
			// unkown option
			printf("unknown option: %s\n", *argv);
			print_help();
			return -1;
		}
	}
	
	
	if(init())
		return -1;
		
	glfwSetKeyCallback(key_callback);
	
	load();
	//glfwDisable(GLFW_MOUSE_CURSOR);
	
	//set_mouse_stuff(6.300158f,7.599995f,16.499973f,-7.399996f,0.000000f,8.099996f);
	
	snd_load_wv("data/music.wv");
	snd_init();
	
	float beat = 0;

	//current_sample = get_time_beat(23,1)*sample_rate;
	while(glfwGetKey(GLFW_KEY_ESC) != GLFW_PRESS)
	{
		int64 frame_start = time_get();
		demo_time = current_sample/2/(float)sample_rate;
		//demo_time = Pa_GetStreamTime(audio_stream);
		
		static int last_beat = 0;
		if(last_beat != get_audio_beat())
		{
			last_beat = get_audio_beat();
			beat = 1;
		}
		
		beat = clamp(beat-0.05f, 0.0f, 1.0f);
		
		//glClearColor(0.025f,0.05f,0.1f,1.0f);
		glClearColor(0.0f,0.0f,0.0f,1.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		// trigger kick		
		for(int i = 0; times_kick[i] > 0; i++)
			trigger_kick = trigger(trigger_kick, times_kick[i]);
		trigger_kick = reduce(trigger_kick, 5.0f);

		// trigger bass
		for(int i = 0; times_bass[i] > 0; i++)
			trigger_bass = trigger(trigger_bass, times_bass[i]);
		trigger_bass = reduce(trigger_bass, 5.0f);

		// trigger blipp
		for(int i = 0; times_blipp[i] > 0; i++)
		{
			if(last_demo_time < times_blipp[i] && demo_time >= times_blipp[i])
			{
				int note = notes_blipp[i];
				if(note == 101) trigger_blipp[0] = 1.0f;
				else if(note == 89) trigger_blipp[1] = 1.0f;
				else if(note == 77) { trigger_blipp[2] = 1.0f; trigger_blipp_high = 1.0f; }
				else if(note == 65) { trigger_blipp[3] = 1.0f; trigger_blipp_low = 1.0f; }
				else if(note == 53) trigger_blipp[4] = 1.0f;
				break;
			}
		}
		
		
		trigger_blipp[0] = reduce(trigger_blipp[0], 4.5f);
		trigger_blipp[1] = reduce(trigger_blipp[1], 4.5f);
		trigger_blipp[2] = reduce(trigger_blipp[2], 4.5f);
		trigger_blipp[3] = reduce(trigger_blipp[3], 4.5f);
		trigger_blipp[4] = reduce(trigger_blipp[4], 4.5f);
		
		trigger_blipp_low = reduce(trigger_blipp_low, 4.5f);
		trigger_blipp_high = reduce(trigger_blipp_high, 4.5f);
		
		//trigger_hexbeat = trigger(trigger_hexbeat, get_time_beat(7,3));
		//trigger_hexbeat = trigger(trigger_hexbeat, get_time_beat(8,1)); // do better
		//trigger_hexbeat = trigger(trigger_hexbeat, get_time_beat(8,3));
		//trigger_hexbeat = trigger(trigger_hexbeat, get_time_beat(8,4));
		
		gfx_ortho();		render_effect(get_time_beat(8,1), get_time_beat(21,1), effect_flum);
		
		gfx_ortho();		render_effect(get_time_beat(1,1), get_time_beat(6,1), effect_starfield);
		gfx_ortho();		plate_offset = 0; render_effect(get_time_beat(1,2), get_time_beat(2,2), effect_selection);
		gfx_ortho();		plate_offset = 1; render_effect(get_time_beat(2,2), get_time_beat(3,2), effect_selection);
		gfx_ortho();		plate_offset = 2; render_effect(get_time_beat(3,2), get_time_beat(4,2), effect_selection);
		gfx_ortho();		plate_offset = 3; render_effect(get_time_beat(4,2), get_time_beat(5,1), effect_selection);
		
		gfx_ortho(); 		render_effect(get_time_beat(5,1), get_time_beat(8,3), effect_shutter);
		gfx_ortho();		render_effect(get_time_beat(8,2), get_time_beat(13,1), effect_bizarro);
		gfx_ortho();		render_effect(get_time_beat(8,2), get_time_beat(10,1), effect_bricks);

		gfx_projection();	render_effect(get_time_beat(5,2), get_time_beat(8,3), effect_hexagons);

		gfx_projection();	render_effect(get_time_beat(12,3), get_time_beat(21,1), effect_cylinder);

		gfx_ortho();		render_effect(get_time_beat(20,2), get_time_beat(21,1), effect_circles);
		gfx_ortho();		render_effect(get_time_beat(21,1), get_time_beat(23,1), effect_white);
		
		gfx_projection();	render_effect(get_time_beat(23,1), get_time_beat(31,1), effect_hexfloor);

		gfx_projection();	render_effect(get_time_beat(21,3), get_time_beat(23,2), effect_spiral);
		gfx_projection();	plate_offset = 10; render_effect(get_time_beat(21,1), get_time_beat(23,2), effect_nameplates);

		gfx_projection();	render_effect(get_time_beat(26,3), get_time_beat(31,1), effect_cylinder);
		
		gfx_ortho();		plate_offset = 0; render_effect(get_time_beat(17,1), get_time_beat(19,1), effect_boxes);
		gfx_ortho();		plate_offset = 5; render_effect(get_time_beat(19,1), get_time_beat(21,1), effect_boxes);
		
		gfx_ortho();		render_effect(get_time_beat(31,1), get_time_beat(32,1), effect_endscreen);
		gfx_ortho();		render_effect(get_time_beat(31,1), get_time_beat(31,2), effect_whiteout);
		
		gfx_ortho();		render_effect(0.0f, get_time_beat(31,1), effect_overlay);
		
		gfx_ortho();
		char buf[64];
		sprintf(buf, "Demo - %.2f %d:%d  %.2f", demo_time, get_audio_measure(), get_audio_beat(), trigger_kick);
		//render_debugtext(10, 10, buf);
		
		glfwSwapBuffers();
		
		frame_time = (time_get()-frame_start)/(float)time_freq();
		last_demo_time = demo_time;
		//printf("%f fps\n", 1.0f / frametime);
	}

	shutdown();
	
	return 0;
}
