#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL.h>
#include "main.h"
#include "tools.h"

#define REPEAT_DELAY      30
#define REPEAT_INTERVAL   30

#define TICK_INTERVAL     10


SDL_Surface *screen = NULL;
int sdl_initialized = 0;

void reshape_gl(int width, int height)
{
  glViewport(0, 0, (GLsizei)(width), (GLsizei)(height));
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  gluPerspective(45.0f, (GLfloat)(width)/(GLfloat)(height), 1.0f, 200.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

void init_gl()
{
  glClearColor(0.0f, 0.0f, 0.0f, 0.5f);
  glClearDepth(1.0f);
  glDepthFunc(GL_LEQUAL);
  glEnable(GL_DEPTH_TEST);
  glShadeModel(GL_SMOOTH);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
}

void init_sdl()
{
  int flags;
  int width = 1280;
  int height = 800;
  
  if ( SDL_Init(SDL_INIT_TIMER|SDL_INIT_VIDEO) < 0 )
    fatal_error("Impossible d'initialiser SDL: %s\n", SDL_GetError());
  
  if (SDL_EnableKeyRepeat(REPEAT_DELAY, REPEAT_INTERVAL) == -1)
  {
    printf("Cannot set key repeat to %d, %d\n", REPEAT_DELAY, REPEAT_INTERVAL);
  }
  
  sdl_initialized = 1;

  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

  flags = 0;
  flags |= SDL_HWSURFACE;
  flags |= SDL_OPENGLBLIT;
  flags |= SDL_FULLSCREEN;
  
  SDL_WM_SetCaption("Wip", NULL);
  screen = SDL_SetVideoMode(width, height, 32, flags);
  if (!screen)
    fatal_error("Unable to set video mode");
  
  SDL_FillRect(screen, NULL, SDL_MapRGBA(screen->format, 0, 0, 0, 0));
  
  SDL_ShowCursor(SDL_DISABLE);
  
  reshape_gl(width, height);
  
  init_gl();
}

void shutdown_sdl()
{
  if (sdl_initialized)
  {
    SDL_ShowCursor(SDL_ENABLE);
    SDL_Quit();
  }
}

void process_sdl_events()
{
  SDL_Event event;
  
  while( SDL_PollEvent( &event ) )
  {
    switch( event.type ){
      case SDL_KEYDOWN:
        key_down_event(&event);
        break;

      case SDL_KEYUP:
        switch (event.key.keysym.sym)
        {
          case  SDLK_ESCAPE:
            shutdown_and_exit();
            break;
          case  SDLK_f:
            SDL_WM_ToggleFullScreen(screen);
          default:
            break;
        }
        break;
        
      case SDL_QUIT:
        shutdown_and_exit();
        break;

      case SDL_VIDEORESIZE:
        reshape_gl(event.resize.w, event.resize.h);
        break;
        
      default:
        break;
    }
  }

}

void load_texture(char *filename, GLuint *dest)
{
  SDL_Surface *image;

  if (!(image = SDL_LoadBMP(filename)))
    fatal_error("unable to load texture %s", filename);
  
  glGenTextures(1, dest);

  glBindTexture(GL_TEXTURE_2D, *dest);

  glTexImage2D( GL_TEXTURE_2D, 0, 3, image->w,
                image->h, 0, GL_BGR,
                GL_UNSIGNED_BYTE, image->pixels );

  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

  SDL_FreeSurface(image);
}

/**
 * TimeLeft returns time left to next frame
 **/
Uint32 TimeLeft(void)
{
  static Uint32 next_time = 0;
  Uint32 now;
  
  now = SDL_GetTicks();
  if ( next_time <= now ) {
    next_time = now + TICK_INTERVAL;
    return(0);
  }
  return(next_time - now);
}

void sdl_render()
{
  SDL_GL_SwapBuffers();
  SDL_Delay(TimeLeft());
}
