/*****************************************************************
|
|      XAUDIO SDK. Sample Output Module
|
|      (c) 1996-1998 MpegTV, LLC
|      Author: Gilles Boccon-Gibod (gilles@mpegtv.com)
|
 ****************************************************************/

/*
** This is a sample output module.
** This module shows a simple implementation of an output module
** that writes the audio PCM samples to a file on disk.
** It also prints messages or shows message windows for each of the
** functions called, to show the order in which the Xaudio library
** makes calls to the module's methods.
*/

/*----------------------------------------------------------------------
|       includes
+---------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "decoder.h"
#include "xaudio.h"

/*----------------------------------------------------------------------
|       macros
+---------------------------------------------------------------------*/
#ifdef WIN32
#include <windows.h>
#define TRACE(x) MessageBox(NULL, x, "Output Module", MB_OK)
#else
#define TRACE(x) printf("OUTPUT MODULE: " x "\n")
#endif

#define NOTIFY_DEBUG(device, level, message, reason)     \
XA_NOTIFY_DEBUG(&(device)->decoder->notification_client, \
                (device)->class_info->id,                \
                level, message, reason)

/*----------------------------------------------------------------------
|       types
+---------------------------------------------------------------------*/
typedef struct  {
    const char               *name;
    XA_OutputModuleClassInfo *class_info;
    XA_DecoderInfo           *decoder;
    FILE                     *file;
} FileOutput;

/*----------------------------------------------------------------------
|       output_module_probe
+---------------------------------------------------------------------*/
static int
output_module_probe(const char *name)
{
    TRACE("output_module_probe");

    /* accept all names */
    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_module_query
+---------------------------------------------------------------------*/
static int  
output_module_query(XA_OutputModuleQuery *query, 
                    unsigned long query_flags)
{
    TRACE("output_module_query");

    if (query_flags & XA_DECODER_OUTPUT_QUERY_MODULE_NAME) {
        strcpy(query->name, "Output To File");
        strcpy(query->description, "Version 1.0");
    }

    if (query_flags & XA_DECODER_OUTPUT_QUERY_NB_DEVICES) {
        query->index = 2;
    }
    
    /* if we're asked for the module name, stop here */
    if (query_flags & XA_DECODER_OUTPUT_QUERY_MODULE_NAME) return XA_SUCCESS;

    if (query_flags & XA_DECODER_OUTPUT_QUERY_DEVICE_NAME) {
        switch(query->index) {
          default: return XA_ERROR_INVALID_PARAMETERS;

          case 0:
            strcpy(query->name, "*, or raw:*");
            strcpy(query->description, "file name (raw PCM format)");
            query->flags = XA_DECODER_INPUT_QUERY_NAME_IS_GENERIC;
            break;

          case 2:
            strcpy(query->name, "-");
            strcpy(query->description, "standard output");
            query->flags = 0;
            break;
        }
    }
    
    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_new
+---------------------------------------------------------------------*/
static int
output_new(void **ret_output, const char *name,
           XA_OutputModuleClassInfo *class_info,
           XA_DecoderInfo *decoder)
{
    /* allocate a file object */
    FileOutput *output = (FileOutput *)malloc(sizeof(FileOutput));
    if (output == NULL) return XA_ERROR_OUT_OF_MEMORY;
    *ret_output = (void *)output;

    TRACE("output_new");

    /* init fields */
    output->file = NULL;

    /* store the parameters */
    output->class_info = class_info;
    output->decoder    = decoder;           

    /* parameter checking */
    if (!name || strlen(name) == 0) {
        NOTIFY_DEBUG(output, 0,
               "internal error", "null params or null/empty output name");
        free((void *)output);
        return XA_ERROR_INVALID_PARAMETERS;
    }
    
    /* store the name */
    output->name = strdup(name);

    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_delete
+---------------------------------------------------------------------*/
static int
output_delete(void *void_output)
{
    FileOutput *output = (FileOutput *)void_output;

    TRACE("output_delete");

    if (output) {
        free((void *)output->name);
        free((void *)output);
    }

    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_open
+---------------------------------------------------------------------*/
static int
output_open(void *void_output)
{
    FileOutput *output = (FileOutput *)void_output;
    const char *name   = output->name;

    TRACE("output_open");

    /* if the name is '-' it means stdout */
    if (name[0] == '-' && name[1] == '\0') {
        output->file = stdout;
    } else {
        /* copy the name and open the file */
        output->file = fopen(name, "wb");
        if (!output->file) {
            char *message = "cannot open output file";
            switch(errno) {
              case EACCES:
                NOTIFY_DEBUG(output, 0, message, "permission denied");
                return XA_ERROR_PERMISSION_DENIED;

              default:
                NOTIFY_DEBUG(output, 0, message, strerror(errno));
                return XA_ERROR_OPEN_FAILED;
            }
        }
    }
    
    return XA_SUCCESS;
}


/*----------------------------------------------------------------------
|       output_close
+---------------------------------------------------------------------*/
static int
output_close(void *void_output)
{
    FileOutput *output = (FileOutput *) void_output;

    TRACE("output_close");

    fclose(output->file);

    return XA_SUCCESS;
}

/*----------------------------------------------------------------------
|       output_write
+---------------------------------------------------------------------*/
static int
output_write(void *void_output, 
             void *buffer, 
             unsigned long size,
             unsigned int bytes_per_sample,
             unsigned int channels,
             unsigned int sample_rate)
{
    FileOutput *output = (FileOutput *) void_output;

    TRACE("output_write");

    if (fwrite(buffer, size, 1, output->file) <= 0) {
        return XA_FAILURE;
    }

    return size;
}

/*----------------------------------------------------------------------
|       output_module_register
+---------------------------------------------------------------------*/
int XA_EXPORT
output_module_register(XA_OutputModule *module)
{
    module->output_module_probe = output_module_probe;
    module->output_module_query = output_module_query;
    module->output_new          = output_new;
    module->output_delete       = output_delete;
    module->output_open         = output_open;
    module->output_close        = output_close;
    module->output_write        = output_write;
    module->output_get_buffer   = NULL;
    module->output_set_control  = NULL;
    module->output_get_control  = NULL;
    module->output_get_status   = NULL;
    module->output_get_caps     = NULL;

    return XA_SUCCESS;
}



