/*****************************************************************
|
|      MPEG audio decoder. Test Player
|
|      (c) 1996-1998 MpegTV, LLC
|
 ****************************************************************/

/*
** This sample application shows how to use the Xaudio SYNC API to
** decode a bitstreams that is in a memory buffer (or has been read
** by a different interface than one of Xaudio's Input Modules.
**
** The application will open a file, read data from it, and then store
** that data into a memory buffer that servers as the input stream for
** the decoding engine. Each time data is put into the buffer, the 
** application will ask the library to decode audio frames until it runs
** out of data in the input buffer. The the application reads more data
** from the file, stores it into the buffer, and starts over, repeating
** the same steps until the end of the file is reached.
** 
** NOTES:
** for each of the decoded audio frames, the application prints out
** the fields of the output audio buffer. A real-world application
** would do something more with the decoded PCM samples, but this is 
** just to show how to use the API.
*/

/*----------------------------------------------------------------------
|       includes
+---------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>

/* xaudio general includes */
#include "decoder.h"

/* xaudio plug-in modules */
#include "memory_input.h"
#include "audio_output.h"
#include "mpeg_codec.h"

/*----------------------------------------------------------------------
|       constants
+---------------------------------------------------------------------*/
#define PCM_BUFFER_SIZE     (1152*4)
#define MP3_BUFFER_SIZE     (4096)

/*----------------------------------------------------------------------
|       error
+---------------------------------------------------------------------*/
static void
error(const char *string, int code)
{
    fprintf(stderr, "%s [error %d: %s]\n",
            string, code, xaudio_error_string(code));
    exit(1);
}

/*----------------------------------------------------------------------
|       main
+---------------------------------------------------------------------*/
int 
main(int argc, char **argv)
{
    XA_DecoderInfo *decoder;
    unsigned char  *mp3_buffer;
    unsigned char  *pcm_buffer;
    FILE           *file;
    int             status; 

    /* allocate a buffer for the MP3 stream */
    mp3_buffer = (unsigned char *)malloc(MP3_BUFFER_SIZE);
    
    /* allocate a buffer for the PCM samples */
    pcm_buffer = (unsigned char *)malloc(PCM_BUFFER_SIZE);
    
    /* create a decoder */
    if (decoder_new(&decoder) != XA_SUCCESS) {
        fprintf(stderr, "cannot create decoder");
        exit(1);
    }
    
    /* register mpeg audio codec */
    {
        XA_CodecModule module;

        mpeg_codec_module_register(&module);
        decoder_codec_module_register(decoder, &module);
    }

    /* register the memory input module */
    {
        XA_InputModule module;

        memory_input_module_register(&module);
        decoder_input_module_register(decoder, &module);
    }

    /* open the mp3 file (name passed as program argument */
    file = fopen(argv[1], "rb");
    if (file == NULL) {
        fprintf(stderr, "cannot open input file\n");
        exit(1);
    }

    /* create and open input object */
    status = decoder_input_new(decoder, NULL, XA_DECODER_INPUT_AUTOSELECT);
    if (status != XA_SUCCESS) {
        error("cannot create input [%d]\n", status);
    }
    if (decoder_input_open(decoder) != XA_SUCCESS) {
        printf("cannot open input\n");
        exit(1);
    }

    /* loop until there is an error */
    do {
        int nb_read;

        /* read some bytes from the file */
        nb_read = fread(mp3_buffer, 1, MP3_BUFFER_SIZE, file);
        if (nb_read == 0) {
            if (feof(file)) {
                fprintf(stderr, "end of file\n");
                exit(0);
            } else {
                fprintf(stderr, "cannot read bytes from input\n");
                exit(1);
            }
        }
        
        /* feed the input buffer */
        memory_input_feed(decoder->input->device, 
                          mp3_buffer, 
                          nb_read);

        /* keep decoding frames until there is an error or the input buffer
           runs out of bytes (XA_ERROR_TIMEOUT) */
        do {
            status = decoder_decode(decoder, pcm_buffer);
            if (status == XA_SUCCESS) {
                printf("decoded buffer:\n");
                printf("changed          = %d\n", decoder->output_buffer->changed);
                printf("pcm_samples addr = %lx\n", 
                       (unsigned long)decoder->output_buffer->pcm_samples);
                printf("size             = %d\n", decoder->output_buffer->size);
                printf("bytes_per_sample = %d\n", decoder->output_buffer->bytes_per_sample);
                printf("channels         = %d\n", decoder->output_buffer->channels);
                printf("sample_rate      = %d\n", decoder->output_buffer->sample_rate);
            }
        } while(status == XA_SUCCESS ||
                status == XA_ERROR_INVALID_FRAME);
    } while(status == XA_SUCCESS ||
            status == XA_ERROR_TIMEOUT);

    return 0;
}
