/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include <stdexcept>
#include "animation.h"

animation_data::animation_data(unsigned int _w, unsigned int _h, bool _dyn,
		sf::Color _dyn_col, sf::Color _dyn_col_range) : w(_w), h(_h),
		dyn(_dyn), dyn_col(_dyn_col), dyn_col_range(_dyn_col_range)
{
}

animation_data::~animation_data()
{
	std::map<std::string, animation_mode>::iterator iter;
	for(iter = modes.begin(); iter != modes.end(); iter++) {
		for(unsigned int a = 0; a < iter->second.frames.size(); a++) {
			delete iter->second.frames[a];
		}
	}
}
	
void animation_data::add_animation(std::string name, animation_mode anim)
{
	modes.insert(std::make_pair(name, anim));
}

unsigned int animation_data::get_w()
{
	return w;
}

unsigned int animation_data::get_h()
{
	return h;
}


const std::map<std::string, animation_data::animation_mode> &animation_data::get_frames()
{
	return modes;
}

animation::animation(animation_data& _data) : data(_data), frame(0), mode(data.get_frames().end()),
	man(false), regen_dyn_img(true)
{ 
}

void animation::set_mode(std::string mode, bool repeat)
{ 
	animation_data::mode_iter iter = data.get_frames().find(mode);
	if(iter == data.get_frames().end()) {
		throw std::invalid_argument("Mode \"" + mode + "\" doesn't exist");
	} else {
		if(this->mode != iter) {
			regen_dyn_img = true;
		}
		frame = 0;
		this->repeat = repeat;
		this->mode = iter; 
	}
}

void animation::set_frame(unsigned int fr)
{
	if(mode->second.frames.size() <= fr) 
		return;
	if(frame != fr)
		regen_dyn_img = true;
	frame = fr;
}

void animation::calc_color(sf::Color &c)
{
	sf::Color col = data.get_dyn_color();
	int t;

	t = c.r;
	t -= col.r;
	t += color.r;
	if(t < 0)
		t = 0;
	c.r = t;

	t = c.g;
	t -= col.g;
	t += color.g;
	if(t < 0)
		t = 0;
	c.g = t;

	t = c.b;
	t -= col.b;
	t += color.b;
	if(t < 0)
		t = 0;
	c.b = t;
}

void animation::set_dynamic_color(sf::Color col)
{
	if(color != col)
	{
		color = col;
		regen_dyn_img = true;
	}
}

void animation::gen_dyn_img()
{
	if(!data.is_dyn())
		return;

	const animation_data::animation_mode &anim = mode->second;
	sf::Image &fr = *anim.frames[frame];
	std::vector<Uint32> buffer;
	buffer.resize(get_w() * get_h());
	Uint32 * buf_ptr = buffer.data();

	sf::Color col = data.get_dyn_color();
	sf::Color range = data.get_dyn_color_range();

	std::memcpy(buf_ptr, fr.GetPixelsPtr(), get_w() * get_h() * 4);
	for(unsigned int y = 0; y < get_h(); y++) {
		for(unsigned int x = 0; x < get_w(); x++) {
			sf::Color p = uint_to_c(buf_ptr[x + get_w() * y]);

			if(absdiff(p.g, col.g) <= range.g &&
				absdiff(p.r, col.r) <= range.r &&
				absdiff(p.b, col.b) <= range.b) {
				calc_color(p);

				buf_ptr[x + get_w() * y] = c_to_uint(p);
			}
		}
	}

	dyn_img.LoadFromPixels(get_w(), get_h(), reinterpret_cast<Uint8*>(buf_ptr));
	regen_dyn_img = false;
}

const sf::Image & animation::get_frame()
{
	if(mode == data.get_frames().end())
		throw std::invalid_argument("Working mode hasn't been set");
	const animation_data::animation_mode &anim = mode->second;

	if(!man && anim.frames.size() != 1 && anim.frame_change_time < timer.get_time()*1000.0f) {
		frame++;
		if(frame >= anim.frames.size()) {
			if(!repeat) 
				frame = anim.frames.size() - 1;
			else
				frame = 0;
		}
		timer.reset();
		regen_dyn_img = true;
	}

	if(regen_dyn_img) { 
		gen_dyn_img();
	}

	if(anim.frames.size() == 0)
		throw std::invalid_argument("Mode doesn't contain any frames");
	else if(data.is_dyn())
		return dyn_img;
	else
		return *anim.frames[frame];
}

