/***************************************************************************
*   Copyright (C) 2010 by 
*    Kai Lindholm <megantti@gmail.com>
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
***************************************************************************/

#ifndef _ANIMATION_H
#define _ANIMATION_H

#ifndef NO_GRAPHICS
#include <SFML/Graphics.hpp>
#include <SFML/System.hpp>
#endif

#include <vector>
#include <map>
#include <string>
#include <memory>
#include <cstring>
#include <cmath>
#include "pauseclock.h"

#ifndef NO_GRAPHICS

using sf::Uint32;
using sf::Uint8;

class animation_data {
	public:
		animation_data(unsigned int w, unsigned int h, bool dyn, 
				sf::Color dyn_col, sf::Color dyn_color_range);
		~animation_data();

		struct animation_mode {
			std::vector<sf::Image*> frames; 
			Uint32 frame_change_time;
		};

		void add_animation(std::string name, animation_mode);

		unsigned int get_w();
		unsigned int get_h();
		const std::map<std::string, animation_mode> &get_frames();

		typedef std::map<std::string, animation_mode>::const_iterator mode_iter;

		bool is_dyn() { return dyn; }
		sf::Color get_dyn_color() { return dyn_col; }
		sf::Color get_dyn_color_range() { return dyn_col_range; }
	private:
		unsigned int w, h;
		bool dyn;
		sf::Color dyn_col;
		sf::Color dyn_col_range;
		std::map<std::string, animation_mode> modes;
};

/**
* Wraps animation_data so that mutable data about state is stored here.
* For example can be used for single creatures.
*/

class animation {
	public:
		animation(animation_data&);

		const sf::Image& get_frame();

		void set_mode(std::string mode, bool repeat = true);

		void set_frame(unsigned int fr);
		unsigned int get_frame_count() { return mode->second.frames.size(); }
		unsigned int get_w() const { return data.get_w(); }
		unsigned int get_h() const { return data.get_h(); }

		void pause() { timer.pause(); }
		void unpause() { timer.unpause(); }
		void update() { timer.step(); }
		void manual() { man = true; }
		void automatic() { man = false; }
		void set_dynamic_color(sf::Color col);
	private:
		void gen_dyn_img();
		void calc_color(sf::Color &c);

		bool repeat;
		pclock timer;
		animation_data& data;
		unsigned int frame;
		animation_data::mode_iter mode;
		bool man;

		sf::Image dyn_img;
		sf::Color color;
		bool regen_dyn_img;
};

inline Uint32 c_to_uint(sf::Color col)
{
	return col.r | (col.g << 8) | (col.b << 16) | (col.a << 24);
}

inline sf::Color uint_to_c(Uint32 col)
{
	return sf::Color(col & 0xFF, (col >> 8) & 0xFF, (col >> 16) & 0xFF,
			(col >> 24) & 0xFF);
}

inline Uint8 absdiff(Uint8 a, Uint8 b) 
{
	if(a > b) {
		return a - b;
	} else {
		return b -a;
	}
}

#else 

class animation_data {

};

class animation {
	public:
		animation(animation_data&) { }
		void set_mode(std::string mode, bool repeat = false) {}
		void set_frame(unsigned int fr) {}
		unsigned int get_frame_count() { return 0; }
		unsigned int get_w() const { return 0; }
		unsigned int get_h() const { return 0; }

		void pause() { }
		void unpause() { }
		void update() { }
		void manual() { }
		void automatic() { }
};

#endif

#endif
