/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include <SFML/Graphics.hpp>
#include <stdexcept>
#include "animationloader.h"

animation_loader::animation_loader()
{
	std::vector<anim_desc> descs;
	
#if 0
	// example: 
	descs.push_back(anim_desc("mushroom", "mushroom.png", 32, 32).
		add(adesc("grow", 4, 10)). // grow has frames 1-4 (4 frames, change frame after 10ms)
		add(adesc("stand", 1)). // frame 5 (1 frame)
		add(adesc("die", 4, 10)) // frames 6-9 (4 frames)
	);
#endif

	//
	// map stuff
	//

	descs.push_back(anim_desc("goal", "gate.png", 128, 128).
		add(adesc("stand", 20, 70))
	);
	descs.push_back(anim_desc("mushroom", "mushroom.png", 64, 64, true,
				sf::Color(0xFF, 0, 0xFF), sf::Color(90, 10, 90)).
		add(adesc("small", 5, 250)).
		add(adesc("stand", 5, 250)).
		add(adesc("die", 4, 200))
	);

	descs.push_back(anim_desc("ant_heap", "ant_heap.png", 64, 64).
		add(adesc("stand", 4, 200))
	);
	
	const char *cc_imgs[] = {
		"command_center.png",
		"command_center2.png",
		"command_center3.png"
	};

	int n = double(rand()) / double(RAND_MAX) * 3.0;
	if(n >= 3)
		n = 2;

	descs.push_back(anim_desc("command_center", cc_imgs[n], 128, 128).
		add(adesc("stand", 1))
	);

	descs.push_back(anim_desc("grass", "grass.png", 1024, 652).
		add(adesc("stand", 1))
	);

	descs.push_back(anim_desc("wall", "wall.png", 32, 32).
		add(adesc("stand", 1)).
		add(adesc("broken1", 1)).
		add(adesc("broken2", 1)).
		add(adesc("broken3", 1))
	);


	//
	// character animations
	//
	
	int unicorn_move_time = 50;
	int minotaur_move_time = 140;
	int griffin_move_time = 50;
	int ant_move_time = 50;

	descs.push_back(anim_desc("unicorn", "unicorn.png", 64, 64).
		add(adesc("stand_e", 1)).
		add(adesc("move_e", 8, unicorn_move_time)).
		
		add(adesc("stand_ne", 1)).
		add(adesc("move_ne", 8, unicorn_move_time)).
		
		add(adesc("stand_n", 1)).
		add(adesc("move_n", 8, unicorn_move_time)).
		
		add(adesc("stand_nw", 1)).
		add(adesc("move_nw", 8, unicorn_move_time)).
		
		add(adesc("stand_w", 1)).
		add(adesc("move_w", 8, unicorn_move_time)).
		
		add(adesc("stand_sw", 1)).
		add(adesc("move_sw", 8, unicorn_move_time)).
		
		add(adesc("stand_s", 1)).
		add(adesc("move_s", 8, unicorn_move_time)).
		
		add(adesc("stand_se", 1)).
		add(adesc("move_se", 8, unicorn_move_time))
	);

	descs.push_back(anim_desc("minotaur", "minotaur.png", 64, 64).
		add(adesc("stand_e", 1)).
		add(adesc("move_e", 9, minotaur_move_time)).

		add(adesc("stand_ne", 1)).
		add(adesc("move_ne", 9, minotaur_move_time)).

		add(adesc("stand_n", 1)).
		add(adesc("move_n", 9, minotaur_move_time)).

		add(adesc("stand_nw", 1)).
		add(adesc("move_nw", 9, minotaur_move_time)).

		add(adesc("stand_w", 1)).
		add(adesc("move_w", 9, minotaur_move_time)).

		add(adesc("stand_sw", 1)).
		add(adesc("move_sw", 9, minotaur_move_time)).

		add(adesc("stand_s", 1)).
		add(adesc("move_s", 9, minotaur_move_time)).

		add(adesc("stand_se", 1)).
		add(adesc("move_se", 9, minotaur_move_time))
	);

	descs.push_back(anim_desc("griffin", "griffin.png", 64, 64).
		add(adesc("stand_e", 10, griffin_move_time)).
		add(adesc("move_e", 10, griffin_move_time)).
		
		add(adesc("stand_ne", 10, griffin_move_time)).
		add(adesc("move_ne", 10, griffin_move_time)).
		
		add(adesc("stand_n", 10, griffin_move_time)).
		add(adesc("move_n", 10, griffin_move_time)).
		
		add(adesc("stand_nw", 10, griffin_move_time)).
		add(adesc("move_nw", 10, griffin_move_time)).
		
		add(adesc("stand_w", 10, griffin_move_time)).
		add(adesc("move_w", 10, griffin_move_time)).
		
		add(adesc("stand_sw", 10, griffin_move_time)).
		add(adesc("move_sw", 10, griffin_move_time)).
		
		add(adesc("stand_s", 10, griffin_move_time)).
		add(adesc("move_s", 10, griffin_move_time)).
		
		add(adesc("stand_se", 10, griffin_move_time)).
		add(adesc("move_se", 10, griffin_move_time))
	);



	descs.push_back(anim_desc("ant", "ant.png", 64, 64).
		add(adesc("stand_e", 1)).
		add(adesc("move_e", 8, ant_move_time)).

		add(adesc("stand_ne", 1)).
		add(adesc("move_ne", 8, ant_move_time)).

		add(adesc("stand_n", 1)).
		add(adesc("move_n", 8, ant_move_time)).

		add(adesc("stand_nw", 1)).
		add(adesc("move_nw", 8, ant_move_time)).

		add(adesc("stand_w", 1)).
		add(adesc("move_w", 8, ant_move_time)).

		add(adesc("stand_sw", 1)).
		add(adesc("move_sw", 8, ant_move_time)).

		add(adesc("stand_s", 1)).
		add(adesc("move_s", 8, ant_move_time)).

		add(adesc("stand_se", 1)).
		add(adesc("move_se", 8, ant_move_time)).
		add(adesc("paralyse_e", 2, 20)).
		add(adesc("paralyse_ne", 2, 20)).
		add(adesc("paralyse_n", 2, 20)).
		add(adesc("paralyse_nw", 2, 20)).
		add(adesc("paralyse_w", 2, 20)).
		add(adesc("paralyse_sw", 2, 20)).
		add(adesc("paralyse_s", 2, 20)).
		add(adesc("paralyse_se", 2, 20))
	);

	//
	// spell icons
	//

	descs.push_back(anim_desc("grow_mushroom", "grow_mushroom.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("upg_to_attack", "upg_to_attack.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("upg_to_spore", "upg_to_spore.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("upg_to_tank", "upg_to_tank.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("fertilize_str", "fertilize_str.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("fertilize_mana", "fertilize_mana.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("fertilize_dur", "fertilize_dur.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("eat_mushroom", "eat_mushroom.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("connect_mushroom", "connect_mushroom.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("dimension_door", "dimension_door.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("invisibility", "invisibility.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("mana_orb", "mana_orb.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("heal", "heal.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("taunt", "taunt.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("attack_wall", "mushroom_attack_wall.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("minotaur_attack_wall", "minotaur_attack_wall.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("target", "target.png", 32, 32).
		add(adesc("icon", 1))
	);
	descs.push_back(anim_desc("paralyse", "paralyse.png", 32, 32).
		add(adesc("icon", 1))
	);
	descs.push_back(anim_desc("resurrect", "resurrect.png", 32, 32).
		add(adesc("icon", 1))
	);
	descs.push_back(anim_desc("diagonal_spawn", "diagonal_spawn.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("explosive_trap_icon", "explosive_trap_icon.png", 32, 32).
		add(adesc("icon", 1))
	);

	descs.push_back(anim_desc("paralyse_trap_icon", "paralyse_trap_icon.png", 32, 32).
		add(adesc("icon", 1))
	);
	
	descs.push_back(anim_desc("kill_ant", "kill_ant.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("boost_stats", "boost_stats.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("immortality", "immortality.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("attack_stance", "attack_stance.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("throw_mushroom", "throw_mushroom.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("build_cc", "build_cc.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("mass_connect", "mass_connect.png", 32, 32).
		add(adesc("icon", 1))
	);	
	
	descs.push_back(anim_desc("ants_bane", "ants_bane.png", 32, 32).
		add(adesc("icon", 1))
	);	

	// misc
	
	descs.push_back(anim_desc("explosive_trap", "explosive_trap.png", 32, 32).
		add(adesc("stand", 1)).
		add(adesc("triggered", 2, 50))
	);
	
	descs.push_back(anim_desc("paralyse_trap", "paralyse_trap.png", 32, 32).
		add(adesc("stand", 1)).
		add(adesc("triggered", 2, 50))
	);

	descs.push_back(anim_desc("butterfly", "butterfly.png", 8, 8).
		add(adesc("right", 2, 100)).
		add(adesc("left", 2, 100))
	);
		

	for(unsigned int a = 0; a < descs.size(); a++) {
		load_animation(descs[a]);
	}
}

animation_loader::~animation_loader()
{
	for(std::map<std::string, animation_data*>::iterator iter = anims.begin(); iter != anims.end(); iter++) {
		delete iter->second;
	}

}

animation_data& animation_loader::get_animation(std::string name)
{
	std::map<std::string, animation_data*>::iterator iter = anims.find(name);
	if(iter != anims.end()) {
		return *iter->second;
	} else {
		throw std::invalid_argument("Animation \"" + name + "\" doesn't exist");
	}
}

void animation_loader::load_animation(anim_desc desc)
{
	sf::Image img;

	animation_data *data = new animation_data(desc.w, desc.h, desc.dyn, desc.dyn_col, desc.dyn_col_range);
	try {
		if(!img.LoadFromFile("gfx/" + desc.file)) {
			throw std::invalid_argument("Couldn't load animation file \"" + desc.file + "\"");
		}

		unsigned int anim_col = 0;
		unsigned int anim_row = 0;

		for(unsigned int i = 0; i < desc.anim_parts.size(); i++) {
			adesc& anim_part = desc.anim_parts[i];
			const unsigned int frames = anim_part.length;
			animation_data::animation_mode amode;
			amode.frame_change_time = anim_part.chtime;
			
			for(unsigned int a = 0; a < frames; a++) {
				sf::Image *frame = new sf::Image(desc.w, desc.h);
				if(img.GetWidth() < desc.w * (anim_col + 1)) {
					anim_col = 0;
					anim_row++;
				}

				if(img.GetHeight() < desc.h * (anim_row + 1)) {
					throw std::invalid_argument("Animation file \"" + desc.file + "\" is too small");
				}

				frame->Copy(img, 0, 0,
								sf::IntRect(desc.w * anim_col, 
											desc.h * anim_row, 
											desc.w * (1 + anim_col), 
											desc.h * (1 + anim_row)));

				++anim_col;
				amode.frames.push_back(frame);
			}

			data->add_animation(anim_part.name, amode);
		}
	} catch (...) {
		delete data;
		throw;
	}

	anims.insert(std::make_pair(desc.name, data));
}
