/***************************************************************************
*   Copyright (C) 2009-2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include <iostream>
#include "gui.h"

namespace Gui {	
	ID Object::id_counter = 0;
	sf::Font &Object::font_default = const_cast<sf::Font&>(sf::Font::GetDefaultFont());
	int Object::count = 0;

	void SetFont(sf::Font &_font_default)
	{ 
		Object::SetDefaultFont(_font_default);
	}
	
	MouseButton MouseConv(sf::Mouse::Button i)
	{
		if(i == sf::Mouse::Left)
			return BUTTON1;
		else if(i == sf::Mouse::Right)
			return BUTTON2;
		return BUTTON1;
	}

	void drawFilledRectangle(sf::RenderTarget &target, const Rect &r, double p, sf::Color color)
	{
		if(p < 0)
			p = 0;
		if(p > 1)
			p = 1;

		if(r.h == 0 || r.w == 0)
			return;

		sf::Shape r1 = sf::Shape::Rectangle(r.x, r.y, r.x+r.w, r.y+r.h, color, 1.0f, color);
		r1.EnableFill(false);
		sf::Shape r2 = sf::Shape::Rectangle(r.x, r.y, r.x+r.w*p, r.y+r.h, color);

		target.Draw(r1);
		target.Draw(r2);
	}

	unsigned int TextWidth(const std::string text, const sf::Font &font, float size)
	{
		sf::String str(text, font);
		str.SetSize(size);
		return static_cast<int>(str.GetRect().Right - str.GetRect().Left);
	}

	unsigned int TextHeight(const std::string text, const sf::Font &font)
	{
		return font.GetCharacterSize(); 
	}

	ActionListener::~ActionListener()
	{

	}

			
	const Rect & Rect::operator=(const Coor &c)
	{ 
		x = c.x; 
		y = c.y; 
		return *this;
	}

	const Rect & Rect::operator=(const Rect &c)
	{ 
		x = c.x; 
		y = c.y; 
		w = c.w; 
		h = c.h; 
		return *this;
	}

	const Coor Object::GetAbsCoor() const
	{
		if(!parent)
			return Coor(0, 0);
		else
			return parent->GetAbsCoor() + rect;
	}
	
	void Object::ResizeAction()
	{
		if(resize_handling == true)
			return;
		resize_handling = true;
		ResizeHandler();
		resize_handling = false;
	}

	void Object::Focus()
	{
		if(parent)
			parent->SetFocus(id);
	}

	Coor Object::GetMousePos() const
	{
		if(parent)
			return parent->GetMousePos();
		else
			return Coor();
	}

	bool Object::IsUnderMouse() const
	{
		Rect r = GetAbsCoor();
		r.w = rect.w;
		r.h = rect.h;
		return GetMousePos().InRect(r);
	}

	void Object::MouseMotionAction(int x, int y, int dx, int dy, const sf::Input&)
	{
		ActionListener *l = GetListener();
		if(l)
			l->MouseOver(id);
	}

	void Object::KeyDownAction(Key k)
	{ 
		ActionListener *l = GetListener();
		if(l)
			l->KeyDown(id, k);
	}
	
	void Object::DrawBackground(sf::RenderTarget &target, const Coor coor) const
	{
		target.Draw(sf::Shape::Rectangle(rect.x+coor.x, rect.y+coor.y, rect.x+coor.x+rect.w, rect.y+coor.y+rect.h, color));
	}

	void Object::DrawBorder(sf::RenderTarget &target, const sf::Color _color, int border) const
	{
		const sf::Color col = _color;
		const unsigned int x1 = GetRelX();
		const unsigned int x2 = x1 + GetW() + 1; 
		const unsigned int y1 = GetRelY();
		const unsigned int y2 = y1 + GetH() + 1; 

		target.Draw(sf::Shape::Line(x1, y1, x2, y1, border/2, col));
		target.Draw(sf::Shape::Line(x2, y1, x2, y2, border/2, col));
		target.Draw(sf::Shape::Line(x1, y2, x2, y2, border/2, col));
		target.Draw(sf::Shape::Line(x1, y1, x1, y2, border/2, col));
	}
	
	void Object::Expand(int dx, int dy) const
	{
		rect.x -= dx / 2;
		rect.w += dx;
		rect.y -= dy / 2;
		rect.h += dy;
	}

	void Object::Raise()
	{
		if(parent)
			parent->RaiseID(id);
	}

	void Object::HandleLogic()
	{
	}

	void Object::SetTip(Tooltip &t)
	{
		if(parent)
			parent->SetTip(t);
	}

	void Object::UnsetTip()
	{
		if(parent)
			parent->UnsetTip();
	}

	Object::~Object()
	{
		count--; 
		if(parent)
			parent->DestroyID(id);
	} 

	Container::~Container()
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			(*it)->SetParent(NULL);
			delete (*it);
		}
	}
		
	Container *Container::Add(Object *o)
	{
		o->SetParent(this);
		o->ResizeAction();
		childs.push_back(o);
		return this;
	}

	Container *Container::Add(Object *o, int x, int y)
	{ 
		o->SetParent(this);
		o->SetRect(Rect(x, y, std::min(x+o->GetW(), GetW()) - x, std::min(y+o->GetH(), GetH()) - y));
		childs.push_back(o);
		return this;
	}

	Container *Container::AddCenter(Object *o)
	{
		o->SetParent(this);
		o->SetRelX((GetW() - o->GetW()) / 2);
		o->SetRelY((GetH() - o->GetH()) / 2);
		childs.push_back(o);
		return this;
	}
	
	void Container::SetFocus(ID id)
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if((*it)->GetID() == id) {
				focus = id;
				focusit = it;
				SetFocus(this->id);
				(*focusit)->FocusAction();
				break;
			}
		}
	}
			
	void Container::UnFocus()
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if((*it)->GetID() == focus) {
				focus = ID_NONE;
				Container *c = dynamic_cast<Container*>(*it);
				if(c)
					c->UnFocus();
				(*it)->UnFocusAction();
				break;
			}
		}
	}

	Coor Container::GetMousePos() const
	{
		if(parent)
			return parent->GetMousePos();
		else
			return Coor(mousex, mousey);
	}
			
	void Container::Render(sf::RenderTarget &target) const
	{
		RenderCoor(target);
	}

	void Container::RenderCoor(sf::RenderTarget &target, const Coor coor) const
	{
		if(!hide) {
			if(drawbg)
				target.Draw(sf::Shape::Rectangle(rect.x+coor.x, rect.y+coor.y, rect.x+coor.x+rect.w, rect.y+coor.y+rect.h, color));
			Expand(2,2);
			DrawBorder(target);
			Expand(-2, -2);
			for(object_c_iter_r it = childs.rbegin(); it != childs.rend(); it++) {
				(*it)->SetPosition((Coor(rect)+coor).sfVector());
				target.Draw(**it);
			}
			if(tip)
				target.Draw(*tip);
		}
	}

	void Container::MouseDown(int x, int y, MouseButton m)
	{
		mousex = x;
		mousey = y;
		MouseDownAction(x, y, m);
	}

	void Container::MouseUp(int x, int y, MouseButton m)
	{
		mousex = x;
		mousey = y;
		MouseUpAction(x, y, m);
	}

	void Container::MouseMotion(int x, int y, const sf::Input &inp)
	{
		int dx = x - mousex;
		int dy = y - mousey;
		mousex = x;
		mousey = y;
		MouseMotionAction(x, y, dx, dy, inp);
	}

	void Container::FocusAction()
	{
		focused = true;
	}

	void Container::UnFocusAction()
	{
		focused = false;
	}

	void Container::HandleLogic()
	{
		Object::HandleLogic();
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if(!(*it)->IsHidden()) {
				if(lock == ID_NONE || (*it)->GetID() == lock) {
					(*it)->HandleLogic();
				}
			}
		}
	}

	void Container::SetTip(Tooltip &t)
	{
		if(parent) {
			parent->SetTip(t);
		} else {
			if(!tip) {
				tip = new TextList(Coor(mousex, mousey), "", 0);
				tip->SetMaxTexts(4);
				tip->SetSize(15);
				tip->SetText(t.str);
				tip->DrawBG(true);
				tip->SetColor(t.bgcolor);
				tip->SetFontColor(t.color);
			} else {
				tip->SetCoor(Coor(mousex, mousey));
			}
		}
	}

	void Container::UnsetTip()
	{
		if(parent) {
			parent->UnsetTip();
		} else {
			delete tip;
			tip = NULL;
		}
	}

	void Container::ResizeHandler()
	{
		// Calculate new stuff?
	}
	
	void Container::Clear() 
	{
		UnFocus();
		while(childs.size()) {
			childs.front()->SetParent(NULL);
			delete (childs.front());
			childs.erase(childs.begin());
		}
	}

	void Container::ClearWithoutDestroy() 
	{
		UnFocus();
		while(childs.size()) {
			childs.front()->SetParent(NULL);
			childs.erase(childs.begin());
		}
	}

	void Container::MouseDownAction(int x, int y, MouseButton m)
	{
		if(focus != ID_NONE)
		{
			UnFocus();
		}

		x -= GetRelX();
		y -= GetRelY();
		object_iter it;
		for(it = childs.begin(); it != childs.end(); it++) {
			if(!(*it)->IsHidden() && (*it)->IsEnabled())
			if(lock == ID_NONE || (*it)->GetID() == lock)
			if((*it)->GetRelX() < x && x < (*it)->GetRelX() + (*it)->GetW() &&
				(*it)->GetRelY() < y && y < (*it)->GetRelY() + (*it)->GetH()) {
				focus = (*it)->GetID();
				focusit = it;
				(*it)->FocusAction();
				(*it)->MouseDownAction(x, y, m);
				break;
			}
		}
		if(it == childs.end()) {
			ActionListener *l = GetListener();
			l->MouseClick(id, x, y, m);
		}
	}
			
	void Container::MouseUpAction(int x, int y, MouseButton m)
	{
		x -= GetRelX();
		y -= GetRelY();
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if(!(*it)->IsHidden() && (*it)->IsEnabled())
			if(lock == ID_NONE || (*it)->GetID() == lock)
			if((*it)->GetRelX() < x && x < (*it)->GetRelX() + (*it)->GetW() &&
				(*it)->GetRelY() < y && y < (*it)->GetRelY() + (*it)->GetH()) {
				(*it)->MouseUpAction(x, y, m);
			}
		}
	}

	void Container::MouseMotionAction(int x, int y, int dx, int dy, const sf::Input &inp)
	{
		x -= GetRelX();
		y -= GetRelY();
		if(focus != ID_NONE) {
			(*focusit)->MouseMotionAction(x, y, dx, dy, inp);
		} else {
			for(object_iter it = childs.begin(); it != childs.end(); it++) {
				if(!(*it)->IsHidden() && (*it)->IsEnabled())
				if(lock == ID_NONE || (*it)->GetID() == lock)
				if((*it)->GetRelX() < x && x < (*it)->GetRelX() + (*it)->GetW() &&
				(*it)->GetRelY() < y && y < (*it)->GetRelY() + (*it)->GetH()) {
					(*it)->MouseMotionAction(x, y, dx, dy, inp);
				}
			}
		}
	}

	void Container::KeyDownAction(Key k)
	{
		if(focus != ID_NONE && (*focusit)->IsEnabled())
			(*focusit)->KeyDownAction(k);
		else
			Object::KeyDownAction(k);
	}

	void Container::TextAction(sf::Uint32 k)
	{
		if(focus != ID_NONE && (*focusit)->IsEnabled())
			(*focusit)->TextAction(k);
		else
			Object::TextAction(k);
	}

	void Container::RaiseID(ID i)
	{
		hide = false;
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if((*it)->GetID() == i) {
				Object *o = *it;
				childs.erase(it);
				childs.insert(childs.begin(), o);
				break;
			}
		}
		if(parent) {
			parent->RaiseID(id);
		}
	}

	void Container::Zip()
	{
		Coor min = Coor(-1, -1), max = Coor(GetW(), GetH());
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			min.x = std::min((*it)->GetRelX(), min.x);
			min.y = std::min((*it)->GetRelY(), min.y);
			max.x = std::max((*it)->GetRelX(), max.x);
			max.y = std::max((*it)->GetRelY(), max.y);
		}
		rect = Rect(min.x + rect.x, min.y + rect.y, max.x - min.x, max.y - min.y);
	}	
			
	void Container::DestroyID(ID id)
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if((*it)->GetID() == id) {
				childs.erase(it);
				break;
			}
		}
		if(focus == id)
			focus = ID_NONE;
		else
			SetFocus(focus);
		ResizeAction();
	}

	void Container::DeleteID(ID id)
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			Object *o = *it;
			if(o->GetID() == id) {
				o->SetParent(NULL);
				delete o;
				childs.erase(it);
				break;
			}
		}
		if(focus == id)
			focus = ID_NONE;
		else
			SetFocus(focus);
		ResizeAction();
	}
			
	Object *Container::GetObject(ID id)
	{
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			if((*it)->GetID() == id)
				return *it;
			Container *c = dynamic_cast<Container*>(*it);
			if(c) {
				Object *o = c->GetObject(id);
				if(o != NULL)
					return o;
			}
		}
		return NULL;
	}

	ID Container::FocusNext(bool focusparent)
	{
		// if something is focused
		if(focus != ID_NONE) {
			// if it is container
			Container *childcont = dynamic_cast<Container*>(*focusit);
			if(childcont && !childcont->FocusAsObject()) {
				// ask the child container to focus next and continue if it has nothing to focus
				ID i = childcont->FocusNext();
				if(i != ID_NONE) {
					return i;
				}
			}
			// take the focus away
			(*focusit)->UnFocusAction();
			// get the next child
			focusit++;
		} else {
			// we will start from the beginning because nothing was focused
			focusit = childs.begin();
		}

		// if we are at the end
		if(focusit == childs.end()) {
			focusit = childs.begin();
			// if we have a parent tell that we have nothing to focus
			if(parent && focusparent) {
				focus = ID_NONE;
				return ID_NONE;
			}
		}

		// if the child doesn't want to be focused let's try to focus again
		focus = (*focusit)->GetID();
		if((*focusit)->skip_focus || !(*focusit)->IsEnabled())
			return FocusNext(focusparent);

		// now we have child to focus
		(*focusit)->FocusAction();

		// if it is a container let's ask it to focus something
		Container *c = dynamic_cast<Container*>(*focusit);
		if(c && !c->FocusAsObject()) {
			c->FocusNone();
			ID i = c->FocusNext();
			if(i == ID_NONE)
				return FocusNext(focusparent);
			else
				return i;
		}

		return focus;
	}

	ID Container::FocusPrev(bool focusparent)
	{
		if(focus != ID_NONE) {
			// if we have a focused child container let's ask it to focus next
			Container *childcont = dynamic_cast<Container*>(*focusit);
			if(childcont && !childcont->FocusAsObject()) {
				ID i = childcont->FocusPrev();
				if(i != ID_NONE) {
					return i;
				}
			}
			// if it didn't work out let's take the focus 
			(*focusit)->UnFocusAction();
		} else {
			// start again from the end
			focusit = childs.end();
		}

		// if we are at the beginning
		if(focusit == childs.begin()) {
			focusit = childs.end();
			// if we have parent tell that we have nothing to focus
			if(parent && focusparent) {
				focus = ID_NONE;
				return ID_NONE;
			}
		}

		// get the next child
		focusit--;
		focus = (*focusit)->GetID();
		// if it doesn't want the focus let's try again
		if((*focusit)->skip_focus || !(*focusit)->IsEnabled())
			return FocusPrev(focusparent);
	
		// let's tell that it has focus
		(*focusit)->FocusAction();

		// if it is a container let's ask it to focus it's last item
		Container *c = dynamic_cast<Container*>(*focusit);
		if(c && !c->FocusAsObject()) {
			c->FocusNone();
			ID i = c->FocusPrev();
			if(i == ID_NONE)
				return FocusPrev(focusparent);
			else
				return i;
		}

		return focus;
	}
	
	ActionListener *Object::GetListener() const
	{
		if(listener)
			return listener;
		if(parent)
			return parent->GetListener();
		return NULL;
	}

	void TextLabel::RenderCoor(sf::RenderTarget &target, const Coor c) const
	{
		if(hide)
			return;
		if(drawbg) {
			DrawBackground(target, c);
		}
		sf::String str(label, font);
		str.SetSize(size);
		str.SetColor(font_color);
		str.SetPosition(GetRelX() + c.x - str.GetRect().Left, GetRelY() + c.y - str.GetRect().Top - 7);
		target.Draw(str);
	}

	void TextLabel::ResizeHandler()
	{
		unsigned int th = TextHeight(label, font);
		unsigned int tw = TextWidth(label, font) + 5;
		SetRect(Rect(GetRelX(), GetRelY(), tw, th));
		if(parent)
			parent->ResizeAction();
	}

	Button::Button(Rect r, std::string _label) : 
		TextLabel(r, _label), hilight_under_mouse(false), start_size(r.w, r.h)
	{ 
		skip_focus = false; 
		SetBorderWidth(2);
	}

	Button::Button(std::string _label) : 
		TextLabel(_label), hilight_under_mouse(false)
	{ 
		skip_focus = false; 
		start_size = Coor(rect.w, rect.h);
		SetBorderWidth(2);
	}
	
	void Button::KeyDownAction(Key k) 
	{ 
		ActionListener *l = GetListener();
		if(k.Code == sf::Key::Return && l)
			l->ButtonClick(id);
	}

	void Button::MouseDownAction(int x, int y, MouseButton m) 
	{ 
		ActionListener *l = GetListener();
		if(l && m == BUTTON1)
			l->ButtonClick(id);
	}

	void Button::ResizeHandler()
	{
		SetW(std::max(start_size.x, GetW()));
		SetH(std::max(start_size.y, GetH()));
	}

	void Button::Render(sf::RenderTarget &target) const
	{
		if(drawbg) DrawBackground(target);
		sf::Color col = bcolor;
		if(hilight_under_mouse && IsUnderMouse()) 
			col += sf::Color(50, 50, 50);
		else if(focused)
			col += sf::Color(30, 30, 30);
		DrawBorder(target, col);

		bool d = drawbg;
		drawbg = false;
		TextLabel::RenderCoor(target,
				Coor(
					(rect.w - start_size.x) / 2, 
					(rect.h - start_size.y) / 2)  + Coor(2, 2));
		drawbg = d;
	}
	
	Image::Image(Rect r) : Object(r)
	{
		skip_focus = true;
	}

	Image::Image(const sf::Image &_img, Rect r) : 
		Object(Rect(r.x+1, r.y+1, 
			r.w ? r.w : _img.GetWidth()-1, r.h ? r.h : _img.GetHeight()-1)), 
		img(_img)
	{
		skip_focus = true;
	}
			
	void Image::Render(sf::RenderTarget &target) const
	{
		RenderCoor(target);
	}

	void Image::RenderCoor(sf::RenderTarget &target, Coor coor) const
	{
		if(hide)
			return;
		sf::Sprite img(this->img);
		img.SetScale(float(GetW()) / float(img.GetImage()->GetWidth()), 
			float(GetH()) / float(img.GetImage()->GetHeight()));
		img.SetPosition(GetRelX() + coor.x, GetRelY() + coor.y);
		target.Draw(img);
	}
		
	void Image::ResizeHandler()
	{
//		SetRect(Rect(GetRelX(), GetRelY(), img.GetImage()->GetWidth(), img.GetImage()->GetHeight()));
	}

	ImgButton::ImgButton(Rect r) 
		: Image(r), tooltip_on(false), tooltip_shown(false) 
	{ 
		skip_focus = false; 
	}

	ImgButton::ImgButton(const sf::Image &img, Rect r) 
		: Image(img, r), tooltip_on(false), tooltip_shown(false)
	{ 
		skip_focus = false; Expand(2, 2);
	}

	ImgButton::~ImgButton()
	{ 
		if(tooltip_shown) {
			UnsetTip();
		}
	}

	void ImgButton::MouseDownAction(int x, int y, MouseButton m) 
	{ 
		ActionListener *l = GetListener();
		if(l && m == BUTTON1)
			l->ButtonClick(id);
	}

	void ImgButton::KeyDownAction(Key k) 
	{ 
		ActionListener *l = GetListener();
		if(k.Code == sf::Key::Return && l)
			l->ButtonClick(id);
	}

	void ImgButton::HandleLogic()
	{
		if(tooltip_on) {
			if(IsUnderMouse() && !focused) {
				if(under_mouse.GetElapsedTime() > 1.0f) {
					SetTip(tooltip);
					tooltip_shown = true;
				}
			} else if(tooltip_shown) {
				tooltip_shown = false;
				under_mouse.Reset();
				UnsetTip();
			} else {
				under_mouse.Reset();
			}
		}
	}

	void ImgButton::SetToolTip(std::vector<std::string> str, sf::Color color, sf::Color bgcolor) {
		tooltip_on = true;
		//				tooltip_shown = false;
		tooltip.str = str;
		tooltip.color = color;
		tooltip.bgcolor = bgcolor;
	}

	void ImgButton::Render(sf::RenderTarget &target) const
	{
		DrawBorder(target, focused ? sf::Color(0xA0, 0, 0xA0) + bcolor : bcolor, bwidth);
		Image::RenderCoor(target, Coor(1, 1));
	}
	
	HorizontalList::HorizontalList(Rect r, int _pad) : Container(r), pad(_pad), width(0)
	{
	}
	
	Container *HorizontalList::Add(Object *o)
	{
		if(rect.h < o->GetH())
			rect.h = o->GetH();
		Container::Add(o, width, 0);
		width += o->GetW() + pad;
		return this;
	}

	Container *HorizontalList::AddCenter(Object *o)
	{
		if(rect.h < o->GetH())
			rect.h = o->GetH();
		Container::Add(o, width, (GetH() - o->GetH()) / 2);
		width += o->GetW() + pad;
		return this;
	}
	
	void HorizontalList::ResizeHandler()
	{
		width = 0;
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			(*it)->SetRelX(width);
			if(rect.h < (*it)->GetH())
				rect.h = (*it)->GetH();
			width += (*it)->GetW() + pad;
		}
		if(parent)
			parent->ResizeAction();
	}

	VerticalList::VerticalList(Rect r, int _pad) : Container(r), pad(_pad), height(0)
	{
	}

	Container *VerticalList::Add(Object *o)
	{
		Container::Add(o, 0, height);
		height += o->GetH() + pad;
		return this;
	}

	Container *VerticalList::AddCenter(Object *o)
	{
		Container::Add(o, (GetW() - o->GetW()) / 2, height);
		height += o->GetH() + pad;
		return this;
	}

	void VerticalList::ResizeHandler()
	{
		height = 0;
		for(object_iter it = childs.begin(); it != childs.end(); it++) {
			(*it)->SetRelY(height);
			height += (*it)->GetH() + pad;
		}
		if(parent)
			parent->ResizeAction();
	}

	TextList::TextList(Rect r, std::string _text, int _pad)
			: Object(r), text(WordWrap(_text, font_default, r.w)), pad(_pad),
			scroll(0), max_texts(r.h / (TextHeight(_text, font) + pad)), size(30.f)
	{ 
		bwidth = 2;
		ResizeAction();
	}

	void TextList::SetText(std::string _text) 
	{ 
		scroll = 0;
		text = WordWrap(_text, font, rect.w); 
		ResizeAction();
	}

	void TextList::SetText(std::vector<std::string> _text)
	{
		scroll = 0;
		text = _text;
		ResizeAction();
	}

	void TextList::MouseDownAction(int x, int y, MouseButton m)
	{
		if(m == BUTTON4)
			ScrollUp();
		else if(m == BUTTON5)
			ScrollDown();
	}

	void TextList::KeyDownAction(Key k)
	{
		if(k.Code == sf::Key::Up)
			ScrollUp();
		else if(k.Code == sf::Key::Down)
			ScrollDown();
	}

	void TextList::ScrollDown()
	{
		scroll++;
		if((signed)(text.size() - scroll) < max_texts)
			scroll = 0;
	}

	void TextList::ScrollUp()
	{
		if(scroll == 0)
			scroll = std::max(static_cast<int>(text.size() - max_texts), 0);
		else
			scroll--;
	}

	void TextList::Render(sf::RenderTarget &target) const
	{
		if(hide)
			return;
		if(drawbg)
			DrawBackground(target, Coor());
		DrawBorder(target, focused ? sf::Color(0xAA, 0xAA, 0xAA) : color);
		for(unsigned int a = scroll; a < text.size() && (signed)(a - scroll) < max_texts; a++) {
			sf::String str(text[a]);
			str.SetColor(font_color);
			str.SetSize(size);
			str.SetPosition(GetRelX() - str.GetRect().Left, 
					(size + pad) * (a - scroll) + 1 + GetRelY());
			target.Draw(str);
		}
	}

	void TextList::ResizeHandler()
	{
		unsigned int th = 0, tw = 0;
		for(unsigned int a = scroll; a < text.size() && (signed)(a - scroll) < max_texts; a++) {
			th += size;
			tw = std::max(tw, TextWidth(text[a], font, size) + 5);
		}

		th += 5;

		SetRect(Rect(GetRelX(), GetRelY(), tw, th));
		if(parent)
			parent->ResizeAction();
	}

	std::vector<std::string> TextList::WordWrap(std::string text, const sf::Font &font, const unsigned int maxw)
	{
		std::vector<std::string> ret;
		unsigned int start = 0;
		unsigned int last = 0;

		for(unsigned int a = 0; a < text.size(); a++) {
			if(text[a] == '\n') {
				ret.push_back(text.substr(start, a - start));
				start = last = a + 1;
			} else if(text[a] == ' ') {
				if(TextWidth(text.substr(start, a - start), font) > maxw) {
					ret.push_back(text.substr(start, last - start));
					start = last + 1;
					last = a;
				} else {
					last = a;
				}
			}
		}

		if(start != text.size()) {
			for(unsigned int a = start; a < text.size(); a++) {
				if(text[a] != ' ') {
					ret.push_back(text.substr(a));
					break;
				}
			}
		}

		return ret;
	}

	ScrollSelection::ScrollSelection(Rect r, sf::Image &_select, sf::Color _color, int _pad) : 
			Container(r), pad(_pad), height(0), scroll(0), selected(0), select(_select)
	{
		color = _color;
		bwidth = 4;
	}

	Container *ScrollSelection::Add(Object *o)
	{
		Container::Add(o, 0, 0);
		height += o->GetH() + pad;
		return this;
	}

	void ScrollSelection::Render(sf::RenderTarget &target) const
	{
		unsigned int cur_height = 0;
		if(hide)
			return;
		if(drawbg)
			target.Draw(sf::Shape::Rectangle(rect.x, rect.y, rect.x+rect.w, rect.y+rect.h, color));
		DrawBorder(target, color);
		for(unsigned int a = scroll; a < childs.size() && cur_height + (unsigned)pad + (unsigned)childs[a]->GetH() < (unsigned)GetH() - 2 * bwidth; a++) {
			Coor r = static_cast<Coor>(rect) + Coor(2 * bwidth, cur_height + 2 * bwidth);
			sf::Sprite select(this->select);
			select.SetPosition(r.sfVector());
			if((unsigned)selected == a)
				target.Draw(select);
			childs[a]->SetPosition((r + Coor(select.GetImage()->GetWidth() + 5, 0)).sfVector());
			target.Draw(*childs[a]);
			cur_height += pad + childs[a]->GetH();
		}
	}

	int ScrollSelection::GetHeight(int a, int b)
	{
		int sum = (b - a) * pad + 4 * bwidth;
		for(int c = a; c < b; c++) {
			sum += childs[c]->GetH();
		}
		return sum;
	}

	int ScrollSelection::GetIndex(int x, int y)
	{
		y -= 2 * bwidth;
		x -= 2 * bwidth + select.GetImage()->GetWidth() + 5;
		for(int a = scroll; a < scroll + MaxTexts(scroll, 1); a++) {
			if(0 <= y && y < childs[a]->GetH()) {
				if(x > childs[a]->GetW() )
					return -1;
				return a;
			}
			y -= childs[a]->GetH() + pad;
		}
		return -1;
	}

	int ScrollSelection::MaxTexts(int a, int d)
	{
		int sum = 4 * bwidth;
		unsigned int c;
		if(d != 1 && d != -1)
			std::cerr << "MaxTexts called with bad d (" << d << ")\n";
		for(c = a; c >= 0 && c < childs.size() && sum + childs[c]->GetH() <= GetH(); c += d) {
			sum += childs[c]->GetH() + pad;
		}
		return abs(c-a);
	}

	void ScrollSelection::Select()
	{
		ActionListener *l = GetListener();
		if(l) {
			l->ScrollSelect(id, childs[selected]->GetID());
		}
	}

	void ScrollSelection::MouseDownAction(int x, int y, MouseButton m)
	{
		x -= GetRelX();
		y -= GetRelY();
		if(m == BUTTON1) {
			int a = GetIndex(x, y);
			if(selected == a && click_timer.GetElapsedTime() < 0.300f) {
				Select();
			}
			if(a != -1) {
				ActionListener *l = GetListener();
				if(l)
					l->ScrollChSelect(id, a);
				selected = a;
			}
		}
		if(m == BUTTON4) {
			ScrollUp();
		}
		if(m == BUTTON5)
			ScrollDown();
		click_timer.Reset();
	}

	void ScrollSelection::ScrollDown()
	{
		selected++;
		if(selected >= (MaxTexts(scroll, 1) + scroll)) {
			if((unsigned)selected >= childs.size()) {
				scroll = 0;
			} else {
				scroll++;
			}
		}
		if((unsigned)selected >= childs.size())
			selected = 0;
		ActionListener *l = GetListener();
		if(l)
			l->ScrollChSelect(id, selected);
	}

	void ScrollSelection::ScrollUp()
	{
		selected--;
		if(selected < scroll) {
			if(selected < 0) {
				scroll = childs.size() - MaxTexts(childs.size() - 1, -1);
			} else {
				scroll--;
			}
		}
		if(selected < 0)
			selected = childs.size() - 1;
		ActionListener *l = GetListener();
		if(l)
			l->ScrollChSelect(id, selected);
	}

	void ScrollSelection::KeyDownAction(Key k)
	{
		if(k.Code == sf::Key::Down) {
			ScrollDown();
		} else if(k.Code == sf::Key::Up) {
			ScrollUp();
		} else if(k.Code == sf::Key::Return) {
			Select();
		}
	}

	void TextSelection::SetText(std::vector<std::string> &text)
	{
		selected = 0;
		Clear();
		for(unsigned int a = 0; a < text.size(); a++) {
			TextLabel *t = new TextLabel();
			t->SetFont(font);
			t->SetFontColor(font_color);
			t->SetLabel(text[a]);
			Add(t);
		}
	}

	void TextSelection::Select()
	{
		ActionListener *l = GetListener();
		if(l) {
			l->ScrollSelect(id, selected);
		}
	}

	Window::Window(Rect r, sf::Color _color, int _bar, sf::Color _barcolor) : Container(r, _color), bar(_bar), barcolor(_barcolor), grabbed(false)
	{
		DrawBG(true);
	}

	void Window::Render(sf::RenderTarget &target) const
	{
		if(hide)
			return;
		rect.h -= bar;
		Container::RenderCoor(target, Coor(0, bar));
		rect.h += bar;

		sf::Shape rc = sf::Shape::Rectangle(rect.x, rect.y, rect.x+rect.w, rect.y+bar, barcolor);
		target.Draw(rc);
	}

	void Window::MouseDownAction(int x, int y, MouseButton m)
	{ 
		if(y - GetRelY() > bar)
			Container::MouseDownAction(x, y - bar, m);
		else
			grabbed = true;
	}

	void Window::MouseUpAction(int x, int y, MouseButton m)
	{
		if(y - GetRelY() > bar)
			Container::MouseUpAction(x, y - bar, m);
		else
			grabbed = false;
	}

	void Window::MouseMotionAction(int x, int y, int dx, int dy, const sf::Input &input) 
	{ 
		if(grabbed) {
			if(input.IsMouseButtonDown(sf::Mouse::Left)) {	
				SetRelY(GetRelY() + dy);
				SetRelX(GetRelX() + dx);
			}
		} else {
			if(y - GetRelY() > bar) {
				Container::MouseMotionAction(x, y - bar, dx, dy, input);
			}
		}
	}

	TextField::TextField(Rect r, sf::Color _color) 
			: Object(r), cursor(0), scroll(0), drawwidth(0)
	{
		rect.h = TextHeight("", font);
		color = _color;
	}

	TextField::TextField(int w, sf::Color _color) 
			: Object(Rect(0, 0, w, 0)), cursor(0), scroll(0), drawwidth(0)
	{
		rect.h = TextHeight("", font) + 5;
		color = _color;
	}

	void TextField::Render(sf::RenderTarget &target) const
	{
		if(hide)
			return;
		target.Draw(sf::Shape::Rectangle(rect.x, rect.y, rect.x+rect.w, rect.y+rect.h, color));
		if(text.size()) {
			sf::String str(text.substr(scroll, drawwidth), font);
			str.SetColor(font_color);
			str.SetPosition(GetRelX(), GetRelY());
			target.Draw(str);
		}
		const unsigned int x1 = rect.x + TextWidth(text.substr(scroll, cursor - scroll), font);
		const unsigned int y1 = rect.y + 2;
		const unsigned int x2 = x1 + 2;
		const unsigned int y2 = y1 + TextHeight("", font);

		sf::Shape rec = sf::Shape::Rectangle(x1, y1, x2, y2, sf::Color::Black); 
		target.Draw(rec);
	}

	void TextField::TextAction(sf::Uint32 k)
	{
		if(32 <= k && k <= 126) {
			text.insert(cursor++, 1, static_cast<char>(k));
			if(TextWidth(text.substr(scroll, cursor - scroll), font) > (unsigned int)GetW())
				CalcScroll();
		}
		CalcDrawWidth();
	}

	void TextField::KeyDownAction(Key k)
	{
		if(k.Code == sf::Key::Left) {
			cursor--;
			if(cursor < 0) {
				cursor = text.size();
				CalcScroll();
			} else if(cursor <= scroll && scroll > 0) {
				scroll--;
			}
		} else if(k.Code == sf::Key::Right) {
			cursor++;
			if((unsigned)cursor > text.size()) {
				cursor = scroll = 0;
			} else {
				if(TextWidth(text.substr(scroll, cursor - scroll), font) > (unsigned int)GetW())
					CalcScroll();
			}
		} else if(k.Code == sf::Key::Back) {
			if(text.size() && cursor > 0) {
				cursor--;
				if(cursor == scroll)
					CalcScroll();
				text.erase(cursor, 1);
			}
		} else if(k.Code == sf::Key::Return) {
			ActionListener *l = GetListener();
			if(l) {
				l->SetText(id, text);
			}
		} else if(k.Code == sf::Key::Home) {
			cursor = 0;
			scroll = 0;
		} else if(k.Code == sf::Key::End) {
			cursor = text.size();
			CalcScroll();
		} else if(k.Code == sf::Key::Delete) {
			if(text.size() && (unsigned)cursor < text.size()) {
				if(cursor == scroll)
					CalcScroll();
				text.erase(cursor, 1);
			}
		}
		CalcDrawWidth();
	}
	
	void TextField::CalcScroll()
	{
		int w = 0;
		for(int a = cursor - 1; a >= 0; a--) {
			w = TextWidth(text.substr(a, cursor - a), font);
			if(GetW() < w) {
				scroll = a + 1;
				return;
			}
		}
		scroll = 0;
	}
			
	void TextField::CalcDrawWidth()
	{
		int w = 0;
		for(unsigned int a = scroll; a < text.size(); a++) {
			w = TextWidth(text.substr(scroll, a - scroll + 1), font);
			if(GetW() < w) {
				drawwidth = a - scroll;
				return;
			}
		}
		drawwidth = text.size();

	}
	std::ostream & operator<<(std::ostream & str, Gui::Rect r) {
		return str << "(" << r.x << ", " << r.y << ", " << r.w << ", " << r.h << ")";
	}
	
	void Bar::Render(sf::RenderTarget &target) const 
	{

		if(!hide) {
			drawFilledRectangle(target, rect, p, color);
			DrawBorder(target);
		}
	}
}

