/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include <iostream>
#include <cstdlib>
#include "map.h"
#include "object.h"

const char *map::img =
	"xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
	"...................................."
	"...................................."
	"................................G..."
	"...................................."
	"...................................."
	".........xxxxxxxxxxxxxxxxxxxxxxxxxxx"
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"...................................."
	"xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
	; 


const int map::img_h = 23;
const int map::img_w = 36;

tile::tile(tile::type _t, int _hp) : 
	t(_t), hp(_hp), anim(NULL)
{
}

tile::tile(tile::type _t, animation_loader *l, int _hp) : 
	t(_t), hp(_hp), anim(NULL)
{
	if(l) {
		switch(t) {
			case grass:
				anim = NULL;
//				anim = new animation(l->get_animation("grass"));
				break;
			case wall:
				anim = new animation(l->get_animation("wall"));
				break;
		}
		if(anim)
			anim->set_mode("stand");
	}
}

tile::tile(const tile &o) :
	t(o.t), hp(o.hp), anim(NULL)
{
	if(o.anim) {
		anim = new animation(*o.anim);
	}
}

tile::~tile()
{
	delete anim;
}

void tile::hit(animation_loader *l, int dmg)
{ 
	if(t == tile::wall) {
		hp -= dmg;
		if(hp >= 80) {
			if(anim)
				anim->set_mode("stand");
		} else if(hp >= 60) {
			if(anim)
				anim->set_mode("broken1");
		} else if(hp >= 40) {
			if(anim)
				anim->set_mode("broken2");
		} else if(hp > 0) {
			if(anim)
				anim->set_mode("broken3");
		} else if(hp <= 0) {
			if(anim) {
				delete anim;
				anim = NULL;
			}
			t = grass;
		}
	}
}

map::map() : 
	w(img_w), h(img_h), anim_loader(0)
{
	load_map();
#ifndef NO_GRAPHICS
	grass = NULL;
#endif
}

map::map(animation_loader *l) : 
	w(img_w), h(img_h), anim_loader(l)
{
	load_map();
#ifndef NO_GRAPHICS
	grass = new animation(anim_loader->get_animation("grass"));
	grass->set_mode("stand");
#endif
}

map::~map()
{
#ifndef NO_GRAPHICS
	delete grass;
#endif
}

void map::load_map()
{
	tiles.clear();
	for(int y = 0; y < img_h; y++) {
		std::vector<tile> tmp;
		for(int x = 0; x < img_w; x++) {
			switch(img[y * img_w + x]) {
				case 'x':
					tmp.push_back(tile(tile::wall, anim_loader));
					break;
				case 'G':
					goals.push_back(goal(sf::Vector2i(x, y), anim_loader));
					tmp.push_back(tile(tile::grass, anim_loader));
					break;
				default:
					tmp.push_back(tile(tile::grass, anim_loader));
					break;
			}
		}
		tiles.push_back(tmp);
	}
}

bool map::los(sf::Vector2f start, sf::Vector2f end)
{
	sf::Vector2f dir = end - start;
	sf::Vector2f pos = start;

	int sx = sgn(dir.x);
	int sy = sgn(dir.y);

	while(coor_to_tile(pos) != coor_to_tile(end)) {
		tile &t = get_tile(coor_to_tile(pos));

		if(t.t == tile::wall || out_of_bounds(coor_to_tile(pos))) {
			return false;
		}

		double bx = nearest_next_border(pos.x, sx, TILE_W);
		double by = nearest_next_border(pos.y, sy, TILE_H);

		double dx = bx - pos.x;
		double dy = by - pos.y;

		if(sy && sx) {
			if(fabs(dx / dy) > fabs(dir.x / dir.y)) {
				pos.y = by;
				pos.x += dir.x / dir.y * dy;
			} else {
				pos.x = bx;
				pos.y += dir.y / dir.x * dx;
			}
		} else {
			if(sy == 0)
				pos.x = bx;
			else if(sx == 0)
				pos.y = by;
		}
	}
	return true;
}

double map::nearest_next_border(double x, int sign, double w)
{
	if(sign == 0)
		return x;
	if(fmod(x, w) < 0.000001)
		return x + sign * w;
	return x - fmod(x, w) + ((sign > 0) ? w : -1.0f);
}


bool map::out_of_bounds(sf::Vector2i p)
{
	return (p.x < 0 || p.x >= w || p.y < 0 || p.y >= h);
}

bool map::collision(sf::Vector2f pos, float w, float h)
{
	float ceil_h = TILE_H * ceil(h / TILE_H);
	float ceil_w = TILE_W * ceil(w / TILE_W);

	for(float y = 0; y <= ceil_h; y += TILE_H) {
		if(abs(y - ceil_h) < 0.00001)
			y = h;
		for(float x = 0; x <= ceil_w; x += TILE_W) {
			if(fabs(x - ceil_w) < 0.00001)
				x = w;

			sf::Vector2f t_pos = pos + sf::Vector2f(x, y);
			sf::Vector2i t = coor_to_tile(t_pos);
			if(tiles[t.y][t.x].t == tile::wall) {
				return true;
			}
		}
	}
	return false;
}

bool map::collision(const object &o)
{
	return collision(o.get_pos(), o.get_w(), o.get_h());
}

sf::Vector2i map::coor_to_tile(sf::Vector2f pos)
{
	if(pos.x < 0) 
		pos.x = 0;
	if(pos.y < 0)
		pos.y = 0;
	if(pos.y >= tiles.size() * TILE_H)
		pos.y = tiles.size() * TILE_H - 1;
	if(pos.x >= tiles[0].size() * TILE_W)
		pos.x = tiles[0].size() * TILE_W - 1;

	return sf::Vector2i(pos.x / TILE_W, pos.y / TILE_H);
}

sf::Vector2i map::nearest_tile(sf::Vector2f pos, double r, tile::type tile_type)
{
	sf::Vector2f l_corner_pos = pos + sf::Vector2f(-r, -r);
	sf::Vector2f r_corner_pos = pos + sf::Vector2f(r, r);

	sf::Vector2i l_corner = coor_to_tile(l_corner_pos);
	sf::Vector2i r_corner = coor_to_tile(r_corner_pos);

	double d = 9999999;
	sf::Vector2i ret = sf::Vector2i(-1, -1);
	for(int x = l_corner.x; x <= r_corner.x; x++) {
		for(int y = l_corner.y; y <= r_corner.y; y++) {
			if(dist_to_tile(pos, sf::Vector2i(x, y)) > r)
				continue;
				
			if(tiles[y][x].t == tile_type) {
				ret = sf::Vector2i(x, y);
				d = dist_to_tile(pos, ret);
			}
		}
	}
	return ret;
}

double map::dist_to_tile(sf::Vector2f pos, sf::Vector2i tile)
{
	sf::Vector2f tile_pos = sf::Vector2f(tile.x*TILE_W, tile.y*TILE_H);
	double d = 999999999;

	sf::Vector2i pos_tile = coor_to_tile(pos);

	if(pos_tile.x == tile.x && pos_tile.y == tile.y) {
		return 0;
	}

	if(pos_tile.x == tile.x) {
		return abs(pos.y - tile_pos.y);
	}

	if(pos_tile.y == tile.y) {
		return abs(pos.x - tile_pos.x);
	}
	
	for(int x = 0; x <= 1; x++) {
		for(int y = 0; y <= 1; y++) {
			d = std::min(d, dist(pos, tile_pos + sf::Vector2f(x * TILE_W, y * TILE_H)));
		}
	}
	return d;
}

void map::update()
{
	for(unsigned int y = 0; y < tiles.size(); y++) {
		for(unsigned int x = 0; x < tiles[y].size(); x++) {
			if(tiles[y][x].anim) {
				tiles[y][x].anim->update();
			}
		}
	}

	for(goal_iter i = goals.begin(); i != goals.end(); i++) {
		if(i->get_anim()) {
			i->get_anim()->update();
		}
	}
}

bool map::near_goal(object *o, double range) 
{
	for(std::list<goal>::const_iterator i = goals.begin(); i != goals.end(); i++) {
		double d = dist(o->get_center(), i->get_center());
		if(d < range) {
			return true;
		}
	}
	return false;
}


#ifndef NO_GRAPHICS
void map::draw(sf::RenderTarget &target)
{
	if(!anim_loader) {
		throw std::logic_error("Trying to draw map though it hasn't been given animation data");
	}

	sf::Sprite spr;
	spr.SetImage(grass->get_frame());
	spr.Resize(sf::Vector2f(1130, 720));
	target.Draw(spr);

	for(unsigned int y = 0; y < tiles.size(); y++) {
		for(unsigned int x = 0; x < tiles[y].size(); x++) {
			if(tiles[y][x].t == tile::grass)
				continue;
			sf::Sprite spr;
			spr.SetPosition(sf::Vector2f(TILE_W * x, TILE_H * y));
			spr.SetImage(tiles[y][x].anim->get_frame());
			spr.Resize(TILE_W + 2, TILE_H + 2);

			target.Draw(spr);
		}
	}


	for(goal_iter i = goals.begin(); i != goals.end(); i++) {
		i->draw(target);
	}
}
#endif

