/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "object.h"
#include "world.h"
#include <sstream>

ID object::id_counter = 0;

object::object() :
	dead(false), hp(0), max_hp(0), 
	energy(0), max_energy(0), anim(NULL), preferred_target(0), immortal(false), invisible(false), w(0), h(0), id(++id_counter)
{ }

object::object(animation_data *anim_data) :
	dead(false), hp(0), max_hp(0), 
	energy(0), max_energy(0), anim(NULL), preferred_target(0), immortal(false), invisible(false), id(++id_counter)
{
#ifndef NO_GRAPHICS
	if(anim_data) {
		anim = new animation(*anim_data);
		w = anim->get_w();
		h = anim->get_h();
	} 
#endif
}

object::object(const object &o) : 
	dead(o.dead), hp(o.hp), max_hp(o.max_hp), energy(o.energy),
	max_energy(o.max_energy), anim(0), immortal(o.immortal), immortal_time(o.immortal_time),
	immortal_timer(o.immortal_timer), invisible(o.invisible), invisible_time(o.invisible_time),
	invisible_timer(o.invisible_timer), pos(o.pos), 
	w(o.w), h(o.h), id(o.id)
{
	if(o.anim) {
		anim = new animation(*o.anim);
	}
}

object::~object()
{
	if(anim)
		delete anim;
}

void object::add_hp(double dhp)
{ 
	if(dhp < 0 && immortal)
		return;
	hp += dhp; 
	if(hp <= 0) { 
		dead = true;
		hp = 0; 
	}	
	if(hp > max_hp) 
		hp = max_hp; 

}

void object::add_energy(double d)
{
	energy += d; 
	if(energy < 0) 
		energy = 0; 
	if(energy > max_energy) 
		energy = max_energy; 
}

#ifndef NO_GRAPHICS
void object::draw(sf::RenderTarget &target)
{
	if(anim) {
		if(!invisible && !immortal) {
			sf::Sprite spr;
			spr.SetPosition(pos);
			spr.SetScale(float(w) / float(anim->get_w()), 
							float(h) / float(anim->get_h()));
			spr.SetImage(anim->get_frame());

			target.Draw(spr);
		} else {
			sf::Color color(0xFF, 0xFF, 0xFF, 0xFF);
			if(invisible)
				color.a -= 0x70;
			if(immortal) {
				color.r -= 0x70;
				color.g -= 0x70;
			}
			draw_color(target, color);
		}
	}
}

void object::draw_color(sf::RenderTarget &target, sf::Color color)
{
	if(anim) {
		sf::Sprite spr;
		spr.SetPosition(get_pos());
		spr.SetImage(anim->get_frame());
		spr.SetScale(float(get_w()) / float(anim->get_w()), 
						float(get_h()) / float(anim->get_h()));
		spr.SetColor(color);
		target.Draw(spr);
	}
}

void object::draw_border(sf::RenderTarget &target, sf::Color color)
{
	sf::Shape sh = sf::Shape::Rectangle(pos.x - 1, pos.y - 1, pos.x+w+2, pos.y+h+2, sf::Color(), 
			1.0f, color);
	sh.EnableFill(false);
	target.Draw(sh);
}

#endif

void object::set_mode(std::string anim_mode, bool loop)
{
	if(anim)
		anim->set_mode(anim_mode, loop);
}

void object::update(world &w)
{
	if(anim) {
		anim->unpause();
		anim->update();
	}

	if(is_dead())
		return;

	for(std::list<attack>::iterator iter = atks.begin(); iter != atks.end(); iter++) {
		iter->update();
		if(!iter->ready())
			continue;
			
		add_hp(-iter->get_dmg());
		iter = atks.erase(iter);

		if(get_hp() <= 0)
			dead = true;
	}
	
	if(immortal) {
		immortal_timer.unpause();
		immortal_timer.step();

		if(immortal_timer.get_time() > immortal_time) {
			immortal = false;
		}
	}

	if(invisible) {
		invisible_timer.unpause();
		invisible_timer.step();

		if(invisible_timer.get_time() > invisible_time) {
			invisible = false;
		}
	}
}	

void object::set_immortal(float time)
{
	immortal_timer.reset();
	immortal_time = time;
	immortal = true;
}

void object::set_invisible(float time)
{
	invisible_timer.reset();
	invisible_time = time;
	invisible = true;
}

void object::pause()
{
	if(anim)
		anim->pause();
	invisible_timer.pause();
	immortal_timer.pause();
}

bool object::collision(const object &obj) const
{
	if((get_x() <= obj.get_x() && obj.get_x() <= get_x() + get_w()) ||
			(obj.get_x() <= get_x() && get_x() <= obj.get_x() + obj.get_w()))
		if((get_y() <= obj.get_y() && obj.get_y() <= get_y() + get_h()) ||
				(obj.get_y() <= get_y() && get_y() <= obj.get_y() + obj.get_h()))
			return true;

	return false;
}

void object::damage(world &w, const attack &at)
{

	atks.push_back(at);
}

std::string object::hp_string()
{
	std::stringstream ss(std::stringstream::out);
	ss <<"Hp: "<< (int)hp << "/" <<max_hp<<std::endl;
	return ss.str();
}

std::string object::energy_string()
{
	std::stringstream ss(std::stringstream::out);
	ss <<"Energy: "<< (int)energy << "/" <<max_energy<<std::endl;
	return ss.str();

}

