/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#ifndef _OBJECT_H
#define _OBJECT_H

#include <list>
#include "animation.h"
#include "attack.h"
#include <SFML/System/Vector2.hpp>

typedef unsigned int ID;
class world;

inline double dist(sf::Vector2f a, sf::Vector2f b)
{
	a -= b;
	return sqrt(a.x*a.x + a.y*a.y);
}

inline sf::Vector2f normalize(sf::Vector2f a)
{
	double len = sqrt(a.x*a.x+a.y*a.y);
	return sf::Vector2f(a.x/len, a.y/len);
}

inline float dot(sf::Vector2f a, sf::Vector2f b)
{
	return a.x*b.x + a.y*b.y;
}

using sf::operator+;
using sf::operator-;
using sf::operator*;

class object {
	public:
		object();
		object(animation_data *anim_data);
		object(const object &o);
		virtual ~object();
		int get_x() const { return pos.x; }
		int get_y() const { return pos.y; }
		int get_w() const { return w; }
		int get_h() const { return h; }
		sf::Vector2f get_pos() const { return pos; }
		sf::Vector2f get_center() const
		{ 
			return sf::Vector2f(pos.x + w/2, pos.y + h/2); 
		}
		void set_pos(sf::Vector2f p) { pos = p; }
		void set_center(sf::Vector2f p) { 
			pos = sf::Vector2f(p.x - get_w() / 2, p.y - get_h() / 2);
		}

		virtual bool collision(const object &obj) const;

#ifndef NO_GRAPHICS
		virtual void draw(sf::RenderTarget &target);
		virtual void draw_color(sf::RenderTarget &target, sf::Color color);
		virtual void draw_border(sf::RenderTarget &target, sf::Color color);
#endif
		virtual void update(world &w);
		virtual void pause();

		void set_hp(double _hp) { hp = _hp; }
		double get_hp() { return hp; }
		void add_hp(double dhp);
		int get_max_hp() { return max_hp; }

		virtual void damage(world &, const attack &);

		double get_energy() { return energy; }
		int get_max_energy() { return max_energy ;}
		void set_max_energy(int _max_energy) { max_energy = _max_energy; }
		void add_energy(double d);
		void use_energy(double d) { add_energy(-d); }
		
		std::string hp_string();
		std::string energy_string();
		
		object* get_target() { return preferred_target; }
		virtual void set_target(object *t) { preferred_target = t; }

		animation *get_anim() { return anim; }
		ID get_id() { return id; }
		bool is_dead() { return dead; }
		virtual bool to_removed () { return is_dead(); }
		
		void set_mode(std::string anim_mode, bool loop = true);
	
		void set_immortal(float time);
		bool is_immortal() { return immortal; }

		void set_invisible(float time);
		bool is_invisible() { return invisible; }
	protected:
		void set_w(int _w) { w = _w; }
		void set_h(int _h) { h = _h; }
		bool dead;
		double hp;
	   	int max_hp;
		double energy;
		int max_energy;

		std::list<attack> atks;

		animation *anim;
	
		object *preferred_target;
		
		bool immortal;
		float immortal_time;
		pclock immortal_timer;
		bool invisible;
		float invisible_time;
		pclock invisible_timer;
	private:
		sf::Vector2f pos;
		int w, h;

		ID id;
		static ID id_counter;	

	friend class moving;
};

#endif
