/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "particle.h"
#include "settings.h"


using sf::operator+;
using sf::operator-;
using sf::operator*;


particle::particle(double _life, sf::Color _col, sf::Vector2f _pos, sf::Vector2f _vel, 
		sf::Vector2f _accel) :
	life(_life), max_life(_life), col(_col), pos(_pos), vel(_vel), accel(_accel),
	anim(0)
{
}

// GROUP

particle_group::particle_group(size_t _size, particle_emitter *e, draw_mode_t mode) 
	: size(_size), alive(0), emitter(e), draw_mode(mode)
{
	particles.resize(_size);
}

particle_group::~particle_group()
{
	delete emitter;
}

bool particle_group::dead()
{ 
	return emitter->dead() && alive == 0; 
}

void particle_group::update()
{
	emitter->update(*this);
	std::vector<particle> &p = particles;

	for(size_t i = 0; i < alive; i++) {
		p[i].life -= DT;
		p[i].col.a = p[i].life * 10.0f;
		if(p[i].dead()) {
			delete p[i].anim;
			--alive;
			p[i] = p[alive];
		}
	}
	move();
}

void particle_group::move()
{
	std::vector<particle> &p = particles;
	for(size_t i = 0; i < alive; i++) {
		p[i].pos += DT * p[i].vel;
		p[i].vel += DT * p[i].accel;
	}
}

void particle_group::add(particle p)
{
	if(alive < size) {
		particles[alive] = p;
		++alive;
	}
}

void particle_group::Render(sf::RenderTarget &target) const
{
	if(draw_mode == line) {
		for(size_t i = 0; i < alive; i++) {
			sf::Vector2f e;
			e += particles[i].vel * 0.4f;

			sf::Shape l = sf::Shape::Line(sf::Vector2f(), e, 2.0f, particles[i].col);
			l.SetPosition(particles[i].pos);
			target.Draw(l);
		}
	} else if(draw_mode == anim) {
		for(size_t i = 0; i < alive; i++) {
			sf::Sprite spr(particles[i].anim->get_frame(), particles[i].pos);
			spr.SetColor(particles[i].col);
			target.Draw(spr);
		}
	} else {
		sf::Vector2f a = sf::Vector2f(1.f, 1.f);
		for(size_t i = 0; i < alive; i++) {
			sf::Color c = particles[i].col;
			c.a = particles[i].life * 500;
			sf::Shape l = sf::Shape::Rectangle(-a, a, c);
			l.SetColor(c);
			l.SetPosition(particles[i].pos);
			target.Draw(l);
		}
	}
}

// EMITTER

particle_emitter::particle_emitter(double _rate) :
	particles(0), rate(_rate), maximum_age(0), age(0), delay_time(0)
{
}


void particle_emitter::update(particle_group &group)
{
	if(dead())
		return;
	age += DT;
	if(age < delay_time)
		return;
	particles += rate * DT;
	for(int a = 0; a < particles; a++) {
		group.add(create());
		particles -= 1.0f;
	}
}

float particle_emitter::uniform_dist(float cent, float r)
{
	return cent + 2 * r * (float(rand()) - float(RAND_MAX) / 2) / float(RAND_MAX);
}

particle particle_emitter::create()
{
	return particle(15.0f, sf::Color::Blue, sf::Vector2f());
}


// ENGINE

particle_engine::~particle_engine()
{
	for(g_iter i = groups.begin(); i != groups.end(); i++) {
		delete *i;
	}
}


void particle_engine::update()
{
	for(g_iter i = groups.begin(); i != groups.end(); i++) {
		(*i)->update();
		if((*i)->dead()) {
			delete (*i);
			i = groups.erase(i);
		}
	}
}

void particle_engine::draw(sf::RenderTarget &target)
{
	for(g_iter i = groups.begin(); i != groups.end(); i++) {
		target.Draw(**i);
	}
}

void particle_engine::add(particle_group *g) 
{
	groups.push_back(g);
}

