/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "record.h"
#include "settings.h"

const char *magic = "MUSHROOM\x1\x3\x3\x7";

const char *action_types[action::type_count] = 
{
	"none",
	"cast spell",
	"select",
	"move player",
	"cc_tree_selection",
	"select id"
};

action::action() : t(none)
{
}

action::action(sf::Packet &in)
{
	sf::Uint32 _t;
	in >> _t;
	

	if(_t < 0 || _t >= type_count)
		throw rec_exception("Invalid action type");
#ifdef DEBUG
	std::cout << "type: " << action_types[_t] << std::endl;
#endif 
	t = (type)_t;
	
	if(t == cast_spell ||
			t == select ||
			t == move_player ||
			t == cc_tree_selection) {
		in >> x;
		in >> y;
#ifdef DEBUG
		std::cout << "x: " << x << " y: " << y << std::endl;
#endif 
	}

	if(t == cast_spell ||
			t == select_id) {
		in >> id;
#ifdef DEBUG
		std::cout << "id: " << id << std::endl;
#endif 
	}
}

void action::write(sf::Packet &out)
{
	out << (sf::Int32)t;
	
	if(t == cast_spell ||
			t == select ||
			t == move_player ||
			t == cc_tree_selection) {
		out << x << y;
	}
	
	if(t == cast_spell ||
			t == select_id) {
		out << id;
	}
}

record::frame::frame() : time(0)
{
}

record::frame::frame(sf::Packet &in) : time(0)
{
	sf::Int32 t;
	in >> t;
	time = t;
	read_vector<action>(in, actions);
}


void record::frame::write(sf::Packet &out)
{
	out << time;
	write_vector<action>(out, actions);
}

record::record(const record &rec) : 
	creature_type(rec.creature_type), frames(rec.frames), it(frames.begin())
{
}

void record::append_action(action &a, sf::Uint32 time)
{
	if(frames.size()) {
		if(frames.back().time == time) {
			frames.back().actions.push_back(a);
			return;
		}
	}
	
	frame f;
	f.time = time;
	f.actions.push_back(a);

	frames.push_back(f);
}

void record::read(std::string file)
{
	std::ifstream in(file.c_str());
	if(!in) {
		throw rec_exception((std::string("Couldn't open file \"") + file + "\"").c_str());
	}
	std::string str((std::istreambuf_iterator<char>(in)), std::istreambuf_iterator<char>());
	sf::Packet pack;
	pack.Append(str.data(), str.size());

	sf::Int32 t;
	sf::Int32 version = -1;
	std::string m;
	pack >> m;
	pack >> version;
	if(m != magic || 0 > version) {
		throw rec_exception("Replay file is corrupted.");
	} else if(version < VERSION) {
		throw rec_version_exception((std::string("Replay file has wrong version (") + version_strings[version] + ") and is incompatible with this version.").c_str());
	} else if(version > VERSION) {
		throw rec_version_exception("Replay file was made with newer version of this game and is incompatible with this version");
	}
	pack >> t;
	if(t < 0 || t > (sf::Int32)creature::griffin) {
		throw rec_exception("Invalid character");
	}
	creature_type = (creature::creature_type) t;
	read_vector<frame>(pack, frames);
	sf::Uint32 chksum = adler32(pack, 4);
	sf::Uint32 chk;
	pack >> chk;
	if(chksum != chk) {
		throw rec_exception("Check sum doesn't match");
	}
	it = frames.begin();
}

void record::write(std::string file)
{
	std::ofstream out(file.c_str());
	sf::Packet pack;
	write_packet(pack);
	out.write(pack.GetData(), pack.GetDataSize());
}

std::string record::to_str()
{
	sf::Packet pack;
	write_packet(pack);
	return std::string(pack.GetData(), pack.GetDataSize());
}

void record::write_packet(sf::Packet &pack)
{
	sf::Int32 t = (sf::Int32)creature_type;
	pack << magic;
	pack << sf::Int32(VERSION);
	pack << t;
	write_vector<frame>(pack, frames);
	sf::Uint32 chksum = adler32(pack);
	pack << chksum;
}

record::frame &record::get_next_frame(sf::Uint32 time)
{
	static frame empty;
	if(it != frames.end()) {
		if(time > it->time) {
			throw rec_exception("Frames skipped! (Probably many frames with same time)");
		} else if(time == it->time) {
			std::vector<frame>::iterator c = it;
			it++;
			return *c;
		}
	}
	return empty;
}

sf::Uint32 record::adler32(sf::Packet &p, size_t no_read)
{
	const int mod = 65521;
	sf::Uint32 a = 1, b = 0;

	if(p.GetDataSize() >= no_read) {
		for(size_t i = 0; i < p.GetDataSize() - no_read; i++) {
			a = (a + p.GetData()[i]) % mod;
			b = (a + b) % mod;
		}
	}

	return (b << 16) | a;
}

