/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "wind.h"
#include "world.h"
#include <algorithm>

const double wind::speed = 100.0f;

wind::wind(double start_time, double _length, double _interval) :
	blowing(false), time_left(start_time), length(_length),
	interval(_interval), afterflow(999999.f), afterflow_stop(false),
	blow_count(0)
{
}

#ifndef NO_GRAPHICS
void wind::draw(sf::RenderTarget &target)
{
	engine.draw(target);
}
#endif

static bool cmp_mushroom(mushroom *m1, mushroom *m2)
{
	return m1->get_x() < m2->get_x();
}

void wind::update(world &w)
{
	time_left -= DT;
#ifndef NO_GRAPHICS
	engine.update();
#endif

	if(!blowing) {
		if(time_left <= 0) {
			w.handler.wind_blow();
			blowing = true;
			blow_count++;
			time_left = length;
			afterflow = 0;
#ifndef NO_GRAPHICS
			particle_emitter *emitter = new wind_emitter(80.0f);
			emitter->set_maximum_age(time_left);
			particle_group *group = new particle_group(600, emitter, particle_group::line);
			engine.add(group);
#endif
		} else {
			afterflow += DT;
		}
	} else {
		afterflow = 0;
		if(time_left <= 0) {
			blowing = false;
			time_left = interval;
			afterflow_stop = false;
		}
	}

	double left = afterflow * speed; 
	double right;
	if(blowing) {
		right = (length - time_left) * speed;
	} else {
		right = 999999.0f;
	}

	if(!afterflow_stop && left > SCREEN_W - SIDEBAR_W) {
		afterflow_stop = true;
		w.handler.wind_stop();
	}

	// TODO: optimize?
	std::vector<mushroom*> shrooms = w.get_mushrooms();
	std::sort(shrooms.begin(), shrooms.end(), cmp_mushroom);

	for(unsigned int i = 0; i < shrooms.size(); i++) { 
		double x = shrooms[i]->get_x();
		if(left < x && x < right) {
			shrooms[i]->wind_damage(double(shrooms.size() - i) / shrooms.size(), blow_count);
		}
	}	
	for(unsigned int i = 0; i < w.get_ants().size(); i++) {
		double x = w.get_ants()[i]->get_x();
		if(left < x && x < right) {
			w.move_object(w.get_ants()[i], sf::Vector2f(1.0f, 0.03f));
		}
	}

	creature &player = w.get_player();
	if(left < player.get_x() && player.get_x() < right) {
		switch(player.get_type()) {
			case creature::griffin:
				w.move_object(&player, sf::Vector2f(1.1f, 0.03f));
				break;
			default:
				break;
		}
	}
}

#ifndef NO_GRAPHICS
particle wind_emitter::create()
{
	sf::Color col = sf::Color(80, 80, 80);
	col.r += rand() % 176;
	col.g += rand() % 176;
	col.b += rand() % 176;
	sf::Vector2f vel;
	vel.x += uniform_dist(wind::speed, wind::speed/3);
	vel.y += uniform_dist(wind::speed / 3.0f, wind::speed / 6.0f);

	sf::Vector2f pos;
	pos.y = uniform_dist(334.0f, 424.0f);
	pos.x = -40.0f;

	double l = uniform_dist(13.0f, 10.0f);

	return particle(l, col, pos, vel);
}
#endif
