#include <iostream>
#include <list>
#include <sstream>
#include <vector>

#include "boost/shared_ptr.hpp"
#include "boost/filesystem.hpp"
#include "boost/foreach.hpp"
#include "boost/program_options.hpp"
#include "boost/gil/image.hpp"
#include "boost/gil/typedefs.hpp"
#include "boost/gil/image_view_factory.hpp"
#include "boost/gil/extension/io/png_io.hpp"
#include "boost/exception/diagnostic_information.hpp"

#include "ft_glyph.hpp"
#include "ft_face.hpp"

namespace fs = boost::filesystem;
namespace gil = boost::gil;
namespace po = boost::program_options;

typedef std::list<boost::shared_ptr<FtFace> > FaceList;
typedef std::vector<boost::shared_ptr<FtGlyph> > GlyphVector;

/** Usage help string. */
static const char usage[] =
"fontcompiler -o <output_file_base> <fontfiles>\n"
"This program will compile a truetype font into a precalculated font texture\n"
"file usable with the Valve algorithm.\n\n"
"The font files used as input will be iterated in the order specified in the\n"
"command line. The glyphs will be extracted from the first font file that\n"
"contains them, successive font files are used as a fallback.\n\n";

/** Listing of all glyphs to generate. */
static int glyph_list[] =
{
	' ', '~', // low ascii and latin
	0xc0, 0xff, // extended latin
	0x370, 0x3ff, // greek
	0x410, 0x44f, // cyrillic
	0x3040, 0x309e, // hiragana
	0x30a0, 0x30fe, // katakana
	0x4e00, 0x9fa5, // unified ideograms
	0
};

/** \brief Main function.
 *
 * @param argc Argument count.
 * @param argv Argument data.
 * @return Program exit code.
 */
int main(int argc, char **argv)
{
  try
  {
    FaceList fonts;
    GlyphVector glyphs;
    fs::path output_path;
    unsigned precalcsize = 2048,
             targetsize = 32;
    bool verbose = false;	

		{
			po::options_description desc("Options");
			desc.add_options()
				("font,f", po::value< std::vector<std::string> >(), "Font input file.")
				("help,h", "Print help text.")
				("outfile,o", po::value<std::string>(), "Output file basename.")
				("precalcsize,p", po::value<unsigned>(), "Size of glyph to use in calculation.")
				("targetsize,t", po::value<unsigned>(), "Target resolution to crunch glyphs to.")
				("verbose,v", "Turn on verbose reporting.");

			po::positional_options_description pdesc;
			pdesc.add("font", -1);

			po::variables_map vmap;
			po::store(po::command_line_parser(argc, argv).options(desc).positional(pdesc).run(), vmap);
			po::notify(vmap);

			if(vmap.count("font"))
			{
				std::vector<std::string> font_names =
					vmap["font"].as< std::vector<std::string> >();

        BOOST_FOREACH(std::string &vv, font_names)
        {
          FtFace *fnt = new FtFace(vv);
          fnt->setSize(static_cast<int>(precalcsize));
          fonts.push_back(boost::shared_ptr<FtFace>(fnt));
        }
      }
      if(vmap.count("help"))
      {
        std::cout << usage << desc;
        return 0;
      }
      if(vmap.count("outfile"))
      {
        if(output_path.file_string().length() > 0)
        {
          std::stringstream err;
          err << "output files already specified";
          BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
        }
        output_path = fs::path(vmap["outfile"].as<std::string>());
        if(output_path.file_string().length() <= 0)
        {
          std::stringstream err;
          err << "invalid output file specification" << output_path.file_string();
          BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
        }
        std::cout << "Using output file base: " << output_path.file_string() <<
          std::endl;
      }
      if(vmap.count("precalcsize"))
      {
        precalcsize = vmap["precalcsize"].as<unsigned>();
        if(precalcsize <= 0)
        {
          std::stringstream err;
          err << "invalid precalculation size" << precalcsize;
          BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
        }
      }
      if(vmap.count("targetsize"))
      {
        targetsize = vmap["targetsize"].as<unsigned>();
        if(targetsize <= 0)
        {
          std::stringstream err;
          err << "invalid crunch size" << targetsize;
          BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
        }
      }
      if(vmap.count("verbose"))
      {
        verbose = true;
      }
    }

		if(output_path.file_string().length() <= 0)
		{
			std::stringstream err;
			err << "output files not specified";
			BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
		}
		if(2048 % targetsize)
		{
			std::stringstream err;
			err << "2048 is not divisible by " << targetsize;
			BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
		}

		if(fonts.size() <= 0)
		{
			std::cerr << "Error: no valid font files" << std::endl;
			return 1;
		}

		// Open the XML file and write the header.
		std::string xmlfilename(output_path.file_string() + std::string(".xml"));
		FILE *xmlfile = fopen(xmlfilename.c_str(), "wt");
		if(!xmlfile)
		{
			std::stringstream err;
			err << "could not open " << xmlfilename << "for writing";
			BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
		}
		fputs("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
				"<font xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
				"xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\">\n",
				xmlfile);

		// Perform the actual generation of the glyphs.
		if(verbose)
		{
			std::cout << "Rendering:";
			std::cout.flush();
		}

		for(int *listiter = glyph_list; (*listiter);)
		{
			int ga = *listiter++,
					gb = *listiter++;

			for(int gidx = ga; (gidx <= gb); ++gidx)
			{
				FtGlyph *gly = NULL;

        for(FaceList::iterator ii = fonts.begin(), ee = fonts.end();
            (ii != ee);
            ++ii)
        {
          gly = (*ii)->renderGlyph(gidx, targetsize);
          if(gly)
          {
            if(verbose)
            {
              std::cout << " " << gidx;
              std::cout.flush();
            }
            glyphs.push_back(boost::shared_ptr<FtGlyph>(gly));
            break;
          }
        }

				if(!gly)
				{
					std::cerr << " (!" << gidx << ")";
					std::cout.flush();
				}
			}
		}

		unsigned gcnt_2048 = 2048 / targetsize,
						 gcnt_1024 = 1024 / targetsize,
						 gcnt_512 = 512 / targetsize,
						 glyph_count = static_cast<unsigned>(glyphs.size()),
						 rem_2048 = glyph_count % (gcnt_2048 * gcnt_2048),
						 rem_1024 = glyph_count % (gcnt_1024 * gcnt_1024),
						 rem_512 = glyph_count % (gcnt_512 * gcnt_512);

		unsigned pagesize = 2048;
		if((rem_1024 < rem_2048) && (gcnt_1024 < rem_512))
		{
			pagesize = 1024;
		}
		else if((rem_512 < rem_2048) && (rem_512 < rem_1024))
		{
			pagesize = 512;
		}
		if(verbose)
		{
			std::cout << " done\nUsing page size: " << pagesize << std::endl;
		}

		// Reserve target array.
		unsigned charsperside = pagesize / targetsize,
						 charsperpage = charsperside * charsperside,
						 bitmapw = pagesize,
						 bitmaph = pagesize * ((glyph_count / charsperpage) + 1);
		uint8_t *bitmap = static_cast<uint8_t*>(malloc(sizeof(uint8_t) * bitmapw * bitmaph));
		memset(bitmap, 0, sizeof(uint8_t) * bitmapw * bitmaph);

    for(unsigned ii = 0; (ii < glyphs.size()); ++ii)
    {
      boost::shared_ptr<FtGlyph> gly = glyphs[ii];

			unsigned page = ii / charsperpage,
							 pageidx = ii % charsperpage,
							 row = pageidx / charsperside,
							 col = pageidx % charsperside;
			float fpage = static_cast<float>(pagesize),
						fside = static_cast<float>(targetsize) / fpage;

			gly->setST(fside * static_cast<float>(col),
					fside * static_cast<float>(row + 1),
					fside * static_cast<float>(col + 1),
					fside * static_cast<float>(row),
					page);
			gly->write(xmlfile);
			gly->copy(bitmap, pagesize, ii);

      glyphs[ii] = boost::shared_ptr<FtGlyph>();
		}

		// Write the bitmap into a PNG file.
		for(unsigned ii = 0; (ii < bitmaph); ii += pagesize)
		{
			std::stringstream sstream;
			sstream << "_" << (ii / pagesize) << ".png";
			fs::path pngfilepath(output_path.file_string() + sstream.str());
			std::string pngfilename(pngfilepath.filename());

			fprintf(xmlfile, "\t<texture>%s</texture>\n", pngfilename.c_str());

			gil::gray8_image_t img(pagesize, pagesize);
			for(unsigned jj = 0; (jj < pagesize); ++jj)
			{
				for(unsigned kk = 0; (kk < pagesize); ++kk)
				{
					gil::view(img)(jj, kk) = bitmap[ii * pagesize + kk * pagesize + jj];
				}
			}
			gil::png_write_view(pngfilepath.file_string(), gil::view(img));

			if(verbose)
			{
				std::cout << "Wrote: " << pngfilepath.file_string() << std::endl;
			}
		}
		free(bitmap);

		// Close the XML index file.
		fputs("</font>", xmlfile);
		fclose(xmlfile);
		xmlfile = NULL;

    return 0;
  }

#ifndef SIMPLER_ERROR_MESSAGES
	catch(const boost::exception & e)
	{
		std::cerr << boost::diagnostic_information(e) << std::endl;
		return 1;
	}
#endif
	catch(const std::exception & e)
	{
		std::cerr << argv[0] << ": " << e.what() << std::endl;
		return 1;
	}
	catch(...)
	{
		std::cerr<<"Unkown exception caught!"<<std::endl;
		return -1;
	}
}

