#ifndef FT_GLYPH_HPP
#define FT_GLYPH_HPP

#ifdef WIN32
#define <pstdint.h>
#else
#include <inttypes.h>
#endif
#include <stdio.h>

/** \brief Represents one rendered glyph.
 */
class FtGlyph
{
	private:
		/** Unicode number. */
		unsigned _unicode;

		/** Freetype glyph data. */
		float _width;

		/** Freetype glyph data. */
		float _height;

		/** Freetype glyph data. */
		float _left;

		/** Freetype glyph data. */
		float _top;

		/** Freetype glyph data. */
		float _advance_x;

		/** Freetype glyph data. */
		float _advance_y;

		/** Bitmap width. */
		unsigned _bitmap_w;

		/** Bitmap height. */
		unsigned _bitmap_h;

		/** Bitmap data. */
		uint8_t *_bitmap;

		/** OpenGL data. */
		float _s1;

		/** OpenGL data. */
		float _t1;

		/** OpenGL data. */
		float _s2;

		/** OpenGL data. */
		float _t2;

		/** Font page. */
		unsigned _page;

	public:
		/** \brief Constructor.
		 *
		 * @param pcode Unicode number.
		 * @param pw Width.
		 * @param ph Height.
		 * @param pleft Left.
		 * @param ptop Top.
		 * @param pax Advance x.
		 * @param pay Advance y.
		 * @param bw Bitmap width.
		 * @param bh Bitmap height.
		 * @param bdata Bitmap data.
		 */
		FtGlyph(int pcode, float pw, float ph, float pleft, float ptop, float pax,
				float pay, unsigned bw, unsigned bh, uint8_t *bdata);

		/** \brief Destructor.
		 */
		~FtGlyph();

	public:
		/** \brief Copy this into a larger bitmap.
		 *
		 * @param tgt Target bitmap.
		 * @param tw Target bitmap width.
		 * @param idx Index to copy to.
		 */
		void copy(uint8_t *tgt, unsigned tw, unsigned idx);

		/** \brief Write the current glyph info into a file.
		 *
		 * @param fptr FILE pointer.
		 */
		void write(FILE *fptr);

	public:
		/** \brief Set the OpenGL texture coordinate data.
		 *
		 * @param s1 First S coordinate.
		 * @param t1 First T coordinate.
		 * @param s2 Second S coordinate.
		 * @param t2 Second T coordinate.
		 * @param page Font page.
		 */
		inline void setST(float s1, float t1, float s2, float t2, unsigned page)
		{
			_s1 = s1;
			_t1 = t1;
			_s2 = s2;
			_t2 = t2;
			_page = page;
		}
};

#endif // FT_GLYPH_HPP
