#ifndef GFX_BUFFER_HPP
#define GFX_BUFFER_HPP

#include "gfx/array.hpp"
#include "gfx/color.hpp"
#include "gfx/triangle.hpp"
#include "math/vec.hpp"

#include <vector>

namespace gfx
{
	class Lod;

	/** \brief Buffer base class.
	 *
	 * This class is just used for containing the OpenGL vertex array buffer.
	 *
	 * There are two main inheritors to Buffer base class, element buffer and
	 * different interleaved buffers.
	 */
	template<GLenum B> class Buffer
	{
		protected:
			GLuint _buffer;

		public:
			/** \brief Constructor. */
			Buffer() : _buffer(0)	{ }

			/** \brief Destructor. */
			~Buffer()
			{
				this->unreserve();
			}

		protected:
			/** \brief Reserve the OpenGL buffer.
			 *
			 * Intentionally protected, the buffer should not be reserved without
			 * something to explicitly put in it.
			 */
			void reserve()
			{
				if(0 == _buffer)
				{
					glGenBuffers(1, &_buffer);
				}
			}

		public:
			/** \brief Bind this buffer.
			 *
			 * The user should not be required to call this method.
			 */
			inline void bind() const
			{
				glBindBuffer(B, _buffer);
			}

			/** \brief Get the OpenGL buffer of this.
			 *
			 * @return Buffer index.
			 */
			inline GLuint getBuffer() const
			{
				return _buffer;
			}

			/** \brief Release the OpenGL buffer.
			 */
			void unreserve()
			{
				if(0 != _buffer)
				{
					glDeleteBuffers(1, &_buffer);
					_buffer = 0;
				}
			}

		public:
			/** Get the buffer type.
			 *
			 * @return Buffer type enum.
			 */
			static inline GLenum get_buffer_type()
			{
				return B;
			}
	};

	/** \brief Element buffer.
	 *
	 * Used to feed element arrays into the video card.
	 *
	 * After creating an element buffer, update it with the contents of an
	 * array. You may later update subsegments of the buffer, but unless you do
	 * it from the same array, the results might be unexpected.
	 *
	 * The buffer may at any time be updated to match the contents of any given
	 * element array.
	 */
	template <typename T> class BufferE : public Buffer<GL_ELEMENT_ARRAY_BUFFER>
	{
		protected:
			/** Size of the contents saved in the last full update. */
			unsigned _array_size;

		public:
			/** \brief Get size of this element buffer.
			 *
			 * @return Size in elements (not in primitives).
			 */
			inline unsigned getSize() const
			{
				return _array_size;
			}

		public:
			/** \brief Constructor. */
			BufferE() { }

			/** \brief Constructor.
			 *
			 * @param array Array to immediately feed.
			 */
			BufferE(const ArrayE<T> &array)
			{
				this->update(array);
			}

			/** \brief Constructor.
			 *
			 * @param array Vector of triangles to immediately feed.
			 */
			BufferE(const std::vector<Triangle> &array)
			{
				this->update(array);
			}

			/** \brief Destructor. */
			~BufferE() { }

		public:
			/** \brief Collect elements.
			 *
			 * Recursively collects all elements from given LOD. Then write back the
			 * element lists in them.
			 */
			void update(Lod &lod);

			/** \brief Update data in the video card.
			 *
			 * Updates all data within an array.
			 *
			 * In most cases, this should be the only function that is necessary to be
			 * called for an element buffer after creating it.
			 *
			 * @param first First element.
			 * @param last Last element.
			 */
			void update(const ArrayE<T> &array)
			{
				_array_size = array.getSize();
				this->reserve();
				this->bind();
				glBufferData(get_buffer_type(),
						_array_size * sizeof(T),
						array.getData(),
						GL_STATIC_DRAW);
			}

			/** \brief Update data in the video card.
			 *
			 * Updates all data within an array.
			 *
			 * In most cases, this should be the only function that is necessary to be
			 * called for an element buffer after creating it.
			 *
			 * @param first First element.
			 * @param last Last element.
			 */
			void update(const std::vector<Triangle> &array)
			{
				_array_size = static_cast<unsigned>(array.size()) * 3;

				T *buf = new T[_array_size];
				unsigned ii = 0;
				BOOST_FOREACH(const Triangle &vv, array)
				{
					buf[ii + 0] = vv.a();
					buf[ii + 1] = vv.b();
					buf[ii + 2] = vv.c();
					ii += 3;
				}

				this->reserve();
				this->bind();
				glBufferData(get_buffer_type(),
						_array_size * sizeof(T),
						buf,
						GL_STATIC_DRAW);

				delete[] buf;
			}

			/** \brief Update a subsegment of data in the video card.
			 *
			 * Numbers are given as full elements, so updating cannot be done for
			 * element boundaries.
			 *
			 * In most situations, the user should have no need to call this function,
			 * as it specifically updates a subsegment of the data within a buffer. If
			 * the full data within an array has not been updated yet, this may have
			 * weird consequences.
			 *
			 * @param first First element.
			 * @param last Last element.
			 */
			void update(const ArrayE<T> &array, unsigned first, unsigned last)
			{
				this->reserve();
				this->bind();
				glBufferSubData(get_buffer_type(),
						first * sizeof(T),
						(last - first) * sizeof(T),
						array.getData() + first);
			}

		public:
			/** \brief Draw according to the elements in this.
			 *
			 * @param type What kind of elements are we drawing?
			 * @param cnt How many elements to feed.
			 */
			inline void draw(unsigned type, unsigned cnt) const
			{
				this->bind();
				glDrawElements(type, cnt, get_opengl_type<T>(), NULL);
			}

			/** \brief Draw according to the elements in this.
			 *
			 * Like draw(unsigned, unsigned), but feeds all elements.
			 *
			 * @param type What kind of elements are we drawing?
			 */
			inline void draw(unsigned type) const
			{
				this->draw(type, _array_size);
			}
	};

	/** \brief Convenience typedef. */
	typedef BufferE<GLuint> BufferElem;

	/** \brief Child version of element buffer.
	 *
	 * Instead of an element buffer, this is a sub-buffer for elements. It does
	 * not contain a buffer instance per se, but is instead initialized into an
	 * index pointer and a size.
	 */
	template <typename T> class BufferSubE
	{
		protected:
			/** Index (in bytes) in the host buffer. */
			unsigned _idx;

			/** Number of elements for drawing. */
			unsigned _count;

		public:
			/** \brief Accessor.
			 *
			 * @return Number of elements in this buffer.
			 */
			inline unsigned getCount() const
			{
				return _count;
			}

			/** \brief Accessor.
			 *
			 * @return Drawing index.
			 */
			inline unsigned getIdx() const
			{
				return _idx;
			}

		public:
			/** \brief Empty constructor. */
			BufferSubE() { }

			/** \brief Constructor.
			 *
			 * @param idx Buffer index.
			 * @param count Number of elements.
			 * */
			BufferSubE(unsigned idx, unsigned count) :
				_idx(idx),
				_count(count) { }

			/** \brief Destructor. */
			~BufferSubE() { }

		public:
			/** \brief Update the corresponding subsegment into the video card.
			 *
			 * This should probably not be called usually. It is provided for
			 * academic interest.
			 *
			 * @param data Pointer to the data from which to update.
			 */
			void update(void *data)
			{
				glBufferSubData(GL_ELEMENT_ARRAY_BUFFER,
						_idx * sizeof(T),
						_count * sizeof(T),
						data);
			}

		public:
			/** \brief Assign new values for this buffer.
			 *
			 * @param idx Buffer index.
			 * @param count Number of elements.
			 */
			inline void assign(unsigned idx, unsigned count)
			{
				_idx = idx;
				_count = count;
			}

			/** \brief Draw according to the elements in this.
			 *
			 * @param type What kind of elements are we drawing?
			 * @param cnt How many elements to feed.
			 */
			inline void draw(unsigned type, unsigned cnt) const
			{
				glDrawElements(type, cnt, get_opengl_type<T>(),
						static_cast<uint8_t*>(NULL) + _idx);
			}

			/** \brief Draw according to the elements in this.
			 *
			 * Like draw(unsigned, unsigned), but feeds all elements.
			 *
			 * @param type What kind of elements are we drawing?
			 */
			inline void draw(unsigned type) const
			{
				this->draw(type, _count);
			}
	};

	/** \brief Convenience typedef. */
	typedef BufferSubE<GLuint> BufferSubElem;

	/** \brief An interleaved buffer for typical static data.
	 */
	class BufferInterleavedBBTCNV : public Buffer<GL_ARRAY_BUFFER>
	{
		public:
			/** Block size of one element in the graphics card buffer. */
			static const unsigned BLOCKSIZE = 4 + 16 + 8 + 4 + 8 + 12;

		public:
			/** \brief Constructor. */
			BufferInterleavedBBTCNV() { }

			/** \brief Destructor. */
			~BufferInterleavedBBTCNV() { }

		public:
			/** \brief Feed.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param br Attribute for 4-component bone reference data.
			 * @param bw Attribute for 4-component bone weight data.
			 * @param tt Attribute for 2-component texture coordinate data.
			 * @param cc Attribute for 4-component color data.
			 * @param nn Attribute for 3-component normal data.
			 * @param vv Attribute for 3-component vertex data.
			 */
			void feed(const Attribute &br, const Attribute &bw, const Attribute &tt,
					const Attribute &cc, const Attribute &nn, const Attribute &vv) const;

			/** \brief Update.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param br Attribute for 4-component bone reference data.
			 * @param bw Attribute for 4-component bone weight data.
			 * @param tt 2-component texture coordinate data.
			 * @param cc 4-component color data.
			 * @param nn 3-component normal data.
			 * @param vv 3-component vertex data.
			 */
			void update(const std::vector<math::vec4u> &br,
					const std::vector<math::vec4f> &bw,
					const std::vector<math::vec2f> &tt,
					const std::vector<Color> &cc,
					const std::vector<math::vec3f> &nn,
					const std::vector<math::vec3f> &vv);
	};

	/** \brief An interleaved buffer for typical static data.
	 */
	class BufferInterleavedTCNV : public Buffer<GL_ARRAY_BUFFER>
	{
		public:
			/** Block size of one element in the graphics card buffer. */
			static const unsigned BLOCKSIZE = 8 + 4 + 8 + 12;

		public:
			/** \brief Constructor. */
			BufferInterleavedTCNV() { }

			/** \brief Destructor. */
			~BufferInterleavedTCNV() { }

		public:
			/** \brief Feed.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt Attribute for 2-component texture coordinate data.
			 * @param cc Attribute for 4-component color data.
			 * @param nn Attribute for 3-component normal data.
			 * @param vv Attribute for 3-component vertex data.
			 */
			void feed(const Attribute &tt, const Attribute &cc, const Attribute &nn,
					const Attribute &vv) const;

			/** \brief Update.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt 2-component texture coordinate data.
			 * @param cc 4-component color data.
			 * @param nn 3-component normal data.
			 * @param vv 3-component vertex data.
			 */
			void update(const std::vector<math::vec2f> &tt,
					const std::vector<Color> &cc,
					const std::vector<math::vec3f> &nn,
          const std::vector<math::vec3f> &vv);
  };

	/** \brief An interleaved buffer for typical static data.
	 */
	class BufferInterleavedCNV : public Buffer<GL_ARRAY_BUFFER>
	{
		public:
			/** Block size of one element in the graphics card buffer. */
			static const unsigned BLOCKSIZE = 4 + 8 + 12;

		public:
			/** \brief Constructor. */
			BufferInterleavedCNV() { }

			/** \brief Destructor. */
			~BufferInterleavedCNV() { }

		public:
			/** \brief Feed.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param cc Attribute for 4-component color data.
			 * @param nn Attribute for 3-component normal data.
			 * @param vv Attribute for 3-component vertex data.
			 */
			void feed(const Attribute &cc, const Attribute &nn,
					const Attribute &vv) const;

			/** \brief Update.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param cc 4-component color data.
			 * @param nn 3-component normal data.
			 * @param vv 3-component vertex data.
			 */
			void update(const std::vector<Color> &cc,
					const std::vector<math::vec3f> &nn,
          const std::vector<math::vec3f> &vv);
  };

	/** \brief An interleaved buffer for overlay graphics (or the like).
	 */
	class BufferInterleavedTCV : public Buffer<GL_ARRAY_BUFFER>
	{
		public:
			/** Block size of one element in the graphics card buffer. */
			static const unsigned BLOCKSIZE = 8 + 4 + 12;

		public:
			/** \brief Constructor. */
			BufferInterleavedTCV() { }

			/** \brief Destructor. */
			~BufferInterleavedTCV() { }

		public:
			/** \brief Feed.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt Attribute for 2-component texture coordinate data.
			 * @param cc Attribute for 4-component color data.
			 * @param vv Attribute for 3-component vertex data.
			 */
			void feed(const Attribute &tt, const Attribute &cc,
					const Attribute &vv) const;

			/** \brief Update.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt 2-component texture coordinate data.
			 * @param cc 4-component color data.
			 * @param vv 3-component vertex data.
			 */
			void update(const std::vector<math::vec2f> &tt,
					const std::vector<Color> &cc,
          const std::vector<math::vec3f> &vv);
  };

	/** \brief An interleaved buffer for static data.
	 *
	 * This assumes most of the data is contained in other sources, for example
	 * textures and normalmaps.
	 *
	 * The texture coordinates are assumed to be non-repeating and clamped.
	 */
	class BufferInterleavedTV : public Buffer<GL_ARRAY_BUFFER>
	{
		public:
			/** Block size of one element in the graphics card buffer. */
			static const unsigned BLOCKSIZE = 8 + 12;

		public:
			/** \brief Constructor. */
			BufferInterleavedTV() { }

			/** \brief Destructor. */
			~BufferInterleavedTV() { }

		public:
			/** \brief Feed.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt Attribute for 2-component texture coordinate data.
			 * @param vv Attribute for 3-component vertex data.
			 */
			void feed(const Attribute &tt, const Attribute &vv) const;

			/** \brief Update.
			 *
			 * Passing order borrows from glInterleavedArrays.
			 *
			 * @param tt 2-component texture coordinate data.
			 * @param vv 3-component vertex data.
			 */
			void update(const std::vector<math::vec2f> &tt,
          const std::vector<math::vec3f> &vv);
  };
}

#endif
