#ifndef GFX_GLYPH_HPP
#define GFX_GLYPH_HPP

#include "gfx/texture_2d.hpp"

namespace gfx
{
	class Font;

	/** \brief One glyph contained in a font.
	 *
	 * Mostly invisible to the user due to being actually accessed by the font
	 * class internally.
	 *
	 * All the measurements are expressed in floating-point values that are
	 * fractions of the full font size.
	 */
	class Glyph
	{
		private:
			/** The unicode number. */
			wchar_t _code;

			/** Width of the character. */
			float _width;

			/** Height of the character. */
			float _height;

			/** Distance from the current write coordinates to the left border of the
			 * glyph. */
			float _left;

			/** Distance from the current write coordinates to the top border of the
			 * glyph. */
			float _top;

			/** Advance value after writing this glyph. */
			math::vec2f _advance;

			/** Lower left texture coordinate. */
			math::vec2f _st1;

			/** Upper right texture coordinate. */
			math::vec2f _st2;

			/** Font page this glyph is contained in in the font. */
			unsigned _page;

			/** Texture reference for easy access. */
			const Texture2D *_texture;

		public:
			/** Constructor. */
			Glyph() { }

			/** Constructor.
			 *
			 * Glyphs should generally not be generated manually.
			 * 
			 * @param pcode Code.
			 * @param pwidth Width.
			 * @param pheight Height.
			 * @param pleft Left.
			 * @param ptop Top.
			 * @param padvx X advance.
			 * @param padvy Y advance.
			 * @param ps1 First S coordinate.
			 * @param pt1 First T coordinate.
			 * @param ps2 Second S coordinate.
			 * @param pt2 Second T coordinate.
			 * @param ppage Page in font.
			 */
			Glyph(wchar_t pcode, float pwidth, float pheight, float pleft,
					float ptop, float padvx, float padvy, float ps1, float pt1,
					float ps2, float pt2, unsigned ppage);

			/** Destructor. */
			~Glyph() { }

		public:
			/** \brief Calculate for use.
			 *
			 * Will fill the font-dependant fields of this.
			 *
			 * @param fnt Host font.
			 */
			void compile(const Font &fnt);

		public:
			/** \brief Bind the texture used in this glyph into use.
			 */
			inline void bind() const
			{
				_texture->bind();
			}

			/** \brief Accessor.
			 *
			 * @return Advance of this.
			 */
			inline const math::vec2f& getAdvance() const
			{
				return _advance;
			}

			/** \brief Accessor.
			 *
			 * @return X advance of this.
			 */
			inline float getAdvanceX() const
			{
				return _advance.x();
			}

			/** \brief Accessor.
			 *
			 * @return Advance of this.
			 */
			inline float getAdvanceY() const
			{
				return _advance.y();
			}

			/** \brief Accessor.
			 *
			 * @return Code of this.
			 */
			inline wchar_t getCode() const
			{
				return _code;
			}

			/** \brief Accessor.
			 *
			 * @return Height of this.
			 */
			inline float getHeight() const
			{
				return _height;
			}

			/** \brief Accessor.
			 *
			 * @return Left of this.
			 */
			inline float getLeft() const
			{
				return _left;
			}

			/** \brief Accessor.
			 *
			 * @return Texture of this.
			 */
			inline const Texture2D* getTexture() const
			{
				return _texture;
			}

			/** \brief Accessor.
			 *
			 * @return st1 of this.
			 */
			inline const math::vec2f& getST1() const
			{
				return _st1;
			}

			/** \brief Accessor.
			 *
			 * @return st2 of this.
			 */
			inline const math::vec2f& getST2() const
			{
				return _st2;
			}

			/** \brief Accessor.
			 *
			 * @return Top of this.
			 */
			inline float getTop() const
			{
				return _top;
			}

			/** \brief Accessor.
			 *
			 * @return Width of this.
			 */
			inline float getWidth() const
			{
				return _width;
			}

		public:
			/** \brief Output this to a stream.
			 *
			 * @param ss Target stream.
			 * @return Stream after input.
			 */
			std::ostream& put(std::ostream &ss) const;

			/** \brief Output to stream.
			 *
			 * @param lhs Left-hand-side operand.
			 * @param rhs Right-hand-side operand.
			 * @return Stream after output.
			 */
			friend inline std::ostream& operator<<(std::ostream &lhs,
					const Glyph &rhs)
			{
				return rhs.put(lhs);
			}
	};
}

#endif
