#ifndef GFX_IMAGE_HPP
#define GFX_IMAGE_HPP

#include "data/generic.hpp"
#include "gfx/color.hpp"
#include "math/vec.hpp"

#include "boost/gil/image.hpp"
#include "boost/gil/typedefs.hpp"
#include "boost/gil/image_view_factory.hpp"
#include "boost/gil/extension/io/jpeg_io.hpp"
#include "boost/gil/extension/io/png_io.hpp"
#include "boost/filesystem.hpp"

namespace gfx
{
	/** \brief Image, as opposed to texture.
	 *
	 * Constructed over boost::gil image types.
	 */
	template<typename T> class Image
	{
		protected:
			/** gil image type. */
			T _img;

			/** \brief Filename this.
			 *
			 * Not set by default, only if loaded from somewhere.
			 */
			std::string m_filename;

		public:
			/** \brief Accessor.
			 *
			 * @return The filename as a string.
			 */
			inline const std::string& getFilename() const
			{
				return m_filename;
			}

			/** \brief Tell if a filename is for a JPEG file.
			 *
			 * Note that not accepting filenames in caps is intentional.
			 *
			 * @param pfname Filename.
			 * @return True if yes, false if no.
			 */
			static inline bool is_jpeg(const std::string &pfname)
			{
				return ((pfname.find(".jpeg") != std::string::npos) ||
						(pfname.find(".jpg") != std::string::npos));
			}

			/** \brief Tell if a filename is for a PNG file.
			 *
			 * Note that not accepting filenames in caps is intentional.
			 *
			 * @param pfname Filename.
			 * @return True if yes, false if no.
			 */
			static inline bool is_png(const std::string &pfname)
			{
				return (pfname.find(".png") != std::string::npos);
			}

		public:
			/** \brief Empty constructor. */
			Image() { }

			/** \brief Create a new image of certain dimensions.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			Image(unsigned pw, unsigned ph) :
				_img(pw, ph) { }

			/** \brief Constructor.
			 *
			 * Will throw an exception on failure.
			 *
			 * @param pfname Filename to load from.
			 */
			Image(const std::string &pfname)
			{
        if(! boost::filesystem::exists( pfname ) )
          BOOST_THROW_EXCEPTION( std::runtime_error(pfname + " not found!") );
				this->loadBase(pfname);
			}

			/** \brief Destructor. */
			~Image() { }

		public:
			/** \brief Replace the contents of this image.
			 *
			 * Replaces the contents  with the image from a file.
			 *
			 * @param pfname File to load.
			 */
			void loadBase(const std::string &pfname)
			{
				std::string filename = data::open_search(pfname);
				if(is_png(pfname))
				{
					boost::gil::png_read_image(filename, _img);
					m_filename = pfname;
				}
				else if(is_jpeg(pfname))
				{
					boost::gil::jpeg_read_image(filename, _img);
					m_filename = pfname;
				}
				else
				{
					std::ostringstream str;
					str << "unknown image type: " << pfname;
					BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
				}
			}

			/** \brief Replace the contents of this image.
			 *
			 * As loadBase(const std::string&), but does not try to load as jpeg.
			 *
			 * @param pfname File to load.
			 */
			void loadBaseDisregardJpeg(const std::string &pfname)
			{
				std::string filename = data::open_search(pfname);
				if(is_png(pfname))
				{
					boost::gil::png_read_image(filename, _img);
					m_filename = pfname;
				}
				else
				{
					std::ostringstream str;
					str << "unknown image type: " << pfname;
					BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
				}
			}

		public:
			/** \brief Get the height of this image.
			 *
			 * @return Height of this.
			 */
			inline unsigned getHeight() const
			{
				return static_cast<unsigned>(_img.height());
			}

			/** \brief Get the width of this image.
			 *
			 * @return Width of this.
			 */
			inline unsigned getWidth() const
			{
				return static_cast<unsigned>(_img.width());
			}
	};

	/** \brief GRAY8 image.
	 */
	class ImageGray8 :
		public Image<boost::gil::gray8_image_t>
	{
		private:
			/** Grayscale view. */
			boost::gil::gray8_view_t _view;

		public:
			/** \brief Empty constructor. */
			ImageGray8() { }

			/** \brief Create a new image of certain dimensions.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			ImageGray8(unsigned pw, unsigned ph);

			/** \brief Constructor.
			 *
			 * Will throw an exception on failure.
			 *
			 * @param pfname Filename to load from.
			 */
			ImageGray8(const std::string &pfname);

			/** \brief Destructor. */
			~ImageGray8() { }

		public:
			/** \brief Get a heightfield value from image.
			 *
			 * Input coordinates outside [0, 1] are clamped.
			 *
			 * @param px Image x.
			 * @param py Image y.
			 * @return Heightfield value [0, 1].
			 */
			float getHeightValue(float px, float py) const;

			/** \brief Load a new image into this.
			 *
			 * @param pfname Filename to load.
			 */
			void load(const std::string &pfname);

			/** \brief Write this into a file.
			 *
			 * @param pfname Filename to write.
			 */
			void write(const std::string &pfname);

		public:
			/** \brief Get pointer to the data within the image.
			 *
			 * @return Data pointer.
			 */
			inline const uint8_t* getData() const
			{
				return &(_view[0][0]);
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color vector.
			 */
			inline float getLuminance(unsigned px, unsigned py) const
			{
				return static_cast<float>(_view(px, py)) / 255.0f;
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color object.
			 */
			inline uint8_t getPixel(unsigned px, unsigned py) const
			{
				return _view(px, py);
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pg Gray component.
			 */
			inline void setLuminance(unsigned px, unsigned py, float pg)
			{
				_view(px, py) = static_cast<uint8_t>(math::lround(pg * 255.0f));
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pg Gray component.
			 */
			inline void setPixel(unsigned px, unsigned py, uint8_t pg)
			{
				_view(px, py) = pg;
			}
	};

	/** \brief GRAY16 image.
	 */
	class ImageGray16 :
		public Image<boost::gil::gray16_image_t>
	{
		private:
			/** Grayscale view. */
			boost::gil::gray16_view_t _view;

		public:
			/** \brief Empty constructor. */
			ImageGray16() { }

			/** \brief Create a new image of certain dimensions.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			ImageGray16(unsigned pw, unsigned ph);

			/** \brief Constructor.
			 *
			 * Will throw an exception on failure.
			 *
			 * @param pfname Filename to load from.
			 */
			ImageGray16(const std::string &pfname);

			/** \brief Destructor. */
			~ImageGray16() { }

		public:
			/** \brief Load a new image into this.
			 *
			 * @param pfname Filename to load.
			 */
			void load(const std::string &pfname);

			/** \brief Write this into a file.
			 *
			 * @param pfname Filename to write.
			 */
			void write(const std::string &pfname);

		public:
			/** \brief Get pointer to the data within the image.
			 *
			 * @return Data pointer.
			 */
			inline const uint8_t* getData() const
			{
				return reinterpret_cast<const uint8_t*>(&(_view[0][0]));
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color object.
			 */
			inline Color getPixel(unsigned px, unsigned py) const
			{
				const uint8_t *data =
					reinterpret_cast<const uint8_t*>(&(_view(px, py)));
				uint8_t gray = data[0],
								alpha = data[1];
				return Color(gray, gray, gray, alpha);
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pg Gray component.
			 */
			inline void setPixel(unsigned px, unsigned py, uint8_t pg, uint8_t pa)
			{
				uint8_t *data = reinterpret_cast<uint8_t*>(&(_view(px, py)));
				data[0] = pg;
				data[1] = pa;
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pg Gray component.
			 * @param pa Alpha component.
			 */
			inline void setPixel(unsigned px, unsigned py, float pg, float pa)
			{
				this->setPixel(px, py,
						static_cast<uint8_t>(math::lround(pg * 255.0f)),
						static_cast<uint8_t>(math::lround(pa * 255.0f)));
			}

			/** \brief Set a pixel.
			 *
			 * Red, Green and Blue in Color are averaged.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param col Color component.
			 */
			inline void setPixel(unsigned px, unsigned py, const Color &col)
			{
				this->setPixel(px, py,
						(col.r() + col.g() + col.b()) * (1.0f / 3.0f),
						col.a());
			}
	};

	/** \brief RGB24 image.
	 */
	class ImageRGB :
		public Image<boost::gil::rgb8_image_t>
	{
		private:
			/** Grayscale view. */
			boost::gil::rgb8_view_t _view;

		public:
			/** \brief Empty constructor. */
			ImageRGB() { }

			/** \brief Create a new image of certain dimensions.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			ImageRGB(unsigned pw, unsigned ph);

			/** \brief Constructor.
			 *
			 * Will throw an exception on failure.
			 *
			 * @param pfname Filename to load from.
			 */
			ImageRGB(const std::string &pfname);

			/** \brief Destructor. */
			~ImageRGB() { }

		public:
			/** \brief Load a new image into this.
			 *
			 * @param pfname Filename to load.
			 */
			void load(const std::string &pfname);

			/** \brief Write this into a file.
			 *
			 * @param pfname Filename to write.
			 */
			void write(const std::string &pfname);

		public:
			/** \brief Get pointer to the data within the image.
			 *
			 * @return Data pointer.
			 */
			inline const uint8_t* getData() const
			{
				return &(_view[0][0]);
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color object.
			 */
			inline Color getPixel(unsigned px, unsigned py)
			{
				boost::gil::rgb8_view_t::iterator iter = _view.at(px, py);
				return Color((*iter)[0], (*iter)[1], (*iter)[2], 255);
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color vector.
			 */
			inline math::vec3f getPixelVec(unsigned px, unsigned py)
			{
				boost::gil::rgb8_view_t::iterator iter = _view.at(px, py);
				return math::vec3f(
						static_cast<float>((*iter)[0]) / 255.0f,
						static_cast<float>((*iter)[1]) / 255.0f,
						static_cast<float>((*iter)[2]) / 255.0f);
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pr R component.
			 * @param pg G component.
			 * @param pb B component.
			 */
			inline void setPixel(unsigned px, unsigned py, uint8_t pr, uint8_t pg,
					uint8_t pb)
			{
				boost::gil::rgb8_view_t::iterator iter = _view.at(px, py);
				(*iter)[0] = pr;
				(*iter)[1] = pg;
				(*iter)[2] = pb;
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param col Color value.
			 */
			inline void setPixel(unsigned px, unsigned py, const Color &col)
			{
				this->setPixel(px, py,
						static_cast<uint8_t>(col.ri()),
						static_cast<uint8_t>(col.gi()),
						static_cast<uint8_t>(col.bi()));
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param col Color value.
			 */
			inline void setPixel(unsigned px, unsigned py, const math::vec3f &col)
			{
				this->setPixel(px, py, static_cast<uint8_t>(col.x() * 255.0f),
						static_cast<uint8_t>(col.y() * 255.0f),
						static_cast<uint8_t>(col.z() * 255.0f));
			}
	};

	/** \brief RGBA32 image.
	 */
	class ImageRGBA :
		public Image<boost::gil::rgba8_image_t>
	{
		private:
			/** Grayscale view. */
			boost::gil::rgba8_view_t _view;

		public:
			/** \brief Empty constructor. */
			ImageRGBA() { }

			/** \brief Create a new image of certain dimensions.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 */
			ImageRGBA(unsigned pw, unsigned ph);

			/** \brief Constructor.
			 *
			 * Will throw an exception on failure.
			 *
			 * @param pfname Filename to load from.
			 */
			ImageRGBA(const std::string &pfname);

			/** \brief Destructor. */
			~ImageRGBA() { }

		public:
			/** \brief Load a new image into this.
			 *
			 * @param pfname Filename to load.
			 */
			void load(const std::string &pfname);

			/** \brief Write this into a file.
			 *
			 * @param pfname Filename to write.
			 */
			void write(const std::string &pfname);

		public:
			/** \brief Get pointer to the data within the image.
			 *
			 * @return Data pointer.
			 */
			inline const uint8_t* getData() const
			{
				return &(_view[0][0]);
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color object.
			 */
			inline Color getPixel(unsigned px, unsigned py)
			{
				boost::gil::rgba8_view_t::iterator iter = _view.at(px, py);
				return Color((*iter)[0], (*iter)[1], (*iter)[2], (*iter)[3]);
			}

			/** \brief Get a color value.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @return Color vector.
			 */
			inline math::vec4f getPixelVec(unsigned px, unsigned py)
			{
				boost::gil::rgba8_view_t::iterator iter = _view.at(px, py);
				return math::vec4f(
						static_cast<float>((*iter)[0]) / 255.0f,
						static_cast<float>((*iter)[1]) / 255.0f,
						static_cast<float>((*iter)[2]) / 255.0f,
						static_cast<float>((*iter)[3]) / 255.0f);
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param pr R component.
			 * @param pg G component.
			 * @param pb B component.
			 * @param pa A component.
			 */
			inline void setPixel(unsigned px, unsigned py, uint8_t pr, uint8_t pg,
					uint8_t pb, uint8_t pa)
			{
				boost::gil::rgba8_view_t::iterator iter = _view.at(px, py);
				(*iter)[0] = pr;
				(*iter)[1] = pg;
				(*iter)[2] = pb;
				(*iter)[3] = pa;
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param col Color value.
			 */
			inline void setPixel(unsigned px, unsigned py, const Color &col)
			{
				this->setPixel(px, py,
						static_cast<uint8_t>(col.ri()),
						static_cast<uint8_t>(col.gi()),
						static_cast<uint8_t>(col.bi()),
						static_cast<uint8_t>(col.ai()));
			}

			/** \brief Set a pixel.
			 *
			 * @param px X coordinate.
			 * @param py Y coordinate.
			 * @param col Color value.
			 */
			inline void setPixel(unsigned px, unsigned py, const math::vec4f &col)
			{
				this->setPixel(px, py, static_cast<uint8_t>(col.x() * 255.0f),
						static_cast<uint8_t>(col.y() * 255.0f),
						static_cast<uint8_t>(col.z() * 255.0f),
						static_cast<uint8_t>(col.w() * 255.0f));
			}
	};

	/** Convenience typedef. */
	typedef boost::shared_ptr<ImageGray8> ImageGray8Sptr;

	/** Convenience typedef. */
	typedef boost::shared_ptr<ImageGray16> ImageGray16Sptr;

	/** Convenience typedef. */
	typedef boost::shared_ptr<ImageRGB> ImageRGBSptr;

	/** Convenience typedef. */
	typedef boost::shared_ptr<ImageRGBA> ImageRGBASptr;
}

#endif
