#ifndef GFX_MESH_HPP
#define GFX_MESH_HPP

#include "data/store.hpp"
#include "gfx/lod.hpp"
#include "gfx/texture_2d.hpp"
#include "math/mat.hpp"

namespace gfx
{
	class Shader;

	/** \brief A basic mesh class.
	 *
	 * Each mesh is either static or animated. The base class contains data
	 * relevant to both of the variants.
	 *
	 * Note that if some of the data is not present, the mesh is just plain empty.
	 */
	class Mesh
	{
		protected:
			/** Attribute data array. */
			std::vector<math::vec4u> m_bone_ref;

			/** Attribute data array. */
			std::vector<math::vec4f> m_bone_weight;

			/** Attribute data array. */
			std::vector<Color> m_color;

			/** Attribute data array. */
			std::vector<math::vec3f> m_normal;

			/** Attribute data array. */
			std::vector<math::vec2f> m_texcoord;

			/** Attribute data array. */
			std::vector<math::vec3f> m_vertex;

			/** Face data. */
			Lod m_lod;

			/** Element buffer. */
			BufferElem m_elem;

			/** Textures in this mesh. */
			std::vector<const Texture2D*> m_textures;

			/** \brief Offset of this mesh.
			 *
			 * Note that for most purposes, the offset will be zero. It is meant as
			 * the mount point within parent objects, if applicable. The parent
			 * bjects should just plain read this offset as their position without
			 * changing it.
			 */
			math::vec3f m_offset;

		public:
			/** \brief Accessor.
			 *
			 * @return Offset.
			 */
			inline const math::vec3f& getOffset() const
			{
				return m_offset;
			}

			/** \brief Setter.
			 *
			 * @param op New offset.
			 */
			inline void setOffset(const math::vec3f &op)
			{
				m_offset = op;
			}

		public:
			/** \brief Empty constructor. */
			Mesh();

			/** \brief Load constructor.
			 *
			 * @param pfname Filename to load from.
			 */
			Mesh(const std::string &pfname);

			/** \brief Destructor. */
			virtual ~Mesh() { }

		protected:
			/** \brief Calculate normals.
			 *
			 * Calculates normals from vertex and face data.
			 */
			void calcNormals();

		public:
			/** \brief Add texture from file.
			 *
			 * Loads the texture in the process.
			 *
			 * Throws an exception on failure.
			 *
			 * @param type Texture type.
			 * @param pfname Filename to load from.
			 * @param clamp True to clamp instead of repeat.
			 */
			void addTextureFile(const std::string &type, const std::string &pfname,
					bool clamp = false);

			/** \brief Get the boundary of this mesh.
			 *
			 * @return Boundary calculated from this.
			 */
			math::rect3f getBoundary() const;

			/** \brief Load this from a file.
			 *
			 * Unknown elements will be passed into a specific interpretation
			 * function that is implemented in inheriting classes.
			 *
			 * @param pfname Filename to load.
			 */
			void load(const std::string &pfname);

			/** \brief Scale this object.
			 *
			 * Multiplies all vertices (but nothing else) with the given values.
			 *
			 * @param svec Scale vector.
			 */
			void scale(const math::vec3f &svec);

			/** \brief Element update task.
			 */
			void taskElem();

			/** \brief Texture task (filename).
			 *
			 * @param type Type to load as.
			 * @param pfname Filename to load from.
			 */
			void taskTextureFile(const char *type, const std::string &pfname);

			/** \brief Texture task (filename, clamped).
			 *
			 * @param type Type to load as.
			 * @param pfname Filename to load from.
			 */
			void taskTextureFileClamped(const char *type, const std::string &pfname);

			/** \brief Texture task (RGB image).
			 *
			 * @param type Type to load as.
			 * @param img Image to use.
			 */
			void taskTextureGray8(const char *type, const ImageGray8 *img);

			/** \brief Texture task (RGB image).
			 *
			 * @param type Type to load as.
			 * @param img Image to use.
			 */
			void taskTextureGray16(const char *type, const ImageGray16 *img);

			/** \brief Texture task (RGB image).
			 *
			 * @param type Type to load as.
			 * @param img Image to use.
			 */
			void taskTextureRGB(const char *type, const ImageRGB *img);

			/** \brief Texture task (RGBA image).
			 *
			 * @param type Type to load as.
			 * @param img Image to use.
			 */
			void taskTextureRGBA(const char *type, const ImageRGBA *img);

			/** \brief Translate this object.
			 *
			 * Translates all vertices (but nothing else) with the given values.
			 *
			 * @param tvec Translation vector.
			 */
			void translate(const math::vec3f &tvec);

		public:
			/** \brief Add one texture into this.
			 *
			 * Virtual function to handle adding a new texture.
			 *
			 * The id contains the type or detailed name of the texture that was either
			 * loaded from an XML file or specified explicitly. Its processing may be
			 * omitted at will. The default loader accepts strings starting with
			 * "texture". Further implementations may also use such strings as
			 * "normalmap" or "volume".
			 *
			 * The parameter is passed as void pointer due to type problems for the
			 * Texture inheritance. The default implementation simply interprets the
			 * pointer as Texture2D and pushes it into the texture array of this.
			 *
			 * @param id Identifier.
			 * @param tex Texture.
			 */
			virtual void addTexture(const std::string &id, const void *tex);

			/** \brief Compile this to be ready for drawing.
			 */
			virtual void compile() = 0;

			/** \brief Draw this.
			 *
			 * It is expected that the camera matrix be loaded into the OpenGL
			 * stack prior to calling this method.
			 *
			 * @param psh Shader to use.
			 * @param pmat Modelview matrix to use.
			 */
			virtual void draw(const Shader &psh, const math::mat4f &op) const = 0;

		protected:
			/** \brief Read one vertex from a property tree.
			 *
			 * Inheriting classes should call this base method.
			 *
			 * @param subtree Tree to read from.
			 */
			virtual void readVertex(const void *subtree);

			/** \brief Free all resources required by the font.
			 * 
			 * Should be done before deinitializing OpenGL.
			 *
			 * Inheriting classes should call this base method.
			 */
			virtual void unreserve();

		public:
			/** \brief Get the face normal for a given face.
			 *
			 * @param tri Triangle to use.
			 * @return Resulting normal.
			 */
			inline const math::vec3f calcNormal(const Triangle &tri)
			{
				const math::vec3f &va = m_vertex[tri.a()];

				return math::cross(m_vertex[tri.b()] - va, m_vertex[tri.c()] - va);
			}

			/** \brief Scale this object.
			 *
			 * Multiplies all vertices (but nothing else) with the given value.
			 *
			 * @param op Scale parameter.
			 */
			inline void scale(float op)
			{
				this->scale(math::vec3f(op, op, op));
			}
	};

	/** Convenience typedef. */
	typedef boost::shared_ptr<Mesh> MeshSptr;

	/** \brief Storage for meshes. */
	class MeshStore :
		public data::Store<Mesh>
	{
		public:
			/** \brief Constructor. */
			MeshStore();

			/** \brief Destructor. */
			virtual ~MeshStore() { }

		public:
			/** \cond */
			virtual void loadMeta(container_type &dst,
					const std::string &pfname,
					const data::StoreCreator<Mesh> &creator);
			/** \endcond */
	};

	/** Global store. */
	extern MeshStore mesh_store;
}

#endif
