#include "gfx/surface_screen.hpp"

#include "boost/lexical_cast.hpp"

#include <sstream>

using namespace gfx;

SurfaceScreen::SurfaceScreen(int pw, int ph, int pb, bool fs) :
	_screen(NULL)
{
	this->setInternalState(pw, ph, pb);

	SDL_Init(SDL_INIT_VIDEO);
	setGlAttributes();
	_screen = SDL_SetVideoMode(pw, ph, pb, createSdlFlags(fs));

	GLenum glew_err = glewInit();
	if(GLEW_OK != glew_err)
	{
		std::stringstream err;
		err << glewGetErrorString(glew_err);
		BOOST_THROW_EXCEPTION(std::runtime_error(err.str()));
	}

	// Reserver internal OpenGL variables.
	try
	{
		if(!shader_2d)
		{
			shader_2d = new Shader("shader/2d.xml");
		}
		if(!shader_2d_font)
		{
			shader_2d_font = new Shader("shader/2d_font.xml");
		}
		if(!shader_2d_texture)
		{
			shader_2d_texture = new Shader("shader/2d_texture.xml");
		}
	}
	catch(std::runtime_error err)
	{
		std::cerr << err.what() << std::endl;
	}
}

SurfaceScreen::~SurfaceScreen()
{
	// Delete all internal OpenGL variables that were reserved.
	delete shader_2d;
	shader_2d = NULL;
	delete shader_2d_font;
	shader_2d_font = NULL;
	delete shader_2d_texture;
	shader_2d_texture = NULL;

	if(_screen)
	{
		SDL_FreeSurface(_screen);
		SDL_Quit();
	}
}

void SurfaceScreen::clear(bool pc, bool pd)
{
	glClear((pc ? GL_COLOR_BUFFER_BIT : 0) | (pd ? GL_DEPTH_BUFFER_BIT : 0));
}
	
void SurfaceScreen::setBoundary(int px, int py, int pw, int ph)
{
	glViewport(px, py, pw, ph);
}

void SurfaceScreen::update()
{
	SDL_GL_SwapBuffers();
}

std::ostream& SurfaceScreen::put(std::ostream &ss) const
{
	return ss << "Screen: " << this->Surface::put(ss);
}

boost::tuple<unsigned, unsigned, unsigned>
SurfaceScreen::parseResolution(const std::string & r)
{
  size_t cx = r.find("x"),
				 ca = r.find("@");
  
  if( cx == std::string::npos ||
      ca == std::string::npos )
  {
		BOOST_THROW_EXCEPTION(
				std::runtime_error("Invalid resolution string '" + r + "'"));
  }

  std::string sw = r.substr(0, cx);
  std::string sh = r.substr(cx+1, ca-cx-1);
  std::string sb = r.substr(ca+1);

  return boost::make_tuple(
      boost::lexical_cast<unsigned>(sw),
      boost::lexical_cast<unsigned>(sh),
      boost::lexical_cast<unsigned>(sb) );  
}

