#include "gfx/texture_2d.hpp"

#include "data/generic.hpp"
#include "gfx/image.hpp"
#include "thr/dispatch.hpp"
#include "ui/generic.hpp"

namespace gil = boost::gil;
using namespace gfx;

Texture2DStore gfx::texture_2d_store;

/** Logging message.
 *
 * @param type Type of logging.
 * @param filename Filename to potentially log.
 */
static void log_open(const char *type, const std::string &fname)
{
	if(0 < fname.length())
	{
		ui::log(std::string("loading ") + std::string(type) +
				std::string(" texture '") + fname + std::string("'"));
	}
}

Texture2D::Texture2D(const std::string &pfname, bool clamp)
{
	this->load(pfname, clamp);
}

Texture2D::Texture2D(const ImageGray8 &pimg, bool clamp)
{
	this->adapt(pimg, clamp);
}

Texture2D::Texture2D(const ImageGray16 &pimg, bool clamp)
{
	this->adapt(pimg, clamp);
}

Texture2D::Texture2D(const ImageRGB &pimg, bool clamp)
{
	this->adapt(pimg, clamp);
}

Texture2D::Texture2D(const ImageRGBA &pimg, bool clamp)
{
	this->adapt(pimg, clamp);
}

void Texture2D::adapt(const ImageGray8 &pimg, bool clamp)
{
	this->setInternalState(pimg.getWidth(), pimg.getHeight(), 8);
	log_open("gray8", pimg.getFilename());
	thr::wait_privileged(boost::bind(&Texture2D::adaptData, this, pimg.getData(), clamp));
}

void Texture2D::adapt(const ImageGray16 &pimg, bool clamp)
{
	this->setInternalState(pimg.getWidth(), pimg.getHeight(), 16);
	log_open("gray16", pimg.getFilename());
	thr::wait_privileged(boost::bind(&Texture2D::adaptData, this, pimg.getData(), clamp));
}

void Texture2D::adapt(const ImageRGB &pimg, bool clamp)
{
	this->setInternalState(pimg.getWidth(), pimg.getHeight(), 24);
	log_open("rgb", pimg.getFilename());
	thr::wait_privileged(boost::bind(&Texture2D::adaptData, this, pimg.getData(), clamp));
}

void Texture2D::adapt(const ImageRGBA &pimg, bool clamp)
{
	this->setInternalState(pimg.getWidth(), pimg.getHeight(), 32);
	log_open("rgba", pimg.getFilename());
	thr::wait_privileged(boost::bind(&Texture2D::adaptData, this, pimg.getData(), clamp));
}

void Texture2D::adaptData(const uint8_t *pdata, bool clamp)
{
	this->unreserve();

	glEnable(GL_TEXTURE_2D);
  glGenTextures(1, &_id);
	this->bind();

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	GLenum wrap = clamp ? GL_CLAMP_TO_EDGE : GL_REPEAT;
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrap);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrap);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, 2.0f);

	GLenum pformat = bpp_to_pformat(m_b);
	gluBuild2DMipmaps(GL_TEXTURE_2D, pformat, m_w, m_h, pformat, GL_UNSIGNED_BYTE,
			pdata);
}

void Texture2D::load(const std::string &pfname, bool clamp)
{
	try
	{
		this->adapt(ImageRGBA(pfname), clamp);
#if defined(ENABLE_DEBUG)
		std::cout << *this << std::endl;
#endif
		return;
	}
	catch(...) { }

	try
	{
		this->adapt(ImageRGB(pfname), clamp);
		/*std::cout << static_cast<int>(ptr[0]) << " " <<
			static_cast<int>(ptr[1]) << " " <<
			static_cast<int>(ptr[2]) << " ;; " <<
			static_cast<int>(ptr[3]) << " " <<
			static_cast<int>(ptr[4]) << " " <<
			static_cast<int>(ptr[5]) << std::endl;*/
#if defined(ENABLE_DEBUG)
		std::cout << *this << std::endl;
#endif
		return;
	}
	catch(...) { }
	
	try
	{
		this->adapt(ImageGray16(pfname), clamp);
#if defined(ENABLE_DEBUG)
		std::cout << *this << std::endl;
#endif
		return;
	}
	catch(...) { }

	try
	{
		this->adapt(ImageGray8(pfname), clamp);
#if defined(ENABLE_DEBUG)
		std::cout << *this << std::endl;
#endif
		return;
	}
	catch(...) { }

	std::ostringstream str;
	str << "image not of a supported type: " << pfname;
	BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
}

std::ostream& gfx::operator<<(std::ostream &lhs, const Texture2D &rhs)
{
	lhs << "Texture: " << rhs.getWidth() << "x" << rhs.getHeight() << " ";
	switch(rhs.getBpp())
	{
		case 8:
			return lhs << "GRAY";

		case 24:
			return lhs << "RGB";

		case 32:
			return lhs << "RGBA";

		default:
			return lhs << "UNKNOWN";
	}
}

Texture2DStore::Texture2DStore() :
	data::Store<Texture2D>(".png", ".jpeg"),
	m_temp_texture(NULL) { }

const Texture2D* Texture2DStore::replaceTempTexture(const char *pfname)
{
	// If clear.
	if(NULL == pfname)
	{
		if(NULL != m_temp_texture)
		{
			this->remove(m_temp_texture_name);
			m_temp_texture = NULL;
			m_temp_texture_name.assign("");
		}
		return NULL;
	}

	// Need canonname if not empty.
	std::string canon = canonize(std::string(pfname));

	// Might already match.
	if(NULL != m_temp_texture)
	{
		if(m_temp_texture_name == canon)
		{
			return m_temp_texture;
		}

		// Remove previous.
		this->remove(m_temp_texture_name);
		m_temp_texture = NULL;
		m_temp_texture_name.assign("");
	}

	// Get new.
	container_type &con = this->load<Texture2D>(pfname);
	m_temp_texture = con[0].get();
	m_temp_texture_name = canon;
	return m_temp_texture;
}

