#ifndef GFX_TEXTURE_2D_HPP
#define GFX_TEXTURE_2D_HPP

#include "data/store.hpp"
#include "gfx/surface_base.hpp"
#include "gfx/texture.hpp"

namespace gfx
{
	class ImageGray8;
	class ImageGray16;
	class ImageRGB;
	class ImageRGBA;

	/** \brief Regular 2D texture.
	 *
	 * Textures can be loaded from files and drawn into, they may also be bound for
	 * drawing other objects.
	 *
	 * The class only supports single-color luminance or RGB formats with or without
	 * alpha.
	 */
	class Texture2D :
		public Texture<GL_TEXTURE_2D>,
		public SurfaceBase
	{
		public:
			/** \brief Empty constructor. */
			inline Texture2D() { }

			/** \brief Image loading constructor.
			 *
			 * @param pfname Filename to load the texture from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture2D(const std::string &pfname, bool clamp = false);

			/** \brief Image initializing consturctor.
			 *
			 * @param pimg Image to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture2D(const ImageGray8 &pimg, bool clamp = false);

			/** \brief Image initializing consturctor.
			 *
			 * @param pimg Image to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture2D(const ImageGray16 &pimg, bool clamp = false);

			/** \brief Image initializing consturctor.
			 *
			 * @param pimg Image to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture2D(const ImageRGB &pimg, bool clamp = false);

			/** \brief Image initializing consturctor.
			 *
			 * @param pimg Image to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture2D(const ImageRGBA &pimg, bool clamp = false);

			/** Destructor. */
			inline ~Texture2D() { }

		public:
			/** \brief Adapt an image.
			 *
			 * @param pimg Image object.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const ImageGray8 &pimg, bool clamp = false);

			/** \brief Adapt an image.
			 *
			 * @param pimg Image object.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const ImageGray16 &pimg, bool clamp = false);

			/** \brief Adapt an image.
			 *
			 * @param pimg Image object.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const ImageRGB &pimg, bool clamp = false);

			/** \brief Adapt an image.
			 *
			 * @param pimg Image object.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const ImageRGBA &pimg, bool clamp = false);

			/** \brief Adapt a data block.
			 *
			 * @param pdata Data.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adaptData(const uint8_t *pdata, bool clamp = false);

			/** \brief Load an image file, replace the current one if it exists.
			 *
			 * Should generally not be used. Just create new textures.
			 *
			 * If loading of the image fails, the current texture will not be modified.
			 *
			 * @param pfname File to load.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void load(const std::string &pfname, bool clamp = false);

		public:
			/** \cond */
			friend std::ostream& operator<<(std::ostream &lhs, const Texture2D &rhs);
			/** \endcond */
	};

	/** Convenience typedef. */
	typedef boost::shared_ptr<Texture2D> Texture2DSptr;

	/** \brief Output texture to a stream.
	 *
	 * @param lhs Left-hand-side operand.
	 * @param rhs Right-hand-side operand.
	 * @return Stream after input.
	 */
	std::ostream& operator<<(std::ostream &lhs, const Texture2D &rhs);

	/** \brief Storage for textures. */
	class Texture2DStore :
		public data::Store<Texture2D>
	{
		private:
			/** Temporary one-shot texture. */
			const Texture2D* m_temp_texture;

			/** Temporary one-shot texture name. */
			std::string m_temp_texture_name;

		public:
			/** \brief Accessor.
			 *
			 * @return Current texture.
			 */
			const Texture2D* getTempTexture() const
			{
				return m_temp_texture;
			}

		public:
			/** \brief Constructor. */
			Texture2DStore();

			/** \brief Destructor. */
			virtual ~Texture2DStore() { }

		public:
			/** \brief Replace the temporary texture.
			 *
			 * @param pfname New filename path to use, NULL to just remove.
			 * @return The texture just replaced.
			 */
			const Texture2D* replaceTempTexture(const char *pfname);
	};

	/** Global store. */
	extern Texture2DStore texture_2d_store;
}

#endif
