#include "gfx/texture_3d.hpp"

#include "gfx/volume.hpp"
#include "thr/dispatch.hpp"
#include "ui/generic.hpp"

using namespace gfx;

Texture3DStore gfx::texture_3d_store;

/** Logging message.
 *
 * @param type Type of logging.
 * @param filename Filename to potentially log.
 */
static void log_open(const char *type, const std::string &fname)
{
	if(0 < fname.length())
	{
		ui::log(std::string("loading ") + std::string(type) +
				std::string(" volume '") + fname + std::string("'"));
	}
}

Texture3D::Texture3D(unsigned px, unsigned py, unsigned pz, unsigned pb) :
	VolumeBase(px, py, pz, pb) { }

Texture3D::Texture3D(unsigned px, unsigned py, unsigned pz, unsigned pb,
		const void *pdata, bool clamp) :
	VolumeBase(px, py, pz, pb)
{
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, pdata, clamp));
}

Texture3D::Texture3D(const std::string &pfname, bool clamp)
{
	this->load(pfname, clamp);
}

Texture3D::Texture3D(const VolumeGray8 &volume, bool clamp)
{
	this->adapt(volume, clamp);
}

Texture3D::Texture3D(const VolumeGray16 &volume, bool clamp)
{
	this->adapt(volume, clamp);
}

Texture3D::Texture3D(const VolumeRGB &volume, bool clamp)
{
	this->adapt(volume, clamp);
}

Texture3D::Texture3D(const VolumeRGBA &volume, bool clamp)
{
	this->adapt(volume, clamp);
}

void Texture3D::adaptData(const void *data, bool clamp)
{
	glEnable(GL_TEXTURE_3D);
	glGenTextures(1, &_id);
	this->bind();

	glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	GLenum wrap = clamp ? GL_CLAMP_TO_EDGE : GL_REPEAT;
	glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, wrap);
	glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, wrap);
	glTexParameteri(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, wrap);

	// The following line replaces the functionality of gluBuild3DMipmaps.
	// Windows does not have OpenGL 1.3, so the function is missing and I'm
	// leaving mipmap generation to the card.
	glTexParameteri(GL_TEXTURE_3D, GL_GENERATE_MIPMAP, GL_TRUE);

	GLenum pformat = bpp_to_pformat(m_b);
	//gluBuild3DMipmaps(GL_TEXTURE_3D, pformat, _w, _h, _d, pformat, GL_UNSIGNED_BYTE, data);
  glTexImage3D(GL_TEXTURE_3D, 0, pformat, m_w, m_h, m_d, 0, pformat,
			GL_UNSIGNED_BYTE, data);
}

void Texture3D::adapt(unsigned pw, unsigned ph, unsigned pd, unsigned pb,
		const void *data, bool clamp)
{
	this->setInternalState(pw, ph, pd, pb);
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, data, clamp));
}

void Texture3D::adapt(const VolumeGray8 &volume, bool clamp)
{
	this->setInternalState(volume.getWidth(), volume.getHeight(), volume.getDepth(), 8);
	log_open("gray8", volume.getFilename());
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, volume.getData(), clamp));
}

void Texture3D::adapt(const VolumeGray16 &volume, bool clamp)
{
	this->setInternalState(volume.getWidth(), volume.getHeight(), volume.getDepth(), 16);
	log_open("gray16", volume.getFilename());
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, volume.getData(), clamp));
}

void Texture3D::adapt(const VolumeRGB &volume, bool clamp)
{
	this->setInternalState(volume.getWidth(), volume.getHeight(), volume.getDepth(), 24);
	log_open("rgb", volume.getFilename());
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, volume.getData(), clamp));
}

void Texture3D::adapt(const VolumeRGBA &volume, bool clamp)
{
	this->setInternalState(volume.getWidth(), volume.getHeight(), volume.getDepth(), 32);
	log_open("rgba", volume.getFilename());
	thr::wait_privileged(boost::bind(&Texture3D::adaptData, this, volume.getData(), clamp));
}

void Texture3D::load(const std::string &pfname, bool clamp)
{
	try
	{
		this->adapt(VolumeRGBA(pfname), clamp);
		return;
	}
	catch(...) { }

	try
	{
		this->adapt(VolumeRGB(pfname), clamp);
		return;
	}
	catch(...) { }
	
	try
	{
		this->adapt(VolumeGray16(pfname), clamp);
		return;
	}
	catch(...) { }

	try
	{
		this->adapt(VolumeGray8(pfname), clamp);
		return;
	}
	catch(...) { }

	std::ostringstream str;
	str << "volume not of a supported type: " << pfname;
	BOOST_THROW_EXCEPTION(std::runtime_error(str.str()));
}

Texture3DStore::Texture3DStore() :
	data::Store<Texture3D>(".vol", "") { }

