#ifndef GFX_TEXTURE_3D_HPP
#define GFX_TEXTURE_3D_HPP

#include "data/store.hpp"
#include "gfx/volume_base.hpp"
#include "gfx/texture.hpp"

namespace gfx
{
	class VolumeGray8;
	class VolumeGray16;
	class VolumeRGB;
	class VolumeRGBA;

	/** \brief 3D texture construct.
	 *
	 * 3D volumetric texture.
	 *
	 * Slight overlap with 2D texturing accounts for some dubious terminology.
	 * 
	 * As per OpenGL coordinate system, the texture S coordinate is X, T is Y
	 * and R is Z. In practice, this means that SR is the plane in texture space
	 * just as XZ is the plane in the world space. However, due to the way
	 * texture information is stored, the data block to be passed is specified
	 * S first, then T, then R.
	 */
	class Texture3D :
		public Texture<GL_TEXTURE_3D>,
		public VolumeBase
	{
		public:
			/** \brief Empty constructor. */
			Texture3D() { }

			/** \brief Constructor.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 * @param pd Depth.
			 * @param pb Bit depth.
			 */
			Texture3D(unsigned pw, unsigned ph, unsigned pd, unsigned pb);

			/** \brief Constructor.
			 *
			 * Immediately adapts data from given block.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 * @param pd Depth.
			 * @param pb Bit depth.
			 * @param pdata Data to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(unsigned pw, unsigned ph, unsigned pd, unsigned pb,
					const void *pdata, bool clamp = false);

			/** \brief Load constructor.
			 *
			 * @param pfname Filename to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(const std::string &pfname, bool clamp = false);

			/** \brief Constructor.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(const VolumeGray8 &volume, bool clamp = false);

			/** \brief Constructor.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(const VolumeGray16 &volume, bool clamp = false);

			/** \brief Constructor.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(const VolumeRGB &volume, bool clamp = false);

			/** \brief Constructor.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			Texture3D(const VolumeRGBA &volume, bool clamp = false);

			/** \brief Destructor. */
			~Texture3D() { }

		public:
			/** \brief Adapt a data block.
			 *
			 * Sets internal parameters before reading the data.
			 *
			 * @param pw Width.
			 * @param ph Height.
			 * @param pd Depth.
			 * @param pb Bit depth.
			 * @param data Texture data.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(unsigned pw, unsigned ph, unsigned pd, unsigned pb,
					const void *data, bool clamp = false);

			/** \brief Adapt a volume.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const VolumeGray8 &volume, bool clamp = false);

			/** \brief Adapt a volume.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const VolumeGray16 &volume, bool clamp = false);

			/** \brief Adapt a volume.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const VolumeRGB &volume, bool clamp = false);

			/** \brief Adapt a volume.
			 *
			 * @param volume Volume to adapt.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adapt(const VolumeRGBA &volume, bool clamp = false);

			/** \brief Adapt a data block.
			 *
			 * Assumes internal parameters already set.
			 *
			 * @param data Texture data.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void adaptData(const void *data, bool clamp = false);

			/** \brief Load texture from file.
			 *
			 * @param pfname Filename to load from.
			 * @param clamp True to clamp texcoords instead of repeat (default: no).
			 */
			void load(const std::string &pfname, bool clamp = false);
	};

	/** \brief Convenience typedef. */
	typedef boost::shared_ptr<Texture3D> Texture3DSptr;

	/** \brief Storage for volumes. */
	class Texture3DStore :
		public data::Store<Texture3D>
	{
		public:
			/** \brief Constructor. */
			Texture3DStore();

			/** \brief Destructor. */
			virtual ~Texture3DStore() { }
	};

	/** Global store. */
	extern Texture3DStore texture_3d_store;
}

#endif
