#include "gfx/volume.hpp"

#include "data/generic.hpp"
#include "gfx/color.hpp"
#include "math/generic.hpp"
#include "math/random.hpp"

#include <fstream>
#include <sstream>

using namespace gfx;

void Perlin::fill(VolumeGray8 &op, unsigned level)
{
	op.fillGarble();
}

Color Perlin::noise(const std::list<VolumeGray8Sptr> &op, const math::vec3d &tt)
{
	float ns = 0.0f,
				wt = 1.0f;

	BOOST_FOREACH(const VolumeGray8Sptr &vv, op)
	{
		ns += wt * (vv->getAveragePixel(tt.x(), tt.y(), tt.z()) - 0.5f);
		wt *= 0.75f;
	}

	ns = ns + 0.5f;
	if(ns < 0.0f)
	{
		ns = -ns;
	}
	else if(ns > 1.0f)
	{
		ns = 2.0f - ns;
	}

	return Color(ns, ns, ns, ns);
}

/** Default Perlin noise parameters. */
Perlin perlin_default;

Volume::Volume() :
	m_data(NULL) { }

Volume::Volume(unsigned pw, unsigned ph, unsigned pd, unsigned pb) :
	VolumeBase(pw, ph, pd, pb)
{
	m_data = new uint8_t[this->getSizeBytes()];
}

Volume::~Volume()
{
	delete[] m_data;
}

void Volume::clear(uint8_t val)
{
	unsigned limit = this->getSizeBytes();
	for(unsigned ii = 0; (ii < limit); ++ii)
	{
		m_data[ii] = val;
	}
}

void Volume::loadBase(const std::string &pfname, unsigned reqbpp)
{
	this->unreserve();

	data::shristr istr = data::open_read(pfname);
	uint32_t ww, hh, dd, bb;

	istr->read(reinterpret_cast<char*>(&ww), sizeof(uint32_t));
	if(istr->fail())
	{
		std::stringstream sstr;
		sstr << "width could not be read";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	istr->read(reinterpret_cast<char*>(&hh), sizeof(uint32_t));
	if(istr->fail())
	{
		std::stringstream sstr;
		sstr << "height could not be read";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	istr->read(reinterpret_cast<char*>(&dd), sizeof(uint32_t));
	if(istr->fail())
	{
		std::stringstream sstr;
		sstr << "depth could not be read";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	istr->read(reinterpret_cast<char*>(&bb), sizeof(uint32_t));
	if(istr->fail())
	{
		std::stringstream sstr;
		sstr << "bpp could not be read";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	if(reqbpp != bb)
	{
		std::stringstream sstr;
		sstr << "bit depth of file (" << bb <<
			") is not the bit depth of object (" << reqbpp << ")";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	this->setInternalState(ww, hh, dd, bb);

	// Read actual data.
	unsigned bsize = this->getSizeBytes();
	m_data =  new uint8_t[bsize];
	istr->read(reinterpret_cast<char*>(m_data), bsize);
	if(istr->fail())
	{
		this->unreserve();
		std::stringstream sstr;
		sstr << "could not read " << bsize << " bytes of data ";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	m_filename = pfname;
}

void Volume::perlinNoise(Perlin *op)
{
	if(NULL == op)
	{
		op = &perlin_default;
	}

	std::list<VolumeGray8Sptr> pn;
	for(unsigned ii = 2, level = 0; (ii <= m_d); ii*= 2, ++level)
	{
		VolumeGray8Sptr inst(new VolumeGray8(ii, ii, ii));
		pn.push_back(inst);
		op->fill(*inst, level);
	}

	for(unsigned ii = 0; (ii < m_d); ++ii)
	{
		double di = static_cast<double>(ii) / static_cast<double>(m_d - 1);

		for(unsigned jj = 0; (jj < m_h); ++jj)
		{
			double dj = static_cast<double>(jj) / static_cast<double>(m_h - 1);

			for(unsigned kk = 0; (kk < m_w); ++kk)
			{
				double dk = static_cast<double>(kk) / static_cast<double>(m_w - 1);

				this->setPixel(kk, jj, ii, op->noise(pn, math::vec3d(dk, dj, di)));
			}
		}
	}
}

void Volume::unreserve()
{
	delete[] m_data;
	m_data = NULL;
	m_w = m_h = m_w = m_b = 0;
}

void Volume::write(const std::string &pfname)
{
	std::ofstream ostr(pfname.c_str());

	uint32_t ww = m_w,
					 hh = m_h,
					 dd = m_d,
					 bb = m_b;

	ostr.write(reinterpret_cast<char*>(&ww), sizeof(uint32_t));
	if(ostr.fail())
	{
		std::stringstream sstr;
		sstr << "width could be not be written";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	ostr.write(reinterpret_cast<char*>(&hh), sizeof(uint32_t));
	if(ostr.fail())
	{
		std::stringstream sstr;
		sstr << "height could be not be written";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	ostr.write(reinterpret_cast<char*>(&dd), sizeof(uint32_t));
	if(ostr.fail())
	{
		std::stringstream sstr;
		sstr << "depth could be not be written";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	ostr.write(reinterpret_cast<char*>(&bb), sizeof(uint32_t));
	if(ostr.fail())
	{
		std::stringstream sstr;
		sstr << "bpp could be not be written";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	unsigned bsize = this->getSizeBytes();
	ostr.write(reinterpret_cast<char*>(m_data), bsize);
	if(ostr.fail())
	{
		std::stringstream sstr;
		sstr << "could not write " << bsize << " bytes of data";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
}

VolumeGray8::VolumeGray8(unsigned pw, unsigned ph, unsigned pd) :
	Volume(pw, ph, pd, 8) { }

VolumeGray8::VolumeGray8(const std::string &pfname)
{
	this->load(pfname);
}

void VolumeGray8::fillGarble()
{
	for(unsigned ii = 0; (ii < (m_w * m_h * m_d)); ++ii)
	{
		m_data[ii] = static_cast<uint8_t>(math::mrand(0, 255));
	}
}

float VolumeGray8::getAveragePixel(float px, float py, float pz)
{
	float rx = -0.5f + px * static_cast<float>(m_w),
				ry = -0.5f + py * static_cast<float>(m_h),
				rz = -0.5f + pz * static_cast<float>(m_d),
				xpwt = 1.0f - (::ceilf(rx) - rx),
				xnwt = 1.0f - (rx - ::floorf(rx)),
				ypwt = 1.0f - (::ceilf(ry) - ry),
				ynwt = 1.0f - (ry - ::floorf(ry)),
				zpwt = 1.0f - (::ceilf(rz) - rz),
				znwt = 1.0f - (rz - ::floorf(rz));
	int ix1 = static_cast<int>(::floorf(rx)),
			ix2 = static_cast<int>(::ceilf(rx)),
			iy1 = static_cast<int>(::floorf(ry)),
			iy2 = static_cast<int>(::ceilf(ry)),
			iz1 = static_cast<int>(::floorf(rz)),
			iz2 = static_cast<int>(::ceilf(rz));
	float	x1y1z1 = this->getIntensityModulod(ix1, iy1, iz1),
				x2y1z1 = this->getIntensityModulod(ix2, iy1, iz1),
				x1y2z1 = this->getIntensityModulod(ix1, iy2, iz1),
				x1y1z2 = this->getIntensityModulod(ix1, iy1, iz2),
				x2y2z1 = this->getIntensityModulod(ix2, iy2, iz1),
				x1y2z2 = this->getIntensityModulod(ix1, iy2, iz2),
				x2y1z2 = this->getIntensityModulod(ix2, iy1, iz2),
				x2y2z2 = this->getIntensityModulod(ix2, iy2, iz2);
	// X weight first.
	float y1z1 = xnwt * x1y1z1 + xpwt * x2y1z1,
				y2z1 = xnwt * x1y2z1 + xpwt * x2y2z1,
				y1z2 = xnwt * x1y1z2 + xpwt * x2y1z2,
				y2z2 = xnwt * x1y2z2 + xpwt * x2y2z2;
	// Then Y.
	float z1 = ynwt * y1z1 + ypwt * y2z1,
				z2 = ynwt * y1z2 + ypwt * y2z2;
	// Then Z.
	return znwt * z1 + zpwt * z2;
}

float VolumeGray8::getIntensityModulod(int px, int py, int pz)
{
	int ix = math::congr(px, static_cast<int>(m_w)),
			iy = math::congr(py, static_cast<int>(m_h)),
			iz = math::congr(pz, static_cast<int>(m_d));
	return static_cast<float>(m_data[iz * (m_w * m_h) + iy * m_w + ix]) * (1.0f / 255.0f);
}

void VolumeGray8::load(const std::string &pfname)
{
	this->loadBase(pfname, 8);
}

void VolumeGray8::setPixel(unsigned px, unsigned py, unsigned pz,
		const Color &col)
{
	this->setPixel(px, py, pz,
			static_cast<uint8_t>(math::lround((col.r() + col.g() + col.b()) * (255.0f / 3.0f))));
}

VolumeGray16::VolumeGray16(unsigned pw, unsigned ph, unsigned pd) :
	Volume(pw, ph, pd, 16) { }

VolumeGray16::VolumeGray16(const std::string &pfname)
{
	this->load(pfname);
}

void VolumeGray16::load(const std::string &pfname)
{
	this->loadBase(pfname, 8);
}

void VolumeGray16::setPixel(unsigned px, unsigned py, unsigned pz,
		const Color &col)
{
	this->setPixel(px, py, pz,
			static_cast<uint8_t>(math::lround((col.r() + col.g() + col.b()) * (255.0f / 3.0f))),
			static_cast<uint8_t>(col.ai()));
}

VolumeRGB::VolumeRGB(unsigned pw, unsigned ph, unsigned pd) :
	Volume(pw, ph, pd, 24) { }

VolumeRGB::VolumeRGB(const std::string &pfname)
{
	this->load(pfname);
}

void VolumeRGB::load(const std::string &pfname)
{
	this->loadBase(pfname, 24);
}

void VolumeRGB::setPixel(unsigned px, unsigned py, unsigned pz,
		const Color &col)
{
	this->setPixel(px, py, pz,
			static_cast<uint8_t>(col.ri()),
			static_cast<uint8_t>(col.gi()),
			static_cast<uint8_t>(col.bi()));
}

VolumeRGBA::VolumeRGBA(unsigned pw, unsigned ph, unsigned pd) :
	Volume(pw, ph, pd, 32) { }

VolumeRGBA::VolumeRGBA(const std::string &pfname)
{
	this->load(pfname);
}

void VolumeRGBA::combine(const VolumeGray8 &v1, const VolumeGray8 &v2,
		const VolumeGray8 &v3, const VolumeGray8 &v4)
{
	if(!this->hasMatchingDimensions(v1) ||
			!this->hasMatchingDimensions(v2) ||
			!this->hasMatchingDimensions(v3) ||
			!this->hasMatchingDimensions(v4))
	{
		std::stringstream sstr;
		sstr << "all combined volume dimensions do not match";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str().c_str()));
	}

	const uint8_t *data1 = v1.getData();
	const uint8_t *data2 = v2.getData();
	const uint8_t *data3 = v3.getData();
	const uint8_t *data4 = v4.getData();

	unsigned bsize = v1.getSizeBytes(); // All have same.
	for(unsigned ii = 0; (ii < bsize); ++ii)
	{
		unsigned idx = ii * 4;
		m_data[idx + 0] = data1[ii];
		m_data[idx + 1] = data2[ii];
		m_data[idx + 2] = data3[ii];
		m_data[idx + 3] = data4[ii];
	}
}

void VolumeRGBA::combine(const VolumeRGB &v1, const VolumeGray8 &v2)
{
	if(!this->hasMatchingDimensions(v1) ||
			!this->hasMatchingDimensions(v2))
	{
		std::stringstream sstr;
		sstr << "all combined volume dimensions do not match";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str().c_str()));
	}

	const uint8_t *data1 = v1.getData();
	const uint8_t *data2 = v2.getData();

	unsigned bsize = v2.getSizeBytes(); // Advance by smaller volume.
	for(unsigned ii = 0; (ii < bsize); ++ii)
	{
		unsigned idx = ii * 4;
		unsigned jj = ii * 3;
		m_data[idx + 0] = data1[jj + 0];
		m_data[idx + 1] = data1[jj + 1];
		m_data[idx + 2] = data1[jj + 2];
		m_data[idx + 3] = data2[ii];
	}
}

void VolumeRGBA::load(const std::string &pfname)
{
	this->loadBase(pfname, 32);
}

void VolumeRGBA::setPixel(unsigned px, unsigned py, unsigned pz,
		const Color &col)
{
	this->setPixel(px, py, pz,
			static_cast<uint8_t>(col.ri()),
			static_cast<uint8_t>(col.gi()),
			static_cast<uint8_t>(col.bi()),
			static_cast<uint8_t>(col.ai()));
}


