#include <boost/test/unit_test.hpp>

#include "math/mat.hpp"

using namespace math;

namespace
{
	const float tolerance = 0.0001f;
}

#define FLOAT_CHECK_EQUAL(x,y) \
	BOOST_CHECK_CLOSE( (x), (y), tolerance)

BOOST_AUTO_TEST_SUITE( mat )

BOOST_AUTO_TEST_CASE( constructors )
{
	mat2f m0; 
	m0.loadIdentity();
	mat2f m1(m0);

	// Copy-ctor
	for(int i=0; i<2; ++i)
		for(int j=0; j<2; ++j)
			FLOAT_CHECK_EQUAL( m0(i,j), ((i==j) ? 1.0f : 0.0f));

	// Fill-ctor
	mat4f m2(5.0f);
	for(int i=0; i!=4; ++i)
	{
		for(int j=0; j!=4; ++j)
		{
			FLOAT_CHECK_EQUAL(m2(i, j), 5.0f);
		}
	}

	float d[] = 
	{
		1.0f, 2.0f,
		3.0f, 4.0f
	};

	// Copy-from-array-ctor
	mat2f m3(d);
	for(int i = 0; (i != 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			FLOAT_CHECK_EQUAL(m3(i, j), d[j * 2 + i]);
		}
	}

	// Copy-from-higher-dimension-ctor
	mat2f m4(m2);
	for(int i = 0; (i != 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			FLOAT_CHECK_EQUAL( m4(i,j), m2(i,j) );
		}
	}

	// Copy-from-lower-dimension-ctor
	mat4f m5(m3);
	for(int i = 0; (i != 4); ++i)
	{
		for(int j = 0; (j != 4); ++j)
		{
			if((i < 2) && (j < 2))
			{
				FLOAT_CHECK_EQUAL(m5(i, j), m3(i, j));
			}
			else
			{
				FLOAT_CHECK_EQUAL(m5(i, j), 0.0f);
			}
		}
	}
}

BOOST_AUTO_TEST_CASE( accessors )
{
	mat4f m0;
	m0.loadIdentity();

	// Check that operator() and method "at()" yields same result
	for(int i = 0; (i != 4); ++i)
	{
		for(int j = 0; (j != 4); ++j)
		{
			FLOAT_CHECK_EQUAL(m0(i, j), m0.at(i, j));
		}
	}

	// Checking that m0 is writable with operator()
	for(int i = 0; (i < 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			m0(i,j) =
				(static_cast<float>(i) * 0.5f) +
				(static_cast<float>(j) * 2.0f);
		}
	}

	for(int i = 0; (i < 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			FLOAT_CHECK_EQUAL(m0(i, j),
					(static_cast<float>(i) * 0.5f) +
					(static_cast<float>(j) * 2.0f));
		}
	}

	// and same with method at()
	for(int i = 0; (i < 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			m0.at(i,j) =
				(static_cast<float>(i) * 2.0f) +
				(static_cast<float>(j) * 0.5f);
		}
	}

	for(int i = 0; (i < 2); ++i)
	{
		for(int j = 0; (j != 2); ++j)
		{
			FLOAT_CHECK_EQUAL(m0(i,j),
					(static_cast<float>(i) * 2.0f) +
					(static_cast<float>(j) * 0.5f));
		}
	}

	BOOST_CHECK_THROW( m0.at(4,4), std::range_error );

	float d[] =
	{
		1.0f, 2.0f, 3.0f, 4.0f,
		5.0f, 6.0f, 7.0f, 8.0f,
		9.0f, 1.0f, 1.1f, 1.2f,
		1.3f, 1.4f, 1.5f, 1.6f
	};

	mat4f m1(d);
	for(int i=0; i!=4; ++i)
	{
		vec4f col = m1.getColumn(i);
		vec4f row = m1.getRow(i);
		for(int j=0; j!=4; ++j)
		{
			FLOAT_CHECK_EQUAL( col[j], d[i*4+j] );
			FLOAT_CHECK_EQUAL( row[j], d[j*4+i] );
		}
	}

	vec4f v(4.0f, 3.0f, 2.0f, 1.0f);
	for(int i=0; i!=4; ++i)
	{
		m1.setColumn(i,v);
	}
	for(int i=0; i!=4; ++i)
	{
		BOOST_CHECK( m1.getColumn(i) == v );
	}
	for(int i=0; i!=4; ++i)
	{
		m1.setRow(i,v);
	}
	for(int i=0; i!=4; ++i)
	{
		BOOST_CHECK( m1.getRow(i) == v );
	}
}


BOOST_AUTO_TEST_CASE( operators )
{
	mat2f m0;
	m0.loadIdentity();

	for(int i=0; i!=2; ++i)
	{
		for(int j=0; j!=2; ++j)
		{
			FLOAT_CHECK_EQUAL( m0(i,j), ((i==j) ? 1.0f : 0.0f));
		}
	}

	float d[] =
	{
		1.0f, 2.0f, 3.0f, 4.0f,
		5.0f, 6.0f, 7.0f, 8.0f,
		9.0f, 1.0f, 1.1f, 1.2f,
		1.3f, 1.4f, 1.5f, 1.6f
	};

	mat4f m1(d);
	mat4f m2=transpose(m1);
	for(int i=0; i!=4; ++i)
	{
		for(int j=0; j!=4; ++j)
		{
			FLOAT_CHECK_EQUAL( m1(i,j), m2(j,i) );
		}
	}
}

BOOST_AUTO_TEST_SUITE_END()
