#include <boost/test/unit_test.hpp>

#include "math/vec.hpp"

using namespace math;

namespace
{
	const float tolerance = 0.0001f;
}

#define FLOAT_CHECK_EQUAL(x, y) \
	BOOST_CHECK_CLOSE( (x), (y), tolerance )

BOOST_AUTO_TEST_SUITE( vec )

BOOST_AUTO_TEST_CASE( constructors_and_accessors )
{
	vec2f v0(2.0f, 3.0f);
	FLOAT_CHECK_EQUAL(v0.x(), 2.0f);
	FLOAT_CHECK_EQUAL(v0.y(), 3.0f);

	vec2f v1(v0); // copy-ctor
	FLOAT_CHECK_EQUAL(v0.x(), v1.x());
	FLOAT_CHECK_EQUAL(v0.y(), v1.y());

	vec3f v2(v0); // copy-from-lower-dimension
	FLOAT_CHECK_EQUAL(v0.x(), v2.x());
	FLOAT_CHECK_EQUAL(v0.y(), v2.y());
	FLOAT_CHECK_EQUAL(v2.z(), 0.0f);

	const float d[] = {4.0f, 5.0f};
	vec2f v3(d); // copy-from-array
	for(int i=0; i!=2; ++i)
		FLOAT_CHECK_EQUAL( v3.at(i), d[i]);

	vec4f v4(6.5f); // Fill
	for(int i=0; i!=4; ++i)
		FLOAT_CHECK_EQUAL( v4.at(i), 6.5f );

	// at accessor
	FLOAT_CHECK_EQUAL( v0.x(), v0.at(0) );
	FLOAT_CHECK_EQUAL( v0.y(), v0.at(1) );

	// ensure that accessor returns references
	v0.x() = 1.0f;
	v0.at(1) = 2.0f;
	FLOAT_CHECK_EQUAL( v0.at(0), 1.0f);
	FLOAT_CHECK_EQUAL( v0.y(), 2.0f);

	v0[0] = 0.0f;
	FLOAT_CHECK_EQUAL( v0.x(), 0.0f );

	BOOST_CHECK_THROW( v0.at(2), std::range_error);
}

BOOST_AUTO_TEST_CASE( operators )
{
	vec2f v0(2.0f, 3.0f);
	vec2f v1(4.0f, 5.0f);

	// Component-wise tests
	vec2f v2 = v0+v1;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()+v1.x());
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()+v1.y());

	v2 = v0-v1;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()-v1.x());
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()-v1.y());

	v2 = v0*v1;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()*v1.x());
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()*v1.y());

	v2 = v0/v1;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()/v1.x());
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()/v1.y());

	// Scalar
	v2 = v0*2.0f;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()*2.0f);
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()*2.0f);

	v2 = v0/2.0f;
	FLOAT_CHECK_EQUAL(v2.x(), v0.x()/2.0f);
	FLOAT_CHECK_EQUAL(v2.y(), v0.y()/2.0f);


	// Comparison
	vec2f v3(v0);
	BOOST_CHECK( v0==v3 );
	BOOST_CHECK( v0!=v1 );
	BOOST_CHECK( !(v0==v1) );
}

BOOST_AUTO_TEST_CASE( functions )
{
	// Cross-product
	vec3f v0(1.0f, 0.0f, 0.0f);
	vec3f v1(0.0f, 1.0f, 0.0f);
	vec3f v2(0.0f, 0.0f, 1.0f);
	vec3f v3=cross(v0, v1);
	BOOST_CHECK(v3==v2);

	// Dot-product
	vec3f v4(2.0f, 0.2f, 0.0f);
	vec3f v5(1.2f, 4.3f, 5.6f);
	FLOAT_CHECK_EQUAL( dot(v0, v1), 0.0f );
	FLOAT_CHECK_EQUAL( dot(v0, v4),
			v0.x()*v4.x() + v0.y()*v4.y() + v0.z()*v4.z() );
	FLOAT_CHECK_EQUAL( dot(v1, v4),
			v1.x()*v4.x() + v1.y()*v4.y() + v1.z()*v4.z() );
	FLOAT_CHECK_EQUAL( dot(v4, v5),
			v4.x()*v5.x() + v4.y()*v5.y() + v4.z()*v5.z() );
	FLOAT_CHECK_EQUAL( dot(v4, v5), dot(v5, v4) );

	// Length
	FLOAT_CHECK_EQUAL( length2(v4), (2.0f*2.0f + 0.2f*0.2f + 0.0f*0.0f) );
	FLOAT_CHECK_EQUAL( length2(v5), (1.2f*1.2f + 4.3f*4.3f + 5.6f*5.6f) );

	FLOAT_CHECK_EQUAL( length2(v4), length(v4) * length(v4) );
	FLOAT_CHECK_EQUAL( length2(v5), length(v5) * length(v5) );

	// Min/max
	vec3f v6 = min(v4, v5);
	FLOAT_CHECK_EQUAL( v5.x(), v6.x() );
	FLOAT_CHECK_EQUAL( v4.y(), v6.y() );
	FLOAT_CHECK_EQUAL( v4.z(), v6.z() );

	v6 = max(v4, v5);
	FLOAT_CHECK_EQUAL( v4.x(), v6.x() );
	FLOAT_CHECK_EQUAL( v5.y(), v6.y() );
	FLOAT_CHECK_EQUAL( v5.z(), v6.z() );

	// Normalize
	vec3f v7 = normalize(v4);
	float v4_len = length(v4);
	FLOAT_CHECK_EQUAL( v7.x(), v4.x()/v4_len);
	FLOAT_CHECK_EQUAL( v7.y(), v4.y()/v4_len);
	FLOAT_CHECK_EQUAL( v7.z(), v4.z()/v4_len);

	// Distance
	FLOAT_CHECK_EQUAL( dist_point_point(v4, v5), length(v4-v5) );

	// Reflect
	vec3f v8 = reflect(v4, v7);
	vec3f v9 = v4 - 2.0f * dot(v4, v7) * v7;
	FLOAT_CHECK_EQUAL( v8.x(), v9.x() );
	FLOAT_CHECK_EQUAL( v8.y(), v9.y() );
	FLOAT_CHECK_EQUAL( v8.z(), v9.z() );
}

BOOST_AUTO_TEST_SUITE_END()
