#include "ob_globals.hpp"

#include "gfx/mesh_static.hpp"
#include "thr/dispatch.hpp"
#include "ob_atmosphere.hpp"
#include "ob_constants.hpp"
#include "ob_game.hpp"
#include "ob_menu_state.hpp"
#include "ob_planet.hpp"
#include "ob_visualization_city.hpp"
#include "ob_visualization_distort.hpp"
#include "ob_visualization_flak.hpp"
#include "ob_visualization_nuke.hpp"
#include "ob_visualization_orbit.hpp"
#include "ob_visualization_railgun.hpp"
#include "ob_visualization_sun.hpp"

#include <sstream>

using namespace ob;

bool Globals::generate_enabled = false;

Fade ob::fade;
Game *ob::game = NULL;
Globals *ob::glob = NULL;

/** The game thread has stopped processing. */
static bool game_is_ready = false;

/** Thread used for game startups. */
static thr::ThreadSptr game_thread;

/** \brief Task for the creation of the next actual game state.
 *
 * This means, cities and population map.
 */
static void glob_task_game()
{
	if(game != NULL)
	{
		std::stringstream sstr;
		sstr << "can't create a game task when previous game task exists";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	new Game(); // Will set the global value by itself.
	game_is_ready = true;
	fade.setDelta(-OB_FADE_DELTA);
}

/** \brirf Log to console.
 *
 * @param op String to log.
 */
static void log_console(const std::string &op)
{
	glob->getConsole().addRow(ui::wstr_utf8(op));
}

Globals::Globals(const gfx::SurfaceScreen &pscreen, const char *pdetail) :
	_detail_level(pdetail),
	_font("fnt/default.xml"),
	_console(_font, OB_CONSOLE_FONT_SIZE, pscreen),
	_cursor_blank(create_cursor_blank()),
	_cursor_default(SDL_GetCursor()),
	m_menu_game(MenuState::create_menu_game()),
	m_menu_main(MenuState::create_menu_main()),
	_precalculated(false)
{
	_console.setProgress(0);
	_console.setProgressMax(100);
}

Globals::Globals(const gfx::SurfaceScreen &pscreen, const std::string &pdetail) :
	_detail_level(pdetail),
	_font("fnt/default.xml"),
	_console(_font, OB_CONSOLE_FONT_SIZE, pscreen),
	_cursor_blank(create_cursor_blank()),
	_cursor_default(SDL_GetCursor()),
	_precalculated(false)
{
	_console.setProgress(0);
	_console.setProgressMax(100);
}

Globals::~Globals()
{
	this->unreserve();
	SDL_SetCursor(_cursor_default);
	SDL_FreeCursor(_cursor_blank);
}

void Globals::precalc()
{
	if(_detail_level.compare("laptop") == 0)
	{
		this->precalc(6, 3, 512, 64);
		return;
	}
	
	if(_detail_level.compare("desktop") == 0)
	{
		this->precalc(7, 3, 1024, 128);
		return;
	}

	if(_detail_level.compare("bleeding") == 0)
	{
		this->precalc(8, 3, 2048, 128);
		//this->precalc(8, 3, 2048, 256); // 256^3 is just plain ridiculous.
		return;
	}

	if(_detail_level.compare("custom") != 0)
	{
		std::stringstream sstr;
		sstr << "unknown detail level: " << _detail_level;
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}

	// TODO: implement.
	BOOST_THROW_EXCEPTION(std::runtime_error("unimplemented"));
}

void Globals::precalc(unsigned subdivide, unsigned coalesce, unsigned texsize,
		unsigned volsize)
{
	thr::wait_privileged(boost::bind(&Globals::unreserve, this));

	snd::play_stream("snd/music_menu.ogg");
	
	m_shader_atmosphere = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_atmosphere.xml"));
	m_shader_distort = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_distort.xml"));
	m_shader_object = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_static.xml"));
	m_shader_overlay = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_overlay.xml"));
	m_shader_overlay_billboard = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_overlay_billboard.xml"));
	m_shader_overlay_line = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_overlay_line.xml"));
	m_shader_planet = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_world.xml"));
	m_shader_planet_schematic = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_world_schematic.xml"));
	m_shader_sun = gfx::ShaderSptr(
			new gfx::Shader("shader/3d_sun.xml"));

	m_texture_flak_ammo = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_flak_ammo", false))[0].get();
	m_texture_icon_bullet_flak = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_bullet_flak", true))[0].get();
	m_texture_icon_bullet_railgun = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_bullet_railgun", true))[0].get();
	m_texture_icon_city = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_city", true))[0].get();
	m_texture_icon_missile_anti_nuke = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_missile_anti_nuke", true))[0].get();
	m_texture_icon_missile_anti_ship = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_missile_anti_ship", true))[0].get();
	m_texture_icon_silo = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/icon_silo", true))[0].get();
	m_texture_menu = (gfx::texture_2d_store.load<gfx::Texture2D>("gfx/general/ob_mainmenu_background", false))[0].get();
	m_texture_missile_anti_nuke = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_missile_anti_nuke", true))[0].get();
	m_texture_missile_anti_nuke_warning = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_warning_yellow_frame_1", true))[0].get();
	m_texture_missile_anti_ship = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_missile_anti_ship", true))[0].get();
	m_texture_missile_anti_ship_warning = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_warning_red_frame_1", true))[0].get();
	m_texture_missile_nuke = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_missile_nuke", true))[0].get();
	m_texture_nuke_ammo = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_nuke_ammo", false))[0].get();
	m_texture_railgun_ammo = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_railgun_ammo", false))[0].get();
	m_texture_reload = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_reload", true))[0].get();
	m_texture_silo_anti_nuke = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_anti_nuke", true))[0].get();
	m_texture_silo_anti_nuke_inactive = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_anti_nuke_new", true))[0].get();
	m_texture_silo_anti_ship = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_anti_ship", true))[0].get();
	m_texture_silo_anti_ship_inactive = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_anti_ship_new", true))[0].get();
	m_texture_silo_both = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_both", true))[0].get();
	m_texture_silo_both_inactive = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_silo_both_new", true))[0].get();
	m_texture_skull = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_skull", false))[0].get();
	m_texture_target = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/billboards/billboard_target", true))[0].get();
	m_texture_trace = (gfx::texture_2d_store.load<gfx::Texture2D>(
				"gfx/textures/texture_trace", false))[0].get();

	const char *fname_particle[OB_PARTICLE_COUNT] =
	{
		"gfx/billboards/particle_crackle_1.png",
		"gfx/billboards/particle_crackle_2.png",
		"gfx/billboards/particle_crackle_3.png",
		"gfx/billboards/particle_crackle_4.png",
		"gfx/billboards/particle_glow_medium.png",
		"gfx/billboards/particle_glow_sharp.png",
		"gfx/billboards/particle_glow_soft.png",
		"gfx/billboards/particle_shockwave_narrow_1.png",
		"gfx/billboards/particle_shockwave_narrow_2.png",
		"gfx/billboards/particle_shockwave_narrow_3.png",
		"gfx/billboards/particle_shockwave_narrow_4.png",
		"gfx/billboards/particle_shockwave_wide_1.png",
		"gfx/billboards/particle_shockwave_wide_2.png",
		"gfx/billboards/particle_shockwave_wide_3.png",
		"gfx/billboards/particle_shockwave_wide_4.png",
		"gfx/billboards/particle_smoke_hard_1.png",
		"gfx/billboards/particle_smoke_hard_2.png",
		"gfx/billboards/particle_smoke_hard_3.png",
		"gfx/billboards/particle_smoke_hard_4.png",
		"gfx/billboards/particle_smoke_soft_1.png",
		"gfx/billboards/particle_smoke_soft_2.png",
		"gfx/billboards/particle_smoke_soft_3.png",
		"gfx/billboards/particle_smoke_soft_4.png",
		"gfx/billboards/particle_sparkle_1.png",
		"gfx/billboards/particle_sparkle_2.png",
		"gfx/billboards/particle_sparkle_3.png",
		"gfx/billboards/particle_sparkle_4.png",
		"gfx/billboards/particle_sparkle_5.png",
		"gfx/billboards/particle_sparkle_6.png"
	};
	for(unsigned ii = 0; (ii < OB_PARTICLE_COUNT); ++ii)
	{
		m_texture_particle[ii] = (gfx::texture_2d_store.load<gfx::Texture2D>(
					fname_particle[ii], true))[0].get();
	}

	snd::sample_store.load<snd::Sample>("snd/orbital_bombardment");
	m_sample_alarm = snd::sample_store.get("ob_contact");
	m_sample_alarm_over = snd::sample_store.get("ob_alarm_over");
	m_sample_contact = snd::sample_store.get("ob_contact");
	m_sample_flak = snd::sample_store.get("ob_flak_short");
	m_sample_illegal = snd::sample_store.get("ob_illegal_action");
	m_sample_locked = snd::sample_store.get("ob_locked");
	m_sample_nuke = snd::sample_store.get("ob_nuke");
	m_sample_railgun = snd::sample_store.get("ob_railgun");
	m_sample_railgun_lock = snd::sample_store.get("ob_railgun_lock_long");
	m_sample_route_change = snd::sample_store.get("ob_route_change");
	m_sample_route_change_accepted = snd::sample_store.get("ob_route_change_accepted");
	m_sample_target_destroyed = snd::sample_store.get("ob_target_destroyed");

	gfx::mesh_store.load<gfx::MeshStatic>("mdl/siegecruiser");

	m_mesh_missile_anti = (gfx::mesh_store.load<gfx::MeshStatic>("mdl/missile_anti"))[0].get();
	m_mesh_missile_nuke = (gfx::mesh_store.load<gfx::MeshStatic>("mdl/missile_icbm"))[0].get();
	m_mesh_silo = (gfx::mesh_store.load<gfx::MeshStatic>("mdl/silo"))[0].get();
	gfx::mesh_store.put("atmosphere", new Atmosphere(subdivide - 2, coalesce));
	gfx::mesh_store.put("city", new VisualizationCity());
	gfx::mesh_store.put("distort", new VisualizationDistort());
	gfx::mesh_store.put("planet", new Planet(subdivide, coalesce, texsize,
				volsize, &m_height_map_planet, generate_enabled));
	gfx::mesh_store.put("orbit", new VisualizationOrbit());
	gfx::mesh_store.put("nuke_marker", new VisualizationNuke());
	gfx::mesh_store.put("sun", new VisualizationSun());
	m_mesh_bullet_flak = gfx::mesh_store.put("bullet_flak",
			new VisualizationFlak());
	m_mesh_bullet_railgun = gfx::mesh_store.put("bullet_railgun",
			new VisualizationRailgun());

	std::string texstr;
	{
		std::stringstream sstr;
		sstr << '_' << texsize;
		texstr = sstr.str();
	}
	std::string textype("texture");
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_back")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_back") + texstr);
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_bottom")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_bottom") + texstr);
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_front")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_front") + texstr);
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_left")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_left") + texstr);
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_right")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_right") + texstr);
	gfx::mesh_store.load<VisualizationMesh>("mdl/skybox_top")[0]->
		addTextureFile(textype, std::string("gfx/maps/enviroment_map_top") + texstr);

	_precalculated = true;
}

void Globals::reserve_shader(gfx::ShaderSptr &dst, const char *cstr)
{
	dst = gfx::ShaderSptr(new gfx::Shader(cstr));
}

void Globals::unreserve()
{
	_precalculated = false;
	m_shader_atmosphere = gfx::ShaderSptr();
	m_shader_distort = gfx::ShaderSptr();
	m_shader_object = gfx::ShaderSptr();
	m_shader_overlay = gfx::ShaderSptr();
	m_shader_overlay_billboard = gfx::ShaderSptr();
	m_shader_overlay_line = gfx::ShaderSptr();
	m_shader_planet = gfx::ShaderSptr();
	m_shader_sun = gfx::ShaderSptr();
	snd::sample_store.clear();
	m_sample_alarm = NULL;
	m_sample_alarm_over = NULL;
	m_sample_contact = NULL;
	m_sample_flak = NULL;
	m_sample_illegal = NULL;
	m_sample_locked = NULL;
	m_sample_nuke = NULL;
	m_sample_railgun = NULL;
	m_sample_route_change = NULL;
	m_sample_route_change_accepted = NULL;
	m_sample_target_destroyed = NULL;
	gfx::mesh_store.clear();
	m_mesh_bullet_flak = NULL;
	m_mesh_bullet_railgun = NULL;
	m_mesh_missile_anti = NULL;
	m_mesh_missile_nuke = NULL;
	m_mesh_silo = NULL;
	gfx::texture_2d_store.clear();
	m_texture_flak_ammo = NULL;
	m_texture_icon_bullet_flak = NULL;
	m_texture_icon_bullet_railgun = NULL;
	m_texture_icon_city = NULL;
	m_texture_icon_missile_anti_nuke = NULL;
	m_texture_icon_missile_anti_ship = NULL;
	m_texture_icon_silo = NULL;
	m_texture_missile_anti_nuke = NULL;
	m_texture_missile_anti_ship = NULL;
	m_texture_missile_nuke = NULL;
	m_texture_railgun_ammo = NULL;
	m_texture_silo_anti_nuke = NULL;
	m_texture_silo_anti_ship = NULL;
	m_texture_silo_both = NULL;
	m_texture_silo_anti_nuke_inactive = NULL;
	m_texture_silo_anti_ship_inactive = NULL;
	m_texture_silo_both_inactive = NULL;
	m_texture_target = NULL;
	m_texture_trace = NULL;
	for(unsigned ii = 0; (ii < m_texture_particle.size()); ++ii)
	{
		m_texture_particle[ii] = NULL;
	}
	gfx::texture_3d_store.clear();
}

SDL_Cursor* Globals::create_cursor_blank()
{
	uint8_t cdata[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
	return SDL_CreateCursor(cdata, cdata, 8, 1, 0, 0);
}

void ob::glob_init(const gfx::SurfaceScreen &pscreen,
		const char *pdetail)
{
	glob_quit();
	ui::log.connect(ui::log_default);
	glob = new Globals(pscreen, pdetail);
	ui::log.connect(log_console);
}

void ob::glob_init(const gfx::SurfaceScreen &pscreen,
		const std::string &pdetail)
{
	glob_init(pscreen, pdetail.c_str());
}

void ob::glob_precalc()
{
	if(!glob)
	{
		std::stringstream sstr;
		sstr << "globals not initialized";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	glob->precalc();
}

void ob::glob_quit()
{
	delete glob;
	glob = NULL;
	// If globals is NULL, set_game will not start a new thread.
	delete glob_get_game();
	game = NULL;
	ui::log.disconnect_all_slots();
}

Game* ob::glob_get_game()
{
	if(game_is_ready)
	{
		if(NULL != game_thread.get())
		{
			game_thread->join();
			game_thread = thr::ThreadSptr();
		}
		return game;
	}
	return NULL;
}

void ob::glob_queue_game()
{
	if(game_is_ready)
	{
		BOOST_THROW_EXCEPTION(std::runtime_error(
					"can't queue a game when a game is ready"));
	}
	// Do not start a game calculator thread if one already running.
	if(NULL == game_thread.get())
	{
		game_thread = thr::ThreadSptr(new boost::thread(glob_task_game));
		fade.setDelta(OB_FADE_DELTA);
	}
}

void ob::glob_set_game(Game *op)
{
	if(op)
	{
		if(game)
		{
			std::stringstream sstr;
			sstr << "trying to set game when previous exists";
			BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
		}
		game = op;
	}
	else
	{
		game = NULL;
		game_is_ready = false;
	}
}

