#ifndef SND_SAMPLE_HPP
#define SND_SAMPLE_HPP

#include "data/store.hpp"
#include "snd/generic.hpp"

#include <string>

namespace snd
{
	/** \brief One OpenAL sample.
	 */
	class Sample
	{
		private:
			/** Buffer for the wave. */
			ALuint m_buffer;

			/** Last play time. */
			uint64_t m_last_play_time;

			/** Sound level. */
			float m_gain;

		public:
			/** \brief Accessor.
			 *
			 * @return Buffer of this.
			 */
			inline ALuint getBuffer() const
			{
				return m_buffer;
			}

			/** \brief Accessor.
			 *
			 * @return Sound level.
			 */
			inline float getGain() const
			{
				return m_gain;
			}

			/** \brief Set the level.
			 *
			 * @param op New level.
			 */
			inline void setGain(float op)
			{
				m_gain = op;
			}

		public:
			/** \brief Empty constructor. */
			Sample();

			/** \brief Constructor.
			 *
			 * @param pfname Filename to load.
			 * @param gain Sample gain (default: 1).
			 */
			Sample(const char *pfname, float gain = 1.0f);

			/** \brief Constructor.
			 *
			 * @param pfname Filename to load.
			 * @param gain Sample gain (default: 1).
			 */
			Sample(const std::string &pfname, float gain = 1.0f);

			/** \brief Constructor.
			 *
			 * @param data Data count.
			 * @param size Size of data.
			 * @param freq Frequency.
			 * @param level General level for this sample.
			 */
			Sample(const uint8_t *data, unsigned size, unsigned freq, float level = 1.0f);

			/** \brief Destructor. */
			~Sample();

		private:
			/** \brief Generate the buffer in this sample.
			 */
			void generateBuffer();

		public:
			/** \brief Check if it's okay to play shis sample.
			 *
			 * Samples may be denied if they're spammed, that is, constantly
			 * requested again and again.
			 */
			bool canPlay();

			/** \brief Load a sample.
			 *
			 * Throws an exception on error.
			 *
			 * @param pfname Filename to load from.
			 */
			void load(const std::string &pfname);

		protected:
			/** \brief Release all resources reserved by this.
			 */
			void unreserve();
	};

	/** Convenience typedef. */
	typedef boost::shared_ptr<Sample> SampleSptr;

	/** \brief Storage for meshes. */
	class SampleStore :
		public data::Store<Sample>
	{
		public:
			/** \brief Constructor. */
			SampleStore();

			/** \brief Destructor. */
			virtual ~SampleStore() { }

		public:
			/** \cond */
			virtual void loadMeta(container_type &dst,
					const std::string &pfname,
					const data::StoreCreator<Sample> &creator);
			/** \endcond */
	};

	/** Global store. */
	extern SampleStore sample_store;
}

#endif
