#include "snd/stream.hpp"

#include "data/generic.hpp"
#include "ui/generic.hpp"

using namespace snd;

/** Vorbis block max size is constant. */
static const unsigned VORBIS_BLOCK_SIZE = 4096;

/** Number of queue block should be about a second? */
static const unsigned QUEUE_BLOCKS = 4;

Stream::Stream(const std::string &pfname) :
	m_stop(false)
{
	this->load(pfname);

	//std::cout << "starting thread\n";
	m_thread = thr::ThreadSptr(new boost::thread(
				boost::bind(&Stream::run, this)));
}

Stream::~Stream()
{
	m_stop = true;
	m_thread->join();
	ov_clear(&m_ogg_file);
}

void Stream::eos()
{
	int err = ov_raw_seek(&m_ogg_file, 0);
	if(err)
	{
		std::stringstream sstr;
		sstr << "could not seek to beginning of file (" << err << ")";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
}

void Stream::load(const std::string &pfname)
{
	std::string filename = data::open_search(pfname);

	int err = ov_fopen(const_cast<char*>(filename.c_str()),
			&m_ogg_file);
	if(0 != err)
	{
		std::stringstream sstr;
		sstr << "could not open ogg vorbis file (" << err << ")";
		BOOST_THROW_EXCEPTION(std::runtime_error(sstr.str()));
	}
	m_vorbis_info = ov_info(&m_ogg_file, -1);
}

void Stream::run()
{
	//std::cout << "thread started\n";
	while(!m_stop)
	{
		if(m_source.numQueuedBuffers() < QUEUE_BLOCKS)
		{
			bool eos_flag = false;
			int stream = 0;
			unsigned idx = 0;
			unsigned sizeleft = STREAM_BUFFER_SIZE;
			while(sizeleft > VORBIS_BLOCK_SIZE)
			{
				long bcnt = ov_read(&m_ogg_file, m_buffer + idx, 4096,
						0, m_vorbis_info->channels, 1, &stream);
				if(bcnt <= 0)
				{
					eos_flag = true;
					break;
				}
				idx += bcnt;
				sizeleft -= bcnt;
			}
			if(idx > 0)
			{
				m_samples.push_front(SampleSptr(
							new Sample(reinterpret_cast<const uint8_t*>(m_buffer), idx,
								m_vorbis_info->rate)));
				m_source.queue(*(m_samples.front()));
				alSourcef(m_source.getId(), AL_GAIN, volume_music);
			}
			// Take care of eos case after other stuff is done.
			if(eos_flag)
			{
				this->eos();
			}
		}
		else
		{
			// Sleep 10ms, it's pretty safe.
			thr::usec_sleep(10000); 
		}
		// Unqueue last buffer.
		ALuint released = m_source.unqueueOne();
		if(released > 0)
		{
			if(m_samples.back()->getBuffer() != released)
			{
				BOOST_THROW_EXCEPTION(std::runtime_error(
							"released buffer is not last in line"));
			}
			m_samples.pop_back();
		}
	}
}

